#!/usr/bin/python
# -*- coding: utf-8 -*-

import sys

sys.path.insert(0, '/opt')

import logging
import requests
import os
import re
import json
import subprocess
import yaml
import urllib3

from startrek_client import Startrek

logging.getLogger('startrek_client').setLevel(logging.WARNING)
urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)

pathlist = ["/usr/local/sbin", "/usr/local/bin", "/usr/sbin", "/usr/bin", "/sbin", "/bin", "/usr/games", "/usr/local/games"]
os.environ["PATH"] = os.pathsep.join(pathlist)

ERROR_STRING = u'произошла ошибка %s'
APPS_CONF_FILE = "/etc/yandex-direct/direct-apps.conf.yaml"
RETRIES = 5


def get_token(token_file):
    try:
        f = open(token_file)
        response = f.readline().strip(u'\n')
        f.close()
        return response

    except Exception:
        logging.error('something wrong with token for startrek')
        raise Exception('Ошибка при чтении токена')

st_token_file = os.environ[u'st_token_file'] if u'st_token_file' in os.environ else u'/etc/direct-tokens/startrek'
st_token = get_token(st_token_file)
client = Startrek(useragent=u'direct-yamb-bot', token=st_token)


trusted_users = set(['zhur', 'ppalex', 'hmepas', 'gerdler', 'lena-san', 'palasonic', 'pe4kin', 'yukaba', 'dspushkin', 'rivik'])


def get_login_by_uid(uid):
    global st_token

    try:
        url = u"https://api.staff.yandex-team.ru/v3/persons?uid=%s" % uid
        headers = {'Authorization': 'OAuth %s' % st_token}
    
        response = requests.get(url=url, headers=headers, verify=False)
        return response.json()['result'][0]['login']

    except Exception as e:
        logging.error(e)
        return ''


def get_prod_version():
    url = u"http://direct-dev.yandex-team.ru/versionica/property?project=prehistoric&reqid=&group=&host=ppcback01f&property=yandex-direct%24&host_group=&as=json"
    try:
        response = requests.get(url=url)
        return response.json()[u'yandex-direct']
    except Exception as e:
        logging.error(e)
        return ERROR_STRING % e


def get_prod_info(**kwargs):
    try:
        global client

        prod_ver = get_prod_version()

        issues = client.issues.find(
            u'queue: direct components: "Releases: Direct" (summary: ' + prod_ver + u' or comment: ' + prod_ver + u')',
            order=[u'-updated'],
            per_page=1
        )

        if len(issues) == 0:
            return u'Версия: %s, а тикет не могу найти' % prod_ver
        else:
            return u'Текущая версия: %s\nРелизный тикет: https://st.yandex-team.ru/%s' % (prod_ver, issues[0].key)

    except Exception as e:
        logging.error(e)
        return ERROR_STRING % e


def get_ticket_info(**kwargs):
    try:
        global client

        cmd_text = kwargs.pop(u'string')

        response = u'\n\n'

        for cur_ticket in re.findall(ur'(?:DIRECT|DAHW|DIRECTADMIN|DIRECTMIGR|DIRECTMOD|INFRASTRUCTUREPI|PI|TESTIRT)-[0-9]{1,7}\b', cmd_text, re.I):
            try:
                issue = client.issues[cur_ticket]

                if response:
                    response += "\n\n"

                response += u'Вижу тикет: %s' % cur_ticket
                response += u'\nНазвание: %s' % issue.summary
                if issue.assignee is None:
                    response += u'\nИсполнитель: Не назначен'
                else:
                    response += u'\nИсполнитель: %s  @%s' % (issue.assignee.display, issue.assignee.login)
                response += u'\nСтатус: %s' % issue.status.name
            except:
                response += u'Не могу найти тикет %s' % cur_ticket

        return response

    except Exception as e:
        logging.error(e)
        return ERROR_STRING % e


def check_ticket_for_deploy(**kwargs):
    try:
        global client

        cmd_text = kwargs.pop('string')
        author = kwargs.pop('author')
        author = get_login_by_uid(author)
        
        tickets = re.findall(ur'(?:DIRECT|DIRECTMIGR)-[0-9]{1,7}', cmd_text, re.I)

        if not tickets:
            if author in trusted_users:
                return u''
            else:
                return u'Без тикета не рекомендуется ничего выкладывать!'

        response = u'Информация по выкладке:'

        for ticket in tickets:
            issue = client.issues[ticket]

            if response:
                response += u"\n"

            response += u"%s: " % issue.key

            # если это релиз
            if issue.type.key == u'release':
                response += check_release_ticket_for_deploy(issue)
                continue

            # если это миграция
            if re.match(ur'^DIRECTMIGR', ticket):
                response += check_migration_for_deploy(issue)
                continue

            # если это про лимтест
            if 'Limtest' in [component.display for component in issue.components]:
                response += check_limtest_for_deploy(issue)
                continue

            # если это что-то непонятное
            response += u'Это не релиз, не лимтест и не миграция (возможно, не получается определить), так что действуйте на собственное усмотрение.'

        return response

    except Exception as e:
        logging.error(e)
        return ERROR_STRING % e


def check_limtest_for_deploy(issue):
    try:
        global client

        result = []

        zk_version_node = yaml.load(open(APPS_CONF_FILE))['apps']['direct']['zookeeper-version-node']
        prod_version = subprocess.check_output(["direct-zkcli", "-H", "ppcback01f.yandex.ru", "cat", zk_version_node])
        prod_version = re.search(ur'(1\.[0-9-\.]+)', prod_version[:prod_version.index("\n")]).group(1)

        limtest_version = re.search(ur'\s(1\.[0-9-\.]+)', issue.summary).group(1)

        if prod_version != limtest_version:
            result.append(u"базовая ревизия не совпадает с продакшеновой")

        if not is_limtest_approved(issue):
            result.append(u"нет комментария-подтверждения от ответственных")

        if not result:
            result.append(u"ок")

        return u", ".join(result)

    except Exception as e:
        logging.error(e)
        return ERROR_STRING % e


def is_limtest_approved(issue):
    try:
        global client

        comments = client.issues[issue.key].comments.get_all()

        for comment in comments:
            if comment.createdBy.id in ['raketa', 'sudar'] and re.search(ur'(?:можно)?.+?(?:выкладывать|выкатывать|катить|выкладывай|выкладывайте|выложите).+?на.+?(?:лимтест|limtest)', comment.text, re.I):
                return True
                
        return False

    except Exception as e:
        logging.error(e)
        return ERROR_STRING % e


def check_migration_for_deploy(issue):
    try:
        if issue.status.key != 'needInfo' and issue.status.key != 'open':
            return u'ок'
        else:
            return u'неправильный статус'

    except Exception as e:
        logging.error(e)
        return ERROR_STRING % e


def check_release_ticket_for_deploy(issue):
    try:
        result = []
        if not (issue.status.key == 'readyToDeploy' or issue.status.key == 'rmAcceptance'):
            result.append(u'неожиданный статус тикета (%s)' % issue.status.display)

        apps_dict = yaml.load(open(APPS_CONF_FILE))['apps']
        component2app = {apps_dict[app]['tracker-component'] : app for app in apps_dict}

        app = None
        for component in issue.components:
            if component.name in component2app:
                app = component2app[component.name]
                break

        version = re.search(r'[0-9-\.]+$', issue.summary).group(0)

        if get_state(apps_dict[app]['package'], version) != 'stable':
            result.append(u'пакет не в stable')

        output = subprocess.check_output(['direct-release', '--app', app, '-s', 'stable', 'show-state'])
        stable_ticket = json.loads(output[:output.find('show-state')])['ticket']

        if stable_ticket != issue.key:
            result.append(u'в direct-release данный релиз не в stable')

        if not result:
            result.append(u'ок')

        return u', '.join(result)
    except Exception as e:
        logging.error(e)
        return ERROR_STRING % e


def get_state(package, version):
    for i in xrange(RETRIES):
        try:
            output = subprocess.check_output(['ssh', '-o', 'StrictHostKeyChecking=no', 'dupload.dist', 'find_package', '-j', '%s_%s' % (package, version)])
            json_output = json.loads(output)

            if not json_output or 'result' not in json_output or len(json_output['result']) != 1:
                continue

            return json_output['result'][0]['environment']

        except:
            pass

    return ''


def compose_response(message):
    resp = u''

    for cmd in commands:
        found_cmd = re.search(cmd['command_re'], message['text'], re.I | re.U)
        if found_cmd:
            logging.info('cmd for message: %s' % cmd[u'code'].__name__)

            resp = cmd[u'code'](string=message['text'], author=message['author']['id'])
            break

    return resp


commands = [
    {
        'command_re': ur'^\?\s+prod_info\b',
        'code': get_prod_info
    },
    {
        'command_re': ur'^\?\s+(?:DIRECT|DAHW|DIRECTADMIN|DIRECTMIGR|DIRECTMOD|INFRASTRUCTUREPI|PI|TESTIRT)-[0-9]{1,7}\b',
        'code': get_ticket_info
    },
    {
        'command_re': ur'(?:^\?\s+выложить|выложите|выложить|выложишь|обновить|обновите|обновишь|выкатить|катнете|катните|катни|катнешь|выкатите|выкатишь|limtest[12]|лимтест|залочить)',
        'code': check_ticket_for_deploy
    }

]
