#!/usr/bin/python
# -*- coding: utf-8 -*-

import sys
sys.path.insert(0, '/opt/direct-chat-bot')
# debug only:
# sys.path.insert(0, 'opt/direct-chat-bot')

import argparse
import time
from bots import TelegramBot, YambBot, CliBot
from bot_commands import ChatTypes, MessageTypes


def parse_options():
    global opts

    parser = argparse.ArgumentParser(description="Директовый бот для различных чатиков")

    # обязательные параметры
    parser.add_argument(
        "-t", "--type",
        dest="type", help="тип чата", type=str, choices=['telegram', 'yamb', 'cli'], required=True
    )
    parser.add_argument(
        "-c", "--conf",
        dest="config", help="файл с конфигом", type=str, required=True
    )

    # группа для выбора действия
    action_group = parser.add_mutually_exclusive_group(required=True)
    action_group.add_argument(
        "--run-bot",
        dest="run_bot", help="запустить бота", action='store_true'
    )
    action_group.add_argument(
        "--one-time-reply",
        dest="one_time_reply", help="ответить на сообщение", action='store_true'
    )

    # группа с параметрами для отправки ответа на сообщение
    message_group = parser.add_argument_group(
        title="Параметры для отправки ответа на команду (сообщение)"
    )
    message_group.add_argument(
        "-m", "--message",
        dest="message", help="текст сообщения", type=str
    )
    message_group.add_argument(
        "--chat-id",
        dest="chat_id", help="id чата, куда отправить ответ", type=str, default=""
    )
    message_group.add_argument(
        "--from-user",
        dest="from_user", help='имя/логин от кого мы "пишем" боту сообщение', type=str, default=""
    )
    message_group.add_argument(
        "--chat-type",
        dest="chat_type", help='тип чата, откуда мы "пишем" боту', choices=ChatTypes.__members__.keys(),
        type=str, default=ChatTypes.private
    )
    message_group.add_argument(
        "--message-type",
        dest="message_type", help='тип сообщения, которое мы "пишем" боту',
        choices=MessageTypes.__members__.keys(), type=str, default=MessageTypes.text
    )
    message_group.add_argument(
        "--sleep",
        dest="sleep", type=int, default=0,
        help='поспать перед завершением (например, чтобы нормально работали локи на запуск в zookeeper)',
    )

    opts, extra = parser.parse_known_args()

    if len(extra) > 0:
        sys.exit("There are unknown parameters: %s" % ", ".join(extra))

    if opts.one_time_reply and (not opts.chat_id or not opts.message):
        parser.error("для отправки ответа на команду нужно указать --message и --chat-id")

    if not isinstance(opts.chat_type, ChatTypes):
        opts.chat_type = ChatTypes[opts.chat_type]

    return opts


def run():
    opts = parse_options()

    bot = None
    if opts.type == u"telegram":
        bot = TelegramBot(opts.config)

    elif opts.type == u"yamb":
        bot = YambBot(opts.config)

    elif opts.type == u"cli":
        bot = CliBot(opts.config)

    # если хотим разово ответить на сообщение
    if opts.one_time_reply:
        bot.reply_to_message(
            opts.message,
            opts.from_user,
            opts.chat_id,
            opts.chat_type,
            opts.message_type,
            'text',
            one_time_reply=opts.one_time_reply
        )
    # иначе запускаем бота
    else:
        bot.run()

    if opts.sleep:
        time.sleep(opts.sleep)


if __name__ == '__main__':
    run()

