#!/usr/bin/python
# -*- coding: utf-8 -*-

import sys

sys.path.insert(0, '/opt')

import logging
import requests
import os
import re
import json
import subprocess
import time
import yaml
import urllib3
import urllib2
import datetime
from io import BytesIO
from enum import Enum
from os import getpid

from startrek_client import Startrek

SESSION_ID = getpid()

logging.getLogger('startrek_client').setLevel(logging.WARNING)
urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)

pathlist = ["/usr/local/sbin", "/usr/local/bin", "/usr/sbin", "/usr/bin", "/sbin", "/bin"]
os.environ["PATH"] = os.pathsep.join(pathlist)

APPS_CONF_FILE = "/etc/yandex-direct/direct-apps.conf.yaml"
UNKNOWN_APP = u"unknown-app"

OBSERVATORIUM_CE_URL = 'https://observatorium.common.yandex.ru/juggler/ce/'
OBSERVATORIUM_TAGS_URL = 'https://observatorium.common.yandex.ru/juggler/get_tags/'

MessageContentTypes = Enum('MessageContentTypes', 'text image document')
ChatTypes = Enum('ChatTypes', 'private group channel')
MessageTypes = Enum('MessageTypes', 'text forward')

DUTY_ALIASES = {
    # маппинг алиасов в duty.mtrs группы: https://duty.mtrs.yandex-team.ru/project/direct/duty_group_dashboard/<group>
    'app-duty': ['direct-prod-1', 'direct-prod-2'],
    'admins': ['admins'],
}

QUEUES = {
    'devsup': {
        'startrek_query': 'Queue: DIRECT Components: devsup AND Status: !Closed "Sort By": Created desc',
    },
}


DASHBOARD_TYPE_BY_UID = 'grafana_by_uid'
DASHBOARD_TYPE_BY_NAME = 'grafana_by_name'

COMMON_GRAFANA_URL = "https://grafana.yandex-team.ru"
DIRECT_GRAFANA_URL = "http://ppcgraphite.yandex.ru/grafana"

DASHBOARDS = {
    'direct-api-monitoring': {
        'type': DASHBOARD_TYPE_BY_NAME,
        'url': DIRECT_GRAFANA_URL
    },
    'direct-group-internal-systems-tv': {
        'type': DASHBOARD_TYPE_BY_NAME,
        'url': DIRECT_GRAFANA_URL
    },
    'direct-group-internal-systems-tv-2': {
        'type': DASHBOARD_TYPE_BY_NAME,
        'url': DIRECT_GRAFANA_URL
    },
    'direct-resync-control': {
        'type': DASHBOARD_TYPE_BY_NAME,
        'url': DIRECT_GRAFANA_URL
    },
    'direct-group-sre-tv': {
        'type': DASHBOARD_TYPE_BY_NAME,
        'url': DIRECT_GRAFANA_URL
    },
    'direct-profile': {
        'type': DASHBOARD_TYPE_BY_NAME,
        'url': DIRECT_GRAFANA_URL
    },
    'testing-direct-time-lag-yt-sync-monitoring': {
        'type': DASHBOARD_TYPE_BY_NAME,
        'url': DIRECT_GRAFANA_URL
    },
    'direct-java-binlog-to-yt-sync': {
        'type': DASHBOARD_TYPE_BY_NAME,
        'url': DIRECT_GRAFANA_URL
    },
    'direct-brandlift': {
        'type': DASHBOARD_TYPE_BY_UID,
        'url': COMMON_GRAFANA_URL,
        'uid': 'BbJhyWWMz'
    },
    'video-constructor': {
        'type': DASHBOARD_TYPE_BY_UID,
        'url': COMMON_GRAFANA_URL,
        'uid': 'Jqks3SEZk',
        'width': 700,
        'height': 1760
    },
    'bannerstorage-creatives': {
        'type': DASHBOARD_TYPE_BY_UID,
        'url': COMMON_GRAFANA_URL,
        'uid': 'tKuwvVeZk',
        'height': 500
    },
    'autooverdraft-porog-otkliucheniia': {
        'type': DASHBOARD_TYPE_BY_UID,
        'url': COMMON_GRAFANA_URL,
        'uid': 'W1fDtB_mz',
        'width': 1500,
        'height': 930
    },
    'direct-nonskippable': {
        'type': DASHBOARD_TYPE_BY_UID,
        'url': COMMON_GRAFANA_URL,
        'uid': 'gtvBM6MMz',
        'width': 1500,
        'height': 750
    },
    'bannerstorage-queues': {
        'type': DASHBOARD_TYPE_BY_NAME,
        'url': COMMON_GRAFANA_URL,
        'uid': '000009638',
        'width': 1500,
        'height': 1900
    },
    'bannerstorage-rapid-health': {
        'type': DASHBOARD_TYPE_BY_NAME,
        'url': COMMON_GRAFANA_URL,
        'uid': '000006034',
        'width': 1500,
        'height': 1130
    },
    'canvas-direct': {
        'type': DASHBOARD_TYPE_BY_UID,
        'url': COMMON_GRAFANA_URL,
        'uid': 'AQ9gPJPWz',
        'width': 1500,
        'height': 1020
    },
    'bannerstorage': {
        'type': DASHBOARD_TYPE_BY_UID,
        'url': COMMON_GRAFANA_URL,
        'uid': '000018056',
        'height': 1440
    },
    'direct-cpv': {
        'type': DASHBOARD_TYPE_BY_UID,
        'url': COMMON_GRAFANA_URL,
        'uid': '-_KFRScMz',
        'height': 560
    },
    'direct-playable': {
        'type': DASHBOARD_TYPE_BY_UID,
        'url': COMMON_GRAFANA_URL,
        'uid': '_vdxTBmMk',
        'height': 600
    },
    'direct-recommendations': {
        'type': DASHBOARD_TYPE_BY_UID,
        'url': COMMON_GRAFANA_URL,
        'uid': 'HiRHcy2iz',
        'width': 1600,
        'height': 1180
    }
}

DASHBOARD_NAMES = sorted(DASHBOARDS.keys())

NEW_CI_INSTRUCTION = 'https://docs.yandex-team.ru/direct-dev/guide/releases/release-newci#vykladka-reliza'
OTHER_RELEASES_INSTRUCTION = 'https://wiki.yandex-team.ru/jeri/app-duty/releases/'

def get_token(token_file):
    try:
        f = open(token_file)
        response = f.readline().strip(u'\n')
        f.close()
        return response

    except Exception:
        logging.error('something wrong with token for startrek')
        raise Exception('Ошибка при чтении токена')

st_token_file = os.environ[u'st_token_file'] if u'st_token_file' in os.environ else u'/etc/direct-tokens/startrek'
st_token = get_token(st_token_file)
st_client = Startrek(useragent=u'direct-chat-bot', token=st_token)

GRAFANA_API_OAUTH_TOKEN = get_token('/etc/direct-tokens/oauth_robot-direct-notify')

trusted_users = set([
    'ppalex',
    'gerdler',
    'pe4kin',
    'yukaba',
])


def get_login_by_uid(uid):
    global st_token

    try:
        url = u"https://api.staff.yandex-team.ru/v3/persons?uid=%s" % uid
        headers = {'Authorization': 'OAuth %s' % st_token}

        response = requests.get(url=url, headers=headers, verify=False)
        return response.json()['result'][0]['login']

    except Exception as e:
        logging.error(e)
        return ''


def get_login_by_telegram_username(tg_username):
    global st_token

    try:
        url = u"https://staff-api.yandex-team.ru/v3/persons?_fields=login&accounts.type=telegram&accounts.value=%s" % (
            tg_username
        )
        headers = {'Authorization': 'OAuth %s' % st_token}
        return requests.get(url=url, headers=headers, verify=False).json()['result'][0]['login']
    except:
        logging.exception('Exception occured')
        return 'Unknown user'


def get_tg_logins_by_staff(staff_login):
    global st_token

    logins = []
    try:
        url = u"https://api.staff.yandex-team.ru/v3/persons?login=%s&_fields=accounts" % staff_login
        headers = {'Authorization': 'OAuth %s' % st_token}

        response = requests.get(url=url, headers=headers, verify=False)
        response.raise_for_status()
        for x in response.json()['result'][0]['accounts']:
            if x.get('type', '') == "telegram" and x.get('value'):
                logins.append(x['value'])
    except Exception as e:
        logging.error(e)

    return logins


def get_duty_staff_login(duty_group):
    try:
        url = u"https://duty.mtrs.yandex-team.ru/api/v1/project/direct/duty_group/%s" % duty_group
        response = requests.get(url=url, verify=False)
        response.raise_for_status()
        return response.json()['data']['duty']

    except Exception as e:
        logging.error(e)
        return ''


def get_prod_version():
    url = (u"http://direct-dev.yandex-team.ru/versionica/property" +
        u"?project=prehistoric&reqid=&group=&host=ppcback01f&property=yandex-direct%24&host_group=&as=json")
    response = requests.get(url=url)
    return response.json()[u'yandex-direct']


def get_prod_info(**kwargs):
    global st_client

    prod_ver = get_prod_version()

    issues = st_client.issues.find(
        u'queue: direct components: "Releases: Direct" (summary: ' + prod_ver + u' or comment: ' + prod_ver + u')',
        order=[u'-updated'],
        per_page=1
    )

    if len(issues) == 0:
        return u'Версия: %s, а тикет не могу найти' % prod_ver
    else:
        return u'Текущая версия: %s\nРелизный тикет: https://st.yandex-team.ru/%s' % (prod_ver, list(issues)[0].key)


def get_ticket_info(**kwargs):
    global st_client

    cmd_text = kwargs.get(u'string')
    cmd = kwargs.get(u'cmd')
    chat_type = kwargs.get(u'chat_type')
    message_type = kwargs.get(u'message_type')

    if (not (chat_type == ChatTypes.private and message_type == MessageTypes.forward)
           and not cmd_text.startswith('/ticket ')):
        return u''

    response = u''

    for cur_ticket in re.findall(cmd['regexp'], cmd_text, re.I):
        try:
            issue = st_client.issues[cur_ticket]

            if response:
                response += "\n\n"

            response += u'Вижу тикет: %s' % cur_ticket
            response += u'\nНазвание: %s' % issue.summary
            if issue.assignee is None:
                response += u'\nИсполнитель: Не назначен'
            else:
                response += u'\nИсполнитель: %s  @%s' % (issue.assignee.display, issue.assignee.login)
            response += u'\nСтатус: %s' % issue.status.name
        except:
            response += u'Не могу найти тикет %s' % cur_ticket

    return response

def memoize(function):
    MAX_AGE = 30 #кешируем значение функции не более чем на 30 секунд
    SESSION_KEY="__session_id"
    cache = {}
    expiration = {SESSION_KEY: SESSION_ID}
    def wrap(*args):
        if (expiration[SESSION_KEY] != SESSION_ID or
                args not in cache or
                expiration[args] < time.time()):
            cache[args] = function(*args)
            expiration[args] = time.time() + MAX_AGE
            expiration[SESSION_KEY] = SESSION_ID
        return cache[args]

    return wrap

@memoize
def get_oldest_unclosed_releases_by_app():
    ST_QUERY = 'Queue: DIRECT Type: Release Status: !"Closed" "Sort by": Created desc'
    releases = st_client.issues.find(ST_QUERY)

    apps_conf = get_apps_conf()
    component2app = get_component2app(apps_conf)

    old_releases_by_app = {}
    for release in releases:
        app = get_app_name(release.components, component2app)
        if (app not in old_releases_by_app) or (old_releases_by_app[app].createdAt > release.createdAt):
            old_releases_by_app[app] = release

    return old_releases_by_app

def releases_for_deploy(**kwargs):
    ST_QUERY = 'Queue: DIRECT Type: Release (Status: "Ready to Deploy" OR  Status: "RM Acceptance") "Sort by": key asc'
    releases = st_client.issues.find(ST_QUERY)
    apps_conf = get_apps_conf()
    component2app = get_component2app(apps_conf)

    msg = u''
    newci_msg =u''
    other_releases_msg = u''

    for release in releases:
        app = get_app_name(release.components, component2app)
        # некоторые релизы выкладываются супер-особенно, про них не напоминаем
        # например, b2yt, mysql2yt-full
        # лучше, чтобы такие исключения не плодились, а наоборот, сокращались
        if app in apps_conf and "releases-for-deploy-notification" in apps_conf[app].get("ignore-features", []):
            continue
        ###

        migr_tickets = []
        release_info = u''
        if release.description:
            migr_tickets = re.findall(r'DIRECTMIGR-[0-9]+', release.description)
        migr_tickets = filter(lambda x: st_client.issues[x].status.key != 'closed', migr_tickets)
        count_migrations = len(migr_tickets)

        release_info += u"(%s) https://st.yandex-team.ru/%s%s" % (
            app,
            release.key,
            u" (миграции: %d)" % count_migrations if count_migrations > 0 else ""
        )
        if is_newci_app(release, component2app) :
            newci_msg += release_info + u" - %s\n" % check_release_ticket_for_deploy(release)
        else:
            other_releases_msg += release_info + u" - %s\n" % check_release_ticket_for_deploy(release)

    if newci_msg:
        msg += u'\n<b>NewCI приложения, <a href=\'%s\'>инструкция</a>:</b>\n%s' % (NEW_CI_INSTRUCTION, newci_msg)
    if other_releases_msg:
        msg += u'\n<b>Другие приложения, <a href=\'%s\'>инструкция</a>:</b>\n%s' % (OTHER_RELEASES_INSTRUCTION, other_releases_msg)

    if not msg:
        msg = u"Нет релизов для выкладки"
    else:
        msg = u"Ждут выкладки релизы:\n" + msg

    bot_type = kwargs.get(u'bot_type')
    if bot_type == "telegram":
        return {'text': msg, 'parse_mode': 'html'}
    else:
        return msg


def check_ticket_for_deploy(**kwargs):
    global st_client

    cmd_text = kwargs.get('string')
    author = kwargs.get('author')
    bot_type = kwargs.get('bot_type')

    if bot_type == 'yamb':
        author = get_login_by_uid(author)

    tickets = re.findall(ur'(?:DIRECT|DIRECTMIGR)-[0-9]{1,7}', cmd_text, re.I)

    if not tickets:
        if author in trusted_users:
            return u''
        else:
            return u'Без тикета не рекомендуется ничего выкладывать!'

    tickets = list(set(tickets))

    response = u'Информация по выкладке:'

    for ticket in tickets:
        try:
            issue = st_client.issues[ticket]
        except:
            continue

        if response:
            response += u"\n"

        response += u"%s: " % issue.key
        # если это релиз
        if issue.type.key == u'release':
            response += check_release_ticket_for_deploy(issue)
            continue

        # если это миграция
        if re.match(ur'^DIRECTMIGR', ticket):
            response += check_migration_for_deploy(issue)
            continue

        # если это про лимтест
        if 'Limtest' in [component.display for component in issue.components]:
            response += check_limtest_for_deploy(issue)
            continue

        # если это что-то непонятное
        response += (u'Это не релиз, не лимтест и не миграция (возможно, не получается определить),' +
            ' так что действуйте на собственное усмотрение.')

    return response


def check_limtest_for_deploy(issue):
    global st_client

    result = []

    zk_version_node = get_apps_conf()['direct']['zookeeper-version-node']
    prod_version = subprocess.check_output(["direct-zkcli", "-H", "ppcback01f.yandex.ru", "cat", zk_version_node])
    prod_version = re.search(ur'(1\.[0-9]+)', prod_version[:prod_version.index("\n")]).group(1)

    limtest_version = re.search(ur'\s(1\.[0-9]+)', issue.summary).group(1)

    if prod_version != limtest_version:
        result.append(u"базовая ревизия не совпадает с продакшеновой")

    if not is_limtest_approved(issue):
        result.append(u"нет комментария-подтверждения от ответственных")

    if not result:
        result.append(u"ок")

    return u", ".join(result)


def is_limtest_approved(issue):
    global st_client

    comments = st_client.issues[issue.key].comments.get_all()

    for comment in comments:
        if comment.createdBy.id in ['raketa', 'sudar', 'ppalex', 'gerdler', 'hmepas', 'maxlog', 'pavryabov', 'buhter'] and \
           re.search(ur'(?:можно\s*)?(?:выкладывать|выкатывать|катить|выкладывай|выкладывайте|выложите)', comment.text, re.I):
            return True

    return False


def check_migration_for_deploy(issue):
    if issue.status.key != 'needInfo' and issue.status.key != 'open':
        return u'ок'
    else:
        return u'неправильный статус'


def get_apps_conf(filename=APPS_CONF_FILE):
    with open(filename, 'r') as fd:
        return yaml.load(fd)['apps']


def get_component2app(apps_conf):
    return {apps_conf[app]['tracker-component']: app for app in apps_conf if 'tracker-component' in apps_conf[app]}


def get_app_name(release_components, component2app):
    app = UNKNOWN_APP
    for component in release_components:
        if component.name in component2app:
            app = component2app[component.name]
            break

    return app


def get_ticket_from_direct_release(app, stage):
    try:
        output = subprocess.check_output(['direct-release', '--app', app, '-s', stage, 'show-state'])
        return json.loads(output[output.find('{'):output.rfind('}') + 1])['ticket']
    except:
        return ""

def get_sandbox_task(app, ticket):
    logging.info(app)
    logging.info(ticket)
    try:
        output = subprocess.check_output(['/usr/local/bin/get-release-sandbox-task', ticket])
        logging.info(output)
        return output.rstrip()
    except:
        return u"не удалось определить"


def is_newci_app(issue, component2app):
    apps_conf = get_apps_conf()
    app = get_app_name(issue.components, component2app)
    return apps_conf[app].get('newci_graph', '')

def check_release_ticket_for_deploy(issue):
    result = []

    apps_conf = get_apps_conf()
    component2app = get_component2app(apps_conf)
    app = get_app_name(issue.components, component2app)

    if app == UNKNOWN_APP:
        return u"Неизвестное приложение, так что действуйте на собственное усмотрение"

    UNKNOWN_TICKET_STATUS_STR = u'неожиданный статус тикета (%s)'

    oldest_unclosed_releases_by_app = get_oldest_unclosed_releases_by_app()
    if ( app in oldest_unclosed_releases_by_app and
                issue.createdAt > oldest_unclosed_releases_by_app[app].createdAt and
                issue.key != oldest_unclosed_releases_by_app[app].key ) :
            result.append(u'предыдущий релиз (%s) не выложен' % oldest_unclosed_releases_by_app[app].key)
    # приложения, для которых акцепт делает аппдьюти
    if issue.status.key == 'readyToDeploy':
        pass
    elif issue.status.key == 'rmAcceptance':
        result.append(u'не забудьте нажать Accept до выкладки')
    else:
        result.append(UNKNOWN_TICKET_STATUS_STR % issue.status.display)
    version = re.search(r'[0-9]+\..+$', issue.summary).group(0)

    if apps_conf[app].get('deploy_type', '') != 'yadeploy':
        if get_state(apps_conf[app]['package'], version) != 'stable':
            result.append(u'пакет не в stable')

    if "direct-release" not in apps_conf[app].get("ignore-features", []) and not apps_conf[app].get('newci_graph', ''):
        allowed_stages = ['stable', 'waiting-1']
        if not any(get_ticket_from_direct_release(app, stage) == issue.key for stage in allowed_stages):
            result.append(u'в direct-release данный релиз не в %s' % u", ".join(allowed_stages))

    try:
        subprocess.check_output(['dt-deps-manager', '-cr', issue.key])
    except:
        result.append(u"нельзя выкладывать из-за зависимостей")

    try:
        subprocess.check_output(['dt-check-release-consistency', issue.key])
    except:
        result.append(
            u"проверка релиза упала (запустите dt-check-release-consistency %s на ppcdev'е)" % issue.key
        )

    if not result:
        result.append(u'ок')

    return u', '.join(result)


def get_state(package, version):
    RETRIES = 5

    for i in xrange(RETRIES):
        try:
            output = subprocess.check_output([
                'ssh', '-o', 'StrictHostKeyChecking=no', 'dupload.dist', 'find_package', '-j', '%s_%s' % (
                    package, version
                )
            ]).replace("\n", "")
            json_output = json.loads(output)

            if not json_output or 'result' not in json_output or len(json_output['result']) != 1:
                continue

            return json_output['result'][0]['environment']

        except:
            pass

    return ''


def issue_text(issue):
    return "%s %s@ (%s)\n%s\nhttps://st.yandex-team.ru/%s" % (
        issue.key, issue.createdBy.id, issue.status.key, issue.summary, issue.key
    )

def compose_message(queue, new, in_progress, report_empty=False):
    message_parts = [ u'## Статус по очереди %s' % queue ]
    if   not len(new) > 0 and not len(in_progress) > 0 and not report_empty:
        return ''
    elif not len(new) > 0 and not len(in_progress) > 0 and     report_empty:
        message_parts += [ u'-- ничего' ]
    else:
        new_text         = "\n".join([issue_text(i) for i in new]) or u'-- ничего'
        in_progress_text = "\n".join([issue_text(i) for i in in_progress]) or u'-- ничего'

        message_parts += [
                u'### Новые запросы',
                new_text,
                u'\n### В процессе',
                in_progress_text,
                ]

    message_parts += [ '-- End --' ]
    return "\n".join(message_parts)


def report_queue(queue, report_empty=False):
    query = QUEUES[queue]['startrek_query']
    issues_generator = st_client.issues.find(query)

    new = []
    in_progress = []
    for i in issues_generator:
        if i.status.key == 'needInfo':
            pass
        elif i.status.key == 'open':
            new += [i]
        else:
            in_progress += [i]

    return compose_message(queue, new=new, in_progress=in_progress, report_empty=report_empty)


def devsup_status(**kwargs):
    return report_queue(queue='devsup', report_empty=True)


def count_time_for_mon(change_time_str):
    change_time_dt = datetime.datetime.fromtimestamp(change_time_str)
    delta = (datetime.datetime.now() - change_time_dt).total_seconds()
    if delta > 60 * 60 * 24:
        return "%i%c" % (delta // (60 * 60 * 24), 'd')
    elif delta > 60 * 60:
        return "%i%c" % (delta // (60 * 60), 'h')
    else:
        return "%i%c" % (delta // 60, 'm')


def mon(**kwargs):
    MAX_EVENTS = 30
    JUGGlER_APIV2_URL = 'http://juggler-api.search.yandex.net/v2'
    CHECKS_METHOD = 'checks/get_checks_state'
    HEADERS = {"Content-Type": "application/json", "Accept": "application/json"}
    DELIMITER = "::"

    cmd_text = kwargs.get(u'string')
    cmd = kwargs.get(u'cmd')
    bot_type = kwargs.get(u'bot_type')

    search_params = re.match(cmd['regexp'], cmd_text, re.I)
    params = search_params.groupdict('')
    if not params['tags']:
        return mon_ls(**kwargs)

    msg = u""
    keyboard = []

    quiet = params['quiet'] == 'quiet' or params['statuses'] == 'quiet'
    statuses = params['statuses'].split(',')
    statuses = [x.upper() for x in statuses if x.upper() in ['CRIT', 'WARN', 'OK']]
    if not statuses:
        statuses = ['CRIT', 'WARN']
    tags = params['tags'].split(',')

    resp = requests.post(
        "%s/%s" % (JUGGlER_APIV2_URL, CHECKS_METHOD),
        data=json.dumps({
            "filters": [
                dict(zip(('host', 'service'), tag[6:].split(DELIMITER))) if tag.startswith("event:") else {"tags": [tag]}
                for tag in tags
            ],
            "statuses": statuses
        }),
        headers=HEADERS
    )
    data = json.loads(resp.content)
    data['items'] = sorted(data['items'], key=lambda x: (x['status'], x['host'], x['service']))

    rows = []

    if data['items'] and cmd['name'] == '/mon' and bot_type == 'telegram':
        keyboard.append([u"/mon_v %s" % params['tags']])

    for event in data['items']:
        new_row = '%s %s%s%s (%s)' % (
            event['status'],
            event['host'],
            DELIMITER,
            event['service'],
            count_time_for_mon(event['change_time']),
        )

        if bot_type == 'telegram':
            if cmd['name'] == '/mon_v':
                new_row = '<b>%s</b>: <pre>%s</pre>' % (new_row, event['description'])
            elif cmd['name'] in ['/mon', '/title']:
                new_row += ' <a href="https://juggler.yandex-team.ru/check_details/?host=%s&service=%s">>></a>' % (
                    event['host'],
                    event['service']
                )
                keyboard.append([u"/mon_v event:%s%s%s" % (event['host'], DELIMITER, event['service'])])
        else:
            if cmd['name'] == '/mon_v':
                new_row += ': %s' % event['description']

        rows.append(new_row)

    msg = u"Juggler/%s\n" % params['tags']
    msg += u" || ".join([u"%s: %d" % (status['status'], status['count']) for status in data['statuses']])
    if rows:
        msg += u"\n\n" + u"\n".join(rows)

    # если все ОК и указан параметр quiet, то сообщение не отсылаем
    if not rows and quiet:
        msg = ""

    if bot_type == 'telegram':
        msg_tg = {
            'text': msg,
            'parse_mode': 'html',
        }
        if keyboard:
            msg_tg['keyboard'] = keyboard

        if cmd['name'] == '/mon_title':
            chat_alias = kwargs['chat_alias']
            # Будем вычислять, какой заголовок установить для чата.
            # Если проверки зеленые -- делаем зеленый заголовок
            # Если проверки желтые или красные -- делаем заголовок желтым или красным
            # Если заголовок не зеленый слишком долго -- убираем цвет из заголовка
            props_dict = {
                    'Direct.AppDuty.Monitoring': {
                        'emoji_ok': "\xF0\x9F\x92\x9A",
                        'emoji_warn': "\xE2\x9A\xA0\xEF\xB8\x8F",
                        'emoji_crit': "\xE2\x9B\x94\xEF\xB8\x8F",
                        'emoji_unknown': "\xF0\x9F\x95\xB8",
                        'title_template': "%s Direct.AppDuty.Monitoring",
                        'unknown_treshold': 3000,
                        },
                    'Direct.NP.Monitoring': {
                        'emoji_ok': "\xE2\x98\x80\xEF\xB8\x8F",
                        'emoji_warn': "\xE2\x98\x81\xEF\xB8\x8F",
                        'emoji_crit': "\xE2\x98\x94",
                        'emoji_unknown': "\xF0\x9F\x95\xB8",
                        'title_template': "%s Direct.NP.Monitoring",
                        'unknown_treshold': 44000,
                        },
                    }

            # если приходит незнакомый chat_alias -- падаем, это пока приемлемо
            # как обобщать надо вообще придумывать
            props = props_dict[chat_alias]
            date_format = '%Y-%m-%d-%H-%M-%S'
            emoji = props['emoji_ok']
            title_ok = props['title_template'] % ( props['emoji_ok'] * 3 )

            for status in data['statuses']:
                if (status['status'] == 'CRIT' and status['count'] > 0) or (status['status'] == 'WARN' and status['count'] > 3):
                    emoji = props['emoji_crit']
                    break
                elif status['status'] == 'WARN' and status['count'] > 0:
                    emoji = props['emoji_warn']

            #emoji = props['emoji_crit'] # for testing & debug
            msg_tg['title'] = props['title_template'] % (emoji * 3)

            if msg_tg['title'] != title_ok and kwargs['chat_info']['title'].encode('utf8') != title_ok:
                # Если новый статус не зеленый и старый не зеленый -- определяем, сколько времени он уже не был зеленым
                # Если в описании чата написано время -- считаем от него
                # Если в описании чата не написано -- считаем, что очень давно
                delta = 10000000
                m = re.match(r'NOT_OK_START: ([0-9\-]+)', kwargs['chat_info'].get('description', ''))
                if m:
                    try:
                        not_ok_start = datetime.datetime.strptime(m.group(1), date_format)
                        now = datetime.datetime.now()
                        delta = (now - not_ok_start).total_seconds()
                    except Exception as e:
                        pass
                # Если чат давно не был зеленым -- считаем, что цвет в заголовке уже ничего не сообщает, и сбрасываем иконку на "непонятно"
                if delta > props['unknown_treshold']:
                    msg_tg['title'] = props['title_template'] % ( props['emoji_unknown'] * 3 )

            if msg_tg['title'] == kwargs['chat_info']['title'].encode('utf8'):
                # сюда попадаем, когда заголовок не меняется
                # ничего не отправляем
                return ""
            elif kwargs['chat_info']['title'].encode('utf8') == title_ok:
                # сюда попадаем, когда заголовок меняется с зеленого на другой
                # записываем в description текущее время
                now = datetime.datetime.now()
                msg_tg['description'] = "NOT_OK_START: %s" % now.strftime(date_format)
            elif msg_tg['title'] == title_ok:
                # сюда попадаем, когда заголовок меняется с красного/желтого на зеленый
                # сбрасываем дату в описании
                if kwargs['chat_info'].get('description', '') != '':
                    msg_tg['description'] = ""

        return msg_tg

    return msg


def appduty(**kwargs):
    cmd_text = kwargs.get(u'string')
    cmd = kwargs.get(u'cmd')
    bot_type = kwargs.get(u'bot_type')

    search_params = re.match(cmd['regexp'], cmd_text, re.I)
    params = search_params.groupdict('')

    quiet = params['quiet'] == 'quiet'
    tickets = sorted(list(
        st_client.issues.find("""
            (Queue: DIRECT Components: prod_crit_today, prod_crit_1h Status: !Closed)
            or (Queue: DIRECTINCIDENTS Status: Open)
            or (Queue: SPI Components: direct Status: New Tags: !"SERVICE:NP")
            """
        )),
        key=lambda x: x.key,
        reverse=True
    )

    tickets_high_sev = []
    tickets_medium_sev = []
    for t in tickets:
        # про DIRECT решить, что считать high, что medium
        if t.queue.key == 'DIRECTINCIDENTS':
            components = set( [ c.name for c in t.components ] )
            if '~sev1' in components or '~sev2' in components:
                tickets_high_sev.append( t )
            elif components.intersection({'~sev1', '~sev2', '~sev3'}) == set([]):
                # если северити не размечена -- считаем, что high
                tickets_high_sev.append( t )
            else:
                tickets_medium_sev.append( t )
        else:
            tickets_medium_sev.append( t )

    title = ""
    if len(tickets_high_sev) > 0:
        title = "\xE2\x80\xBC\xEF\xB8\x8F%s " % len(tickets_high_sev)
    if len(tickets_medium_sev) > 0:
        title += "\xE2\x86\x97%s " % len(tickets_medium_sev)
    title += "direct-app-duty"

    if quiet:
        return {'text': '', 'title': title}

    if bot_type == 'telegram':
        msg = "\n".join('<a href="https://st.yandex-team.ru/%s">%s</a> (%s)' % (
            ticket.key, ticket.key, ticket.summary
        ) for ticket in tickets)
        return {'text': msg, 'parse_mode': 'html', 'title': title}

    msg = "\n".join("%s (%s)" % (ticket.key, ticket.summary) for ticket in tickets)
    return {'text': msg, 'title': title}


def mon_ls(**kwargs):
    bot_type = kwargs.get(u'bot_type')

    req = urllib2.Request(OBSERVATORIUM_TAGS_URL)
    req.add_header('Content-Type', 'application/json')

    response = json.loads(urllib2.urlopen(req).read())

    text = u"Juggler-тэги: \n\n%s" % ("\n".join(response))

    if bot_type == 'telegram':
        return {'text': text, 'keyboard': [[u"/mon %s" % tag] for tag in response]}

    return text


def d_list(**kwargs):
    bot_type = kwargs.get(u'bot_type')

    text = u"Список дашбордов:\n%s" % u"\n".join(DASHBOARD_NAMES)

    if bot_type == 'telegram':
        return {'text': text, 'keyboard': [[u"/d_panels %s" % dash] for dash in DASHBOARD_NAMES]}
    else:
        return text, MessageContentTypes.text


def duty_call(**kwargs):
    cmd_text = kwargs.get(u"string")
    cmd = kwargs.get(u"cmd")
    bot_type = kwargs.get(u"bot_type")

    author = kwargs.get('author')
    if bot_type == 'yamb':
        author = get_login_by_uid(author)

    duty_alias = "app-duty"

    tg_logins = set()
    # если алиас есть в конфиге - берем группы оттуда, если нет - app-duty
    for group in DUTY_ALIASES[duty_alias]:
        staff_login = get_duty_staff_login(group)
        tg_logins.update(get_tg_logins_by_staff(staff_login))
    if not tg_logins:
        tg_logins.add("NO_DUTY_FOUND")

    duty_msg = u"%s - посмотрите, пожалуйста" % ", ".join("@" + x for x in sorted(list(tg_logins)))
    text = u"@%s говорит, что в проде, вероятно, массовая критичная проблема.\n\n%s" % (author, duty_msg)

    if bot_type == "telegram":
        return {"text": text}
    else:
        return text, MessageContentTypes.text


def duty_respond(**kwargs):
    bot_type = kwargs.get(u"bot_type")

    author = kwargs.get('author')
    if bot_type == 'yamb':
        author = get_login_by_uid(author)

    notes = [
        u"  * Оцени масштаб аварии: https://nda.ya.ru/3VrXnL",
        u"  * Затрудняешься - сразу проси помощи (https://nda.ya.ru/3VrXza)",
        u"  * Общая инструкция: https://nda.ya.ru/3VrXmq",
    ]
    text = u"@%s говорит, что занимается проблемой.\n\n@%s:\n%s" % (author, author, "\n".join(notes))

    if bot_type == "telegram":
        return {"text": text}
    else:
        return text, MessageContentTypes.text


def duty_cancel(**kwargs):
    bot_type = kwargs.get(u"bot_type")

    author = kwargs.get('author')
    if bot_type == 'yamb':
        author = get_login_by_uid(author)

    text = u"@%s говорит, что критичных проблем в проде нет, действий от дежурных не требуется" % author

    if bot_type == "telegram":
        return {"text": text}
    else:
        return text, MessageContentTypes.text


def get_headers_by_grafana_url(url):
    headers = {}
    if url == COMMON_GRAFANA_URL:
        headers = {"Authorization": "OAuth %s" % GRAFANA_API_OAUTH_TOKEN, 'Cookie': 'sessionid2=xxx'}

    return headers


def get_dashboard_data(dashboard_name):
    if DASHBOARDS[dashboard_name]['type'] == DASHBOARD_TYPE_BY_UID:
        url = "%s/api/dashboards/uid/%s" % (
            DASHBOARDS[dashboard_name]['url'], DASHBOARDS[dashboard_name]['uid']
        )
    elif DASHBOARDS[dashboard_name]['type'] == DASHBOARD_TYPE_BY_NAME:
        url = "%s/api/dashboards/db/%s" % (
            DASHBOARDS[dashboard_name]['url'], dashboard_name
        )
    else:
        return {}

    return requests.get(url, headers=get_headers_by_grafana_url(DASHBOARDS[dashboard_name]['url'])).json()


def get_dashboard_panels(dashboard_name, dashboard_data):
    if DASHBOARDS[dashboard_name]['type'] == DASHBOARD_TYPE_BY_UID:
        return [
            {
                'id': panel['id'],
                'title': panel['title']
            }
            for panel in dashboard_data['dashboard']['panels']
        ]
    elif DASHBOARDS[dashboard_name]['type'] == DASHBOARD_TYPE_BY_NAME:
        return [
            {
                'id': panel['id'],
                'title': panel['title']
            }
            for row in dashboard_data['dashboard']['rows'] for panel in row['panels']
        ]

    return []


def d_panels(**kwargs):
    cmd_text = kwargs.get(u'string')
    cmd = kwargs.get(u'cmd')
    bot_type = kwargs.get(u'bot_type')

    dashboard_name = re.search(cmd['regexp'], cmd_text).group(1)

    if not dashboard_name:
        return u"Укажите название дашборда", MessageContentTypes.text

    dashboard_data = get_dashboard_data(dashboard_name)

    if 'dashboard' not in dashboard_data:
        return u"Дашборд '%s' не найден" % dashboard_name, MessageContentTypes.text

    panels_data = get_dashboard_panels(dashboard_name, dashboard_data)

    panels = [u"Весь дашборд %s" % dashboard_name] + [
        u"%s (id: %s)" % (
            panel['title'] if panel['title'] else u"Пустое название",
            panel['id']
        ) for panel in panels_data
    ]

    panels_text = u"\n".join([u"%d. %s" % (idx + 1, panels[idx]) for idx, item in enumerate(panels)])
    text = u"Список панелей для дашборда %s:\n%s" % (dashboard_name, panels_text)

    if bot_type == 'telegram':
        return {
            'text': text,
            'keyboard': (
                [[u"/d_graph %s" % dashboard_name]] +
                [[u"/d_graph %s %s" % (dashboard_name, panel['id'])] for panel in panels_data]
            )
        }
    else:
        return text, MessageContentTypes.text


def get_dashboard_url(dashboard_name, panel_id=None, render=False):
    url = DASHBOARDS[dashboard_name]['url']

    if render:
        url += "/render"

    if DASHBOARDS[dashboard_name]['type'] == DASHBOARD_TYPE_BY_UID:
        if panel_id:
            url += "/d-solo/%s/%s?panelId=%s" % (DASHBOARDS[dashboard_name]['uid'], dashboard_name, panel_id)
        else:
            url += "/d/%s/%s?" % (DASHBOARDS[dashboard_name]['uid'], dashboard_name)

    elif DASHBOARDS[dashboard_name]['type'] == DASHBOARD_TYPE_BY_NAME:
        if panel_id:
            url += "/dashboard-solo/db/%s?panelId=%s" % (dashboard_name, panel_id)
        else:
            url += "/dashboard/db/%s?" % dashboard_name

    if render and not panel_id:
        if DASHBOARDS[dashboard_name]['url'] == COMMON_GRAFANA_URL:
            url += "&width=%d&height=%d" % (
                DASHBOARDS[dashboard_name].get("width", 1200),
                DASHBOARDS[dashboard_name].get("height", 1200)
            )
        else:
            url += "&width=2000"

    return url


def get_dashboard_render(dashboard_name, panel_id=None):
    url = get_dashboard_url(dashboard_name, panel_id=panel_id, render=True)

    render_response = requests.get(url, headers=get_headers_by_grafana_url(DASHBOARDS[dashboard_name]['url']))

    if render_response.status_code == 200:
        return render_response.content

    return None


def d_graph(**kwargs):
    cmd_text = kwargs.get(u'string')
    cmd = kwargs.get(u'cmd')

    dashboard_name, panel_id = re.search(cmd['regexp'], cmd_text).groups('')

    if not dashboard_name:
        return u"Укажите название дашборда (и возможно id таблицы в нем)", MessageContentTypes.text

    if panel_id.endswith('n'):
        dashboard_data = get_dashboard_data(dashboard_name)

        if 'dashboard' not in dashboard_data:
            return u"Дашборд '%s' не найден" % dashboard_name, MessageContentTypes.text

        if panel_id[:-1].isdigit():
            ind = int(panel_id[:-1])

            panels_data = get_dashboard_panels(dashboard_name, dashboard_data)
            if ind < 1 or ind > len(panels_data):
                return (
                    u"Панели с порядковым номером %d на дашборде %s не существует" % (ind, dashboard_name),
                    MessageContentTypes.text
                )

            panel_id = panels_data[ind - 1]['id']
        else:
            return u"Неправильный формат номера панели", MessageContentTypes.text

    dashboard_render = get_dashboard_render(dashboard_name, panel_id=panel_id)
    if dashboard_render:
        caption = u'Дашборд: %s' % dashboard_name
        if panel_id:
            caption += u", панель: %s" % panel_id

        caption += u' <a href="%s">(перейти в графану)</a>' % get_dashboard_url(dashboard_name, panel_id=panel_id)
        return {'photo': BytesIO(dashboard_render), 'caption': caption, 'parse_mode': 'html'}, MessageContentTypes.image
    else:
        return u"График не найден", MessageContentTypes.text


def get_staff_login(user_id, bot_type):
    staff_login = user_id
    if bot_type == 'telegram':
        staff_login = get_login_by_telegram_username(user_id)
    elif bot_type == 'yamb':
        staff_login = get_login_by_uid(user_id)

    return staff_login


def review_list(**kwargs):
    bot_type = kwargs.get('bot_type')
    issues = st_client.issues.find('Queue: DIRECT Components: "design_review" Status: Open "Sort by": Key desc')

    text = u"Список открытых ревьюшных тикетов:\n%s" % (
        u"\n".join(
            "%s: %s" % (
                ('<a href="https://st.yandex-team.ru/%s">%s</a>' % (issue.key, issue.key)
                 if bot_type == 'telegram'
                 else issue.key),
                issue.summary
            ) for issue in issues
        )
    )
    if bot_type == 'telegram':
        return {
            'text': text,
            'keyboard': [[u"/start_meeting %s" % issue.key] for issue in issues],
            'parse_mode': 'html'
        }
    else:
        return text, MessageContentTypes.text


def start_meeting(**kwargs):
    cmd_text = kwargs.get(u'string')
    cmd = kwargs.get(u'cmd')
    bot_type = kwargs.get(u'bot_type')
    author = kwargs.get(u'author')

    search_params = re.match(cmd['regexp'], cmd_text, re.I)
    params = search_params.groupdict('')
    if not params.get('ticket', ''):
        return "Укажи тикет!"

    issue = st_client.issues[params['ticket']]
    if "design_review" not in [comp.display for comp in issue.components]:
        return "К сожалению, это не дизайн-ревью тикет :("

    opened_events = list(st_client.issues.find('"Is Subtask For": %s Status: open' % issue.key))
    if len(opened_events) > 0:
        return u"Тикет на встречу уже создан (%s)" % opened_events[0].key

    event_issue = st_client.issues.create(
        queue='DIRECT',
        summary=u'Дизайн-ревью %s' % datetime.datetime.today().strftime('%Y-%m-%d'),
        components="design_review_meeting",
        parent=issue.key
    )

    st_client.issues[issue.key].comments.create(
        text=u"@%s создал сегодняшнюю встречу, тикет: %s" % (get_staff_login(author, bot_type), event_issue.key)
    )

    text = u"Встреча создана! Держи тикет: %s" % (
        '<a href="https://st.yandex-team.ru/%s">%s</a>' % (event_issue.key, event_issue.key)
        if bot_type == 'telegram'
        else issue.key
    )

    if bot_type == 'telegram':
        return {
            'text': text,
            'parse_mode': 'html'
        }
    else:
        return text, MessageContentTypes.text


def write_comment_to_review_event(**kwargs):
    cmd_text = kwargs.get(u'string')
    cmd = kwargs.get(u'cmd')
    bot_type = kwargs.get(u'bot_type')
    author = kwargs.get(u'author')
    content_type = kwargs.get(u'content_type')

    opened_events = list(
        st_client.issues.find('Queue: DIRECT Components: "design_review_meeting" Status: open "Sort by": key desc')
    )

    if len(opened_events) == 0:
        return u"Нет открытых дизайн-ревьюшных встреч"

    if len(opened_events) > 1:
        return u"Найдено несколько открытых дизайн-ревьюшных встреч, надо бы оставить только одну!\nВот они: %s" % (
            u", ".join([event.key for event in opened_events])
        )

    opened_event = opened_events[0]

    if content_type in [MessageContentTypes.image, MessageContentTypes.document]:
        comment_id = st_client.issues[opened_event.key].comments.create(
            text=u"@%s:%s" % (get_staff_login(author, bot_type), cmd_text['caption']),
            attachments=[cmd_text['filename']]
        ).longId
    else:
        text_to_write = u"Пишем\U0001f60e"
        text_to_cancel = u"Не пишем\U0001f624"

        if cmd_text.startswith(text_to_write) or bot_type != 'telegram' or len(cmd_text) > len(text_to_write) + 30:
            comment_id = st_client.issues[opened_event.key].comments.create(
                text=u"@%s: %s" % (
                    get_staff_login(author, bot_type),
                    cmd_text[len(text_to_write) + 1 if cmd_text.startswith(text_to_write) else 0:]
                )
            ).longId
        elif cmd_text == text_to_cancel:
            return u"Ok..."
        else:
            return {
                'text': u'Коммент будет записан к тикету <a href="https://st.yandex-team.ru/%s">%s</a>.\nУверен? :)' % (
                    opened_event.key, opened_event.key
                ),
                'keyboard': [[u"%s %s" % (text_to_write, cmd_text), text_to_cancel]],
                'parse_mode': 'html'
            }

    common_prefix = u"Записал!\n"
    comment_link = u"https://st.yandex-team.ru/%s#%s" % (opened_event.key, comment_id)
    if bot_type == "telegram":
        return {
            'text': u'%s<a href="%s">(коммент)</a>' % (common_prefix, comment_link),
            'parse_mode': 'html'
        }
    else:
        return u"%s%s" % (common_prefix, comment_link), MessageContentTypes.text


def create_incident_ticket(**kwargs):
    cmd_text = kwargs.get(u'string')
    cmd = kwargs.get(u'cmd')
    bot_type = kwargs.get(u'bot_type')
    bot_name = kwargs.get(u'bot_name')
    author = kwargs.get(u'author')

    search_params = re.match(cmd['regexp'], cmd_text, re.I)
    params = search_params.groupdict('')

    title = params.get('title', '').strip()
    if not title:
        return "Укажи краткое название для тикета!"

    title = u"Директ, %s: %s" % (datetime.datetime.now().strftime("%Y-%m-%d"), title)

    staff_login = get_staff_login(author, bot_type)

    incident_issue = st_client.issues.create(
         queue='DIRECTINCIDENTS',
         summary=title,
         followers=[staff_login]
    )

    try:
        st_client.issues[incident_issue.key].comments.create(
            text=u"@%s создал тикет через бота для инцидентов" % staff_login
        )
    except:
        pass

    text = u"Тикет на инцидент создан! Вот он: %s" % (
        '<a href="https://st.yandex-team.ru/%s">%s</a>' % (incident_issue.key, incident_issue.key)
        if bot_type == 'telegram'
        else issue.key
    )

    if bot_type == 'telegram':
        return {
            'text': text,
            'parse_mode': 'html'
        }
    else:
        return text, MessageContentTypes.text


def convert_ticket_to_link(key, bot_type):
    return '<a href="https://st.yandex-team.ru/%s">%s</a>' % (key, key) if bot_type == 'telegram' else key


def list_incident_tickets(**kwargs):
    bot_type = kwargs.get('bot_type')
    open_issues = st_client.issues.find('Queue: DIRECTINCIDENTS Status: Open "Sort by": Key desc')
    resolved_issues = list(st_client.issues.find('Queue: DIRECTINCIDENTS Status: Resolved "Sort by": Key desc'))[:4]

    if open_issues:
        text = u"Список открытых инцидентных тикетов:\n%s" % (
            u"\n".join(
                "%s: %s" % (convert_ticket_to_link(issue.key, bot_type), issue.summary)
                for issue in open_issues
            )
        )
    else:
        text = u"Открытых инцидентных тикетов нет"

    text = u"Список инцидентных тикетов\U0001F4D6"
    for title, issues in [(u'Открытые', open_issues),
                          (u'Последние решенные', resolved_issues)]:
        text += u"\n\U000027A1%s:\n" % title
        if issues:
            text += u"\n".join(
                "%s: %s" % (convert_ticket_to_link(issue.key, bot_type), issue.summary) for issue in issues
            )
        else:
            text += u"Нет"
        text += "\n"

    text = text.strip()
    idx_of_number = len("DIRECTINCIDENTS-")
    if bot_type == 'telegram':
        return {
            'text': text,
            'keyboard': ([[u"/resolve %s" % issue.key[idx_of_number:]] for issue in open_issues] +
                [[u"/reopen %s" % issue.key[idx_of_number:]] for issue in resolved_issues]),
            'parse_mode': 'html'
        }
    else:
        return text, MessageContentTypes.text


def get_incident_ticket_or_error(ticket_number):
    if not ticket_number or not ticket_number.isdigit():
        return None, u"Укажи номер инцидентного тикета без очереди DIRECTINCIDENTS!"
    ticket = "DIRECTINCIDENTS-" + ticket_number

    try:
        return st_client.issues[ticket], ''
    except:
        return None, u"К сожалению, не могу найти тикет '%s' \U0001F614" % ticket


def reopen_or_resolve_incident_ticket(transition_type, **kwargs):
    cmd_text = kwargs.get(u'string')
    cmd = kwargs.get(u'cmd')
    bot_type = kwargs.get(u'bot_type')
    author = kwargs.get(u'author')

    search_params = re.match(cmd['regexp'], cmd_text, re.I)
    params = search_params.groupdict('')

    issue, error = get_incident_ticket_or_error(params.get('ticket', ''))

    if error:
        return error

    try:
        transitions = ['inReview', 'open'] if transition_type == 'reopen' else ['resolved']
        for trans in transitions:
            for transition in st_client.issues[issue.key].transitions.get_all():
                if transition.id == trans:
                    st_client.issues[issue.key].transitions[trans].execute()
                    break

        st_client.issues[issue.key].comments.create(
            text=u"@%s %s тикет через бота" % (
                get_staff_login(author, bot_type),
                u"переоткрыл" if transition_type == 'reopen' else u"зарезолвил"
            )
        )
        return u"Готово!\U0001F60E"
    except:
        logging.exception("unexpected exception")
        return u"К сожалению, не могу %s тикет '%s' \U0001F614" % (
            u"переоткрыть" if transition_type == 'reopen' else u"зарезолвить",
            issue.key
        )


def reopen_incident_ticket(**kwargs):
    return reopen_or_resolve_incident_ticket('reopen', **kwargs)


def resolve_incident_ticket(**kwargs):
    return reopen_or_resolve_incident_ticket('resolve', **kwargs)


def rename_incident_ticket(**kwargs):
    cmd_text = kwargs.get(u'string')
    cmd = kwargs.get(u'cmd')
    bot_type = kwargs.get(u'bot_type')
    author = kwargs.get(u'author')

    search_params = re.match(cmd['regexp'], cmd_text, re.I)
    params = search_params.groupdict('')

    issue, error = get_incident_ticket_or_error(params.get('ticket', ''))

    if error:
        return error

    title = params.get('title', '').strip()
    if not title:
        return u"Укажи новое название для тикета!"

    try:
        st_client.issues[issue.key].update(summary=title)
        st_client.issues[issue.key].comments.create(
            text=u"@%s поменял название тикета на '%s' через бота" % (
                get_staff_login(author, bot_type),
                title
            )
        )
        return u"Готово!\U0001F60E"
    except:
        return u"К сожалению, не могу изменить название на '%s' \U0001F614" % title


def post_comment_to_incident_ticket(**kwargs):
    cmd_text = kwargs.get(u'string')
    cmd = kwargs.get(u'cmd')
    bot_type = kwargs.get(u'bot_type')
    author = kwargs.get(u'author')
    content_type = kwargs.get(u'content_type')

    if content_type in [MessageContentTypes.image, MessageContentTypes.document]:
        search_params = re.match(cmd['regexp'], cmd_text['caption'], re.I)
    else:
        search_params = re.match(cmd['regexp'], cmd_text, re.I)
    params = search_params.groupdict('')

    issue, error = get_incident_ticket_or_error(params.get('ticket', ''))

    if error:
        return error

    text = params.get('message', '').strip()

    try:
        if content_type in [MessageContentTypes.image, MessageContentTypes.document]:
            comment_id = st_client.issues[issue.key].comments.create(
                text=u"@%s:%s" % (get_staff_login(author, bot_type), text),
                attachments=[cmd_text['filename']]
            ).longId
        else:
            if not text:
                return u"Укажи текст, который надо записать к тикету!"
            comment_id = st_client.issues[issue.key].comments.create(
                text=u"@%s: %s" % (get_staff_login(author, bot_type), text)
            ).longId
    except:
        logging.exception("unexpected exception")
        return u"К сожалению, у меня не получилось \U0001F614"

    common_prefix = u"Записал!\U0001F60E\n"
    comment_link = u"https://st.yandex-team.ru/%s#%s" % (issue.key, comment_id)
    if bot_type == "telegram":
        return {
            'text': u'%s<a href="%s">\U0001F517Ссылка на коммент</a>' % (common_prefix, comment_link),
            'parse_mode': 'html'
        }
    else:
        return u"%s%s" % (common_prefix, comment_link), MessageContentTypes.text


def get_chat_id(**kwargs):
    return u"chat_id: %s" % kwargs.get(u'chat_id')


def print_help(**kwargs):
    bot_name = kwargs.get('bot_name')
    msg = "\n".join([
        u"%s - %s" % (cmd['name'], cmd['description'])
        for cmd in commands_common + commands.get(bot_name, [])
    ])
    extra = ''
    if bot_name == 'direct_incidents':
        extra = u"\nВ командах под тикетом подразумевается только его номер, без очереди DIRECTINCIDENTS!"
    return (
        u'Привет! Это %s бот.%s\n\nСписок команд, которые я могу выполнить:\n%s' % (bot_name, extra, msg),
        MessageContentTypes.text
    )


def compose_response(message, author, bot_type, bot_name, chat_id, chat_alias, chat_type, message_type, content_type, chat_info):
    resp = (u'', None)

    global SESSION_ID
    SESSION_ID += 1

    if content_type in [MessageContentTypes.image, MessageContentTypes.document]:
        text_cmd = message['caption']
    else:
        text_cmd = message

    for cmd in commands_common + commands.get(bot_name, []):
        found_cmd = re.search(cmd['regexp'], text_cmd, re.I | re.U)
        if found_cmd:
            logging.info('cmd for message: %s' % cmd[u'cmd'].__name__)

            try:
                resp = cmd[u'cmd'](
                    string=message,
                    author=author,
                    cmd=cmd,
                    bot_type=bot_type,
                    bot_name=bot_name,
                    chat_id=chat_id,
                    chat_alias=chat_alias,
                    chat_type=chat_type,
                    message_type=message_type,
                    content_type=content_type,
                    chat_info=chat_info
                )

            except Exception as e:
                logging.exception('Exception occured')
                resp = u'Произошла ошибка :('

            # по дефолту считаем, что текст
            if not type(resp) is tuple:
                resp = tuple([resp, MessageContentTypes.text])

            break

    return resp


commands_common = [
    {
        'name': u'/chat_id',
        'regexp': r'^/chat_id\s*$',
        'description': u'получить chat_id текущего чата',
        'cmd': get_chat_id
    },
    {
        'name': '/help',
        'regexp': r'^/help\s*$',
        'description': u'вывести хэлп и список команд',
        'cmd': print_help
    }
]

commands = {
    'direct_incidents': [
        {
            'name': u'/create [TICKET_TITLE]',
            'regexp': r'/create\s*(?P<title>.*)\s*$',
            'description': u'создать тикет в очереди DIRECTINCIDENTS с заданным заголовком',
            'cmd': create_incident_ticket
        },
        {
            'name': u'/list',
            'regexp': r'/list\s*',
            'description': u'вывести последние инцидентные тикеты',
            'cmd': list_incident_tickets
        },
        {
            'name': u'/reopen [TICKET]',
            'regexp': r'/reopen\s*(?P<ticket>[^\s]*)\s*$',
            'description': u'переоткрыть решенный инцидентный тикет',
            'cmd': reopen_incident_ticket
        },
        {
            'name': u'/resolve [TICKET]',
            'regexp': r'/resolve\s*(?P<ticket>[^\s]*)\s*$',
            'description': u'зарезолвить открытый инцидентный тикет)',
            'cmd': resolve_incident_ticket
        },
        {
            'name': u'/rename [TICKET] [TITLE]',
            'regexp': r'/rename\s*(?P<ticket>[^\s]*)\s*(?P<title>.*)\s*$',
            'description': u'переименовать инцидентный тикет',
            'cmd': rename_incident_ticket
        },
        {
            'name': u'/post [TICKET] [MESSAGE or IMAGE or FILE]',
            'regexp': r'/post\s*(?P<ticket>[^\s]*)\s*(?P<message>.*)\s*$',
            'description': u'запостить к инцидентному тикету сообщение, возможно с картинкой или файлом (команда пишется в заголовке)',
            'cmd': post_comment_to_incident_ticket
        }
    ],
    'design_review': [
        {
            'name': u'/list',
            'regexp': r'^/list\s*$',
            'description': u'вывести список текущих открытых ревьюшных тикетов',
            'cmd': review_list
        },
        {
            'name': u'/start_meeting [TICKET]',
            'regexp': ur'^/start_meeting\s*(?P<ticket>[^\s]*)\s*$',
            'description': u'начать дизайн-ревью встречу данного тикета',
            'cmd': start_meeting
        },
        {
            'name': u'/текст без команды',
            'regexp': ur'.*',
            'description': u'если есть тикет-событие для дизайн-ревью, то к нему записать комментарий',
            'cmd': write_comment_to_review_event,
        }
    ],
    'herald': [
        {
            'name': u'/prod_info',
            'regexp': ur'^/prod_info\b',
            'description': u'получить информацию о продакшене',
            'cmd': get_prod_info
        },
        {
            'name': u'/releases_for_deploy',
            'regexp': ur'^/releases_for_deploy\s*',
            'description': u'получить информацию о текущих релизах для выкладки',
            'cmd': releases_for_deploy
        },
        {
            'name': u'/выложить',
            'regexp': ur'(?:вылож(?:ить|ите|ишь|им|у)|обновить|обновите|обновишь|выкатить|катнете|катните|катни|катнешь|выкатите|выкатишь|limtest[12]|лимтест|залочить)',
            'description': u'получить информацию о выкладке релиза',
            'cmd': check_ticket_for_deploy
        },
        {
            'name': '/devsup_status',
            'regexp': r'^/devsup_status\s*$',
            'description': u'узнать статус по devsup-тикетам',
            'cmd': devsup_status
        },
        {
            'name': '/duty_call',
            'regexp': r'/duty_call\b',
            'description': u'призвать дежурных, жалоба в проде',
            'cmd': duty_call
        },
        {
            'name': '/duty_respond',
            'regexp': r'/duty_respond\b',
            'description': u'дежурный готов разбираться с проблемой',
            'cmd': duty_respond
        },
        {
            'name': '/duty_cancel',
            'regexp': r'/duty_cancel\b',
            'description': u'отбой, проблемы не наблюдается, отмена ложного призыва',
            'cmd': duty_cancel
        },
        {
            'name': '/mon',
            'regexp': r'^/mon\b\s*(?P<tags>[^\s]*)\s*(?P<statuses>[^\s]*)\s*(?P<quiet>[^\s]*)\s*$',
            'description': (u'[TAG] [crit,warn,ok] [quiet] узнать статистику juggler-событий по тэгу (можно указать' +
                u' несколько через запятую), без указания тэга выводит список всех тэгов'),
            'cmd': mon
        },
        {
            'name': '/mon_v',
            'regexp': r'^/mon_v\b\s*(?P<tags>[^\s]*)\s*(?P<statuses>[^\s]*)\s*(?P<quiet>[^\s]*)\s*$',
            'description': (u'[TAG] [crit,warn,ok] [quiet] то же самое, что и mon + выводит описания событий + можно' +
                u' указать отдельное событие с префиксом "event:" (например,' +
                u' "/mon_v event:direct.prod_xtradb::xtradb_wsrep_local_state_comment")'),
            'cmd': mon
        },
        {
            'name': '/mon_title',
            'regexp': r'^/mon_title\b\s*(?P<tags>[^\s]*)\s*(?P<statuses>[^\s]*)\s*(?P<quiet>[^\s]*)\s*$',
            'description': u'то же, что и /mon, но только меняет название чата в зависимости от количества горящих проверок',
            'cmd': mon
        },
        {
            'name': '/appduty',
            'regexp': r'^/appduty\s*(?P<quiet>[^\s]*)\s*$',
            'description': (u'[quiet] вывести список тикетов про аварии в Appduty-чате и поменять название чата в' +
                u' зависимости от количества проблем (с quite только меняет название чата)'),
            'cmd': appduty,
        },
        {
            'name': '/d_list',
            'regexp': r'^/d_list\s*$',
            'description': u'вывести список дашбордов графаны',
            'cmd': d_list
        },
        {
            'name': '/d_panels',
            'regexp': r'^/d_panels\s*([^\s]*)\s*$',
            'description': u'[DASHBOARD_URI] вывести список панелей дашборда',
            'cmd': d_panels
        },
        {
            'name': '/d_graph',
            'regexp': r'^/d_graph\s*([^\s]*)\s*([^\s]*)\s*$',
            'description': (u'[DASHBOARD_URI][PANEL] показать график дашборда или панели в нем (у панели либо id, либо' +
                u' порядковый номер на дашборде с суффиксом n)'),
            'cmd': d_graph
        },
        {
            'name': u'/ticket',
            'regexp': ur'(?:DIRECT|DAHW|DIRECTADMIN|DIRECTMIGR|DIRECTMOD|INFRASTRUCTUREPI|PI|TESTIRT)-[0-9]+',
            'description': (u'[TICKET] получить информацию о тикете (также достает тикеты из пересланных сообщений в' +
                u' личку без указания команды)'),
            'cmd': get_ticket_info
        }
    ]
}
