# -*- coding: utf-8 -*-

import sys

sys.path.insert(0, '/var/lib')
sys.path.insert(0, '/opt/direct-py/startrek-python-client-sni-fix')

import logging
import bot_commands as bc
import os
import yaml
import datetime
import re
import traceback
import getpass
import urllib3
import threading
import requests
import time
import tempfile

logging.getLogger("urllib3").setLevel(logging.WARNING)


class DirectUsers(object):
    def __init__(self, users_url, users_getter):
        self.lock = threading.Lock()
        self.st_token = bc.get_token(u'/etc/direct-tokens/startrek')
        self.__users = set()
        self.users_url = users_url
        self.users_getter = users_getter
        self.current_timer = None
        self.__get_direct_users()

    def __get_direct_users(self):
        self.current_timer = threading.Timer(3600, self.__get_direct_users)
        self.current_timer.start()
        try:
            req = requests.get(self.users_url, headers={'Authorization': 'OAuth %s' % (self.st_token)})
            new_users = self.users_getter(req.json())

            with self.lock:
                self.__users = new_users
        except:
            pass

    def get_users(self):
        with self.lock:
            return self.__users

    def stop(self):
        self.current_timer.cancel()


class Bot(object):
    MAX_TRIES_TO_SEND = 3
    TIME_TO_SLEEP = 1
    alias2chatid = {}

    def __init__(self, config_file, need_logging):
        self.direct_users = []
        with open(config_file, "r") as config_fh:
            self.config = yaml.load(config_fh)

        self.chatid2alias = {chat_id: alias for alias, chat_id in self.alias2chatid.items()}

        if need_logging:
            logging.basicConfig(
                format=u'%(levelname)-8s [%(asctime)s] %(message)s',
                level=logging.INFO,
                filename=self.config['log_file']
            )

        self.bot_name = self.config.get('bot_name', 'herald')
        self.bot_token = None

        if 'token_file' in self.config:
            with open(self.config['token_file'], "r") as token_file:
                self.bot_token = token_file.read().strip()

    def reply_to_message(self, message, author, chat_id, chat_type, message_type, content_type,
                         one_time_reply=False, reply_to_message_id=None):
        logging.info('got message')

        if not one_time_reply and author not in self.direct_users.get_users():
            logging.info("not Direct user %s" % author)
            return

        if chat_id in self.alias2chatid:
            chat_id = self.alias2chatid[chat_id]

        chat_alias = ""
        if chat_id in self.chatid2alias:
            chat_alias = self.chatid2alias[chat_id]

        resp = ''
        resp_type = ''

        try:
            resp, resp_type = bc.compose_response(
                message,
                author,
                self.bot_type,
                self.bot_name,
                chat_id,
                chat_alias,
                chat_type,
                message_type,
                content_type,
                chat_info=self.get_chat_info(chat_id)
            )
        except:
            logging.error(u'error while compose response\n%s' % traceback.format_exc().decode('utf-8'))
            if one_time_reply:
                raise

        if not resp:
            logging.warning('empty message to send')
            return

        for i in xrange(self.MAX_TRIES_TO_SEND):
            try:
                self.send_reply(resp, resp_type, chat_id, reply_to_message_id=reply_to_message_id, chat_type=chat_type)
                logging.info('message sent')
                break
            except:
                if i + 1 == self.MAX_TRIES_TO_SEND:
                    logging.exception("unexpected error")
                    if one_time_reply:
                        raise
                else:
                    time.sleep(self.TIME_TO_SLEEP)

        return

    def get_chat_info(self, chat_id):
        return {}

    def get_direct_users(self):
        self.direct_users = DirectUsers(self.users_url, self.users_getter)

    def send_reply(self, message, message_type, chat_id, reply_to_message_id=None, chat_type=None):
        pass

    def __del__(self):
        if isinstance(self.direct_users, DirectUsers):
            self.direct_users.stop()
        self.direct_users = None


class TelegramBot(Bot):
    exec "import telepot" in globals()
    exec "from telepot.loop import MessageLoop" in globals()
    exec "from telepot.namedtuple import ReplyKeyboardMarkup, KeyboardButton, ReplyKeyboardRemove" in globals()

    bot_type = 'telegram'

    users_url = 'https://staff-api.yandex-team.ru/v3/persons?_doc=0&_page=1&_limit=1000&_fields=accounts' + \
                '&official.is_dismissed=false&groups.group.ancestors.url=yandex_monetize_search_direct_interface_dep16266' + \
                '&accounts.type=telegram'

    @staticmethod
    def users_getter(response):
        """Выбирает из ответа стаффа все телеграм аккаунты (может быть несколько аккаунтов на одного человека)"""
        logins = set()
        for user in response['result']:
            for account in user[u'accounts']:
                if account['type'] == 'telegram':
                    logins.add(account['value'])
        return logins

    # id чата можно узнать через Геральда: добавить в чат бота @yd_herald_bot и написать /chat_id@yd_herald_bot
    alias2chatid = {
        "Direct.Monitoring": "-1001149314926",
        "Direct.Monitoring.Calls": "-1001303668967",
        "Direct.Admin.Chat": "-1001063817057",
        "Direct.AppDuty.Monitoring": "-1001447316217",
        "Direct.NP.Monitoring": "-1001251297561",
        "Direct.App.Duty": "-1001350752218",
        "Lena.Test": "-1001166016259",
        "Grafana Daily": "-1001210825157"
    }
    MAX_MESSAGE_LENGTH = 4096

    def __init__(self, config_file, need_logging=True):
        super(TelegramBot, self).__init__(config_file, need_logging)
        self.bot = telepot.Bot(self.bot_token)
        self.start_time = datetime.datetime.now()
        self.bot_login = u'@' + self.bot.getMe()['username']

    def run(self):
        self.get_direct_users()
        MessageLoop(self.bot, self.handle_message).run_forever(timeout=10)

    def handle_message(self, msg):
        # пропускаем старые
        if datetime.datetime.fromtimestamp(msg['date']) < self.start_time:
            return

        content_type, chat_type, chat_id = telepot.glance(msg)
        chat_id = str(chat_id)

        if chat_type == 'private':
            chat_type = bc.ChatTypes.private
        elif chat_type == 'channel':
            chat_type = bc.ChatTypes.channel
        else:
            chat_type = bc.ChatTypes.group

        text = None
        author = msg['from']['username']
        reply_to_message_id = msg['message_id']

        if content_type == 'text':
            content_type = bc.MessageContentTypes.text
            cmd_message = re.sub(ur'\s*%s\s*' % self.bot_login, '', msg['text'])
        elif content_type in ['photo', 'document']:
            cmd_message = {'caption': msg.get('caption', '')}
            if content_type == 'photo':
                file_id = max((photo['file_size'], photo['file_id']) for photo in msg[content_type])[1]
                content_type = bc.MessageContentTypes.image
            else:
                file_id = msg[content_type]['file_id']
                content_type = bc.MessageContentTypes.document

            f = tempfile.NamedTemporaryFile(delete=False, prefix=author + "_")
            self.bot.download_file(file_id, f.file)
            f.close()

            cmd_message['filename'] = f.name
        else:
            return

        message_type = bc.MessageTypes.forward if "forward_from" in msg else bc.MessageTypes.text

        self.bot.sendChatAction(chat_id, 'typing')
        self.reply_to_message(
            cmd_message, author, chat_id,
            chat_type, message_type, content_type, reply_to_message_id=reply_to_message_id
        )

        if content_type in [bc.MessageContentTypes.image, bc.MessageContentTypes.document]:
            os.unlink(cmd_message['filename'])

        return

    def send_reply(self, message, message_type, chat_id, reply_to_message_id=None, chat_type=None):
        kwargs = {}

        if chat_type != bc.ChatTypes.channel:
            kwargs['reply_markup'] = ReplyKeyboardRemove(selective=True)

        if reply_to_message_id:
            kwargs['reply_to_message_id'] = reply_to_message_id

        message_to_send = message

        if message_type == bc.MessageContentTypes.text:
            if isinstance(message, dict):
                message_to_send = message['text']

                # клавиатуру показываем только тому, кто отправил сообщение
                if 'keyboard' in message and reply_to_message_id:
                    kwargs['reply_markup'] = ReplyKeyboardMarkup(
                        keyboard=message['keyboard'],
                        resize_keyboard=True,
                        one_time_keyboard=True,
                        selective=True
                    )

                if 'parse_mode' in message:
                    kwargs['parse_mode'] = message['parse_mode']

                if 'title' in message:
                    self.bot.setChatTitle(chat_id, message['title'])

                if 'description' in message:
                    self.bot.setChatDescription(chat_id, message['description'])

            # если длина сообщения больше максимальной, то отправляем по частям
            if not message_to_send:
                return
            elif len(message_to_send) > self.MAX_MESSAGE_LENGTH:
                for i in xrange(0, len(message_to_send), self.MAX_MESSAGE_LENGTH):
                    self.bot.sendMessage(chat_id, message_to_send[i: i + self.MAX_MESSAGE_LENGTH], **kwargs)
            else:
                self.bot.sendMessage(chat_id, message_to_send, **kwargs)

        elif message_type == bc.MessageContentTypes.image:
            self.bot.sendChatAction(chat_id, 'upload_photo')

            if isinstance(message, dict):
                message_to_send = message['photo']

                if 'caption' in message:
                    kwargs['caption'] = message['caption']

                if 'parse_mode' in message:
                    kwargs['parse_mode'] = message['parse_mode']

            self.bot.sendPhoto(chat_id, message_to_send, **kwargs)

    def get_chat_info(self, chat_id):
        return self.bot.getChat(chat_id)


class YambBot(Bot):
    exec "import telepot_yamb" in globals()
    exec "from telepot_yamb.loop import MessageLoop" in globals()
    exec "from telepot_yamb.namedtuple import ReplyKeyboardMarkup, KeyboardButton" in globals()

    bot_type = 'yamb'

    users_url = 'https://staff-api.yandex-team.ru/v3/persons?_doc=0&_page=1&_limit=1000&_fields=uid' + \
                '&official.is_dismissed=false&groups.group.ancestors.url=yandex_monetize_search_direct_interface_dep16266'
    users_getter = staticmethod(lambda x: set([user['uid'] for user in x['result']]))
    alias2chatid = {
        "Direct.Admin.Chat": u"0/0/f10c386c-4b05-4333-836c-b2f14473d97f"
    }

    def __init__(self, config_file, need_logging=True):
        super(YambBot, self).__init__(config_file, need_logging)
        self.start_time = datetime.datetime.now()
        self.bot = telepot_yamb.Bot(self.bot_token)
        self.bot_login = u'@' + self.bot.getMe()['guid']

    def run(self):
        self.get_direct_users()
        MessageLoop(self.bot, self.handle_message).run_forever(relax=0.5)

    def handle_message(self, msg):
        # пропускаем старые
        if datetime.datetime.fromtimestamp(msg['date']) < self.start_time:
            return

        text = ''
        if 'text' in msg:
            text = msg['text']
            message_type = bc.MessageTypes.text
        elif 'reply_to_message' in msg and 'text' in msg['reply_to_message']:
            text = msg['reply_to_message']['text']
            message_type = bc.MessageTypes.forward

        if not text:
            return

        text = re.sub(ur'\s*%s\s*' % self.bot_login, '', text)
        if msg['chat']['id'].replace('-', '').find(msg['from']['id']) != -1:
            chat_type = bc.ChatTypes.private
        else:
            chat_type = bc.ChatTypes.group

        self.reply_to_message(
            text, str(msg['from']['uid']), msg['chat']['id'], chat_type, message_type, bc.MessageContentTypes.text
        )

        return

    def send_reply(self, message, message_type, chat_id, reply_to_message_id=None, chat_type=None):
        kwargs = {}
        message_to_send = message

        if message_type == bc.MessageContentTypes.text:
            if isinstance(message, dict):
                message_to_send = message['text']

                if 'keyboard' in message:
                    kwargs['reply_markup'] = ReplyKeyboardMarkup(
                        keyboard=message['keyboard'],
                        resize_keyboard=True,
                        one_time_keyboard=True
                    )

                if 'parse_mode' in message:
                    kwargs['parse_mode'] = message['parse_mode']

            if not message_to_send:
                return

            self.bot.sendMessage(chat_id, message_to_send, **kwargs)

        elif message_type == bc.MessageContentTypes.image:
            if isinstance(message, dict):
                message_to_send = message['photo']

                if 'caption' in message:
                    kwargs['caption'] = message['caption']

                if 'parse_mode' in message:
                    kwargs['parse_mode'] = message['parse_mode']

            self.bot.sendPhoto(chat_id, message_to_send, **kwargs)


class CliBot(Bot):
    bot_type = 'cli'

    users_url = 'https://staff-api.yandex-team.ru/v3/persons?_doc=0&_page=1&_limit=1000&_fields=login' + \
                '&official.is_dismissed=false&groups.group.ancestors.url=yandex_monetize_search_direct_interface_dep16266'
    users_getter = staticmethod(lambda x: set([user['login'] for user in x['result']]))

    def __init__(self, config_file, need_logging=True):
        super(CliBot, self).__init__(config_file, need_logging)
        self.author = getpass.getuser()

    def run(self):
        self.get_direct_users()
        while 1:
            try:
                text = raw_input(u'>> Write command (type /help to see commands):\n<< ').decode('utf-8')
            except:
                exit(0)
            self.reply_to_message(
                text, self.author, "chat_id", bc.ChatTypes.private, bc.MessageTypes.text, bc.MessageContentTypes.text
            )

        return

    def send_reply(self, message, message_type, chat_id, reply_to_message_id=None, chat_type=None):
        message_to_send = message
        if isinstance(message, dict):
            message_to_send = message['text']
        if message_type == bc.MessageContentTypes.text:
            print u">>\n" + "\n".join([u" " * 3 + line for line in message_to_send.split("\n")])

