#!/usr/bin/perl

use common::sense;
use Yandex::Clickhouse;
use Getopt::Long qw/:config no_ignore_case/;
use Data::Dump 'pp';
use JSON;
use Term::ReadLine;
use Pod::Usage;
$|++;

=head1 USAGE

clhi.pl - Clickhouse interactive

Usage:

clhi.pl OPTIONS

OPTIONS:

 -h --host <hostname>
 -u --user <user>
 -n --names -- show column names
 -e --exec | --query <query> -- non-interactive mode
 -f --format <pp|tsv|json> -- output format
 -O --file <file> -- output to file, requires --exec
    output to file is allways tsv (TabSeparated[WithNames] in ClH terms)
 -T --timeout <sec> -- query timeout (600 by default)
 -p --profile <profile> -- set profile
 -s --settings key=value -- clickhouse settings
    -s max_threads=2 -s max_block_size=1024
 --help -- show help and exit

Ctrl-c | quit | exit to exit

=cut

my @settings;
my %opt;
$opt{settings} = \@settings;
GetOptions(\%opt,
    "host|h=s",
    "user|u=s",
    "names|n",
    "exec|e=s",
    "format|f=s",
    "file|O=s",
    "timeout|T=s",
    "profile|p=s",
    "settings|s=s@",
    "help",
) or die pod2usage();

if ($opt{help}) {
    pod2usage()
}

$opt{host} //= 'ppchouse01e.yandex.ru';
$opt{format} //= 'pp';

my $clh = Yandex::Clickhouse->new(
    host => $opt{host},
    user => $opt{user},
    port => 8123,
    timeout => $opt{timeout} // 600,
    profile => $opt{profile},
    settings => {
        map { split /=/,$_,2 } @settings,
    }
);

if ($opt{names}) {
    $clh->query_format('TabSeparatedWithNames');
}

if ($opt{file}) {
    if (!$opt{exec}) {
        die "--file requires -e";
    }
    $clh->output_file($opt{file});
    $clh->query($opt{exec});
    exit;
}

if ($opt{exec}) {
    exec_query($opt{exec});
    exit;
}

my $term = Term::ReadLine->new('clh-repl');

while (1) {
    my $query = $term->readline("[$opt{host}] :) ");
    if (!defined $query || $query =~ /^(q|quit|exit)$/) {
        print "bye\n";
        last;
    }
    next unless $query;
    exec_query($query);
    $term->addhistory($query) if $query =~ /\S/;
}

sub exec_query
{
    my $query = shift;
    $query =~ s!;\s*$!!;
    eval {
        my $res = $clh->query($query)->tsv(names => $opt{names});
        print_res($res);
        #print pp $res;
        #print "\n";
    };
    warn $@ if $@;
}

sub print_res
{
    my ($res) = @_;
    if ($opt{format} eq 'pp') {
        print pp $res;
        print "\n";
    }
    elsif ($opt{format} eq 'tsv') {
        for my $row (@$res) {
            print ''.(join "\t", @$row)."\n";
        }
    }
    elsif ($opt{format} eq 'json') {
        print to_json($res, { pretty => 1 });
        print "\n";
    }
    else {
        warn "unknown format '$opt{format}'";
        die pod2usage();
    }
}

