#!/usr/bin/perl

=head1 DESCRIPTION

 find-packages-to-dmove.pl -c etc/direct-dmove.yaml -t packages/yandex-direct/debian/control -f testing

 Поиск "что надо dmove-нуть на dist'е, чтобы пакеты выложились".

 Получает файл control и ветку в репозитории. 
 Ищет: есть ли в указанной ветке что-либо, что участвует в зависимостях в указанном control
 Логика: если есть -- это надо передвинуть

 Вывод совместим с dmove_*_from_file

 Параметры:
     -p, --package - передать название пакета, например --package "yandex-package (>= 1.23)" (можно передать несколько раз)
     -c, --conf - путь до файла с конфигом
     -t, --control-file - путь до control-файла
     -r, --repository - ссылка на репозиторий
     -f, --from - unstable/testing/stable

=head1 TODO

  + переименовать
  + не привязываться к direct-precise (брать из PRojectSpecific или еще как-то)
  + не привязываться к yandex-direct
  * интеграцию в релизный шелл

=cut 



use strict;
use warnings;

use Net::INET6Glue::INET_is_INET6;
use YAML;
use LWP::UserAgent;
use IO::Uncompress::Gunzip qw(gunzip $GunzipError) ;
use Path::Tiny;
use Getopt::Long;

my $ubuntu_release = `lsb_release -c`;
chomp $ubuntu_release;
$ubuntu_release =~ s/Codename:\s*//;

run() unless caller();

sub run
{
    my $opt = parse_options();

    # если ищем для какого-то конкретного пакета
    my @package_deps;
    if ($opt->{package}) {
        @package_deps = @{find_deps_for_package($opt->{package}, $opt) || []};
    }

    #for my $rep ( qw/unstable testing stable/ ){
    for my $rep ( $opt->{from} ){
        my %IN_REPO;
        my %DEPS;
        for my $arc (qw/i386 amd64 all/){
            my @packages = get_packages($rep, $arc, $opt->{repository});

            for my $p ( @packages ){
                $p =~ m/Package:\s+(\S*)\n.*?Version:\s+(\S+)/s;
                die "can't parse package and version from description:\n$p" unless $1 && $2;
                my ($package, $version) = ($1, $2);
                $IN_REPO{$package}{$version} = 1;

                $DEPS{$package}{$version} = parse_deps($p, 2); 
            }
        }

        my @deps;
        if ($opt->{package}) {
            @deps = @package_deps;
        } else {
            my $control = path($opt->{control_file})->slurp_utf8;
            my @p = ($control =~ /(Package: .*?Depends: .*?Description:)/gsm);
            die "can't parse control file, stop\n" unless @p;
            @deps = map { @{parse_deps($_, 1)} } @p;
        }

        my %NEVER_DMOVE = map { $_ => 1 } @{ $opt->{never_dmove}->{$rep} || [] };

        my @to_check = @deps;
        my %checked;
        my %to_dmove;

        my $i = 0;
        while ( @to_check ){
            die "dependencies went too deep" if $i++ >= 3000;
            my $d = shift @to_check;
            # если пакет уже проверяли -- не повторяем, выкидываем из очереди
            next if $checked{$d->{package}}->{$d->{version}};

            $checked{$d->{package}}->{$d->{version}} = 1;

            # если пакет требуется не с ">=" -- ничего не проверяем, выкидываем из очереди
            next if $d->{cmp} ne ">=";

            # если требуемый пакет не лежит в нашем репозитории -- больше ничего не делаем
            next if ! $IN_REPO{$d->{package}}->{$d->{version}};

            # если пакет запрещено перемещать -- пропускаем, больше ничего не делаем
            next if $NEVER_DMOVE{$d->{package}};

            # требуемый пакет лежит в нашем репозитории, надо dmove + рекурсивно проверить зависимости
            push @{$to_dmove{$d->{package}}}, $d->{version};
            push @to_check, @{$DEPS{$d->{package}}->{$d->{version}}};
        }

        for my $package ( sort keys %to_dmove ){
            for my $version ( sort @{$to_dmove{$package}} ){
                print "$package $version\n";
            }
        }
    }
}


sub find_deps_for_package
{
    my ($package_str, $opt) = @_;

    my ($main_package, $main_version, $main_deps);
    die "can't parse package name or version" unless $package_str =~ /([^=]+)=(.+)/;
    ($main_package, $main_version) = ($1, $2);

    for my $rep (qw/unstable testing stable/) {
        for my $arc (qw/i386 amd64 all/){
            my @packages = get_packages($rep, $arc, $opt->{repository});
            for my $p ( @packages ){
                $p =~ m/Package:\s+(\S*)\n.*?Version:\s+(\S+)/s;
                die "can't parse package and version from description:\n$p" unless $1 && $2;
                my ($package, $version) = ($1, $2);
                if ($package eq $main_package && $version eq $main_version) {
                    $main_deps = parse_deps($p, 2);
                    return $main_deps;
                }
            }
        }
    }
    return undef;
}


sub get_packages
{
    my ($rep, $arc, $repository) = @_;

    my $ua = LWP::UserAgent->new;
    $ua->timeout(10);

    my $Packages_url = "$repository/$rep/$arc/Packages.gz";

    my $response;
    for (1 .. 3){
        $response = $ua->get($Packages_url);
        last if $response->is_success;
    }
    die "can't download $Packages_url" unless $response->is_success;
    my $packages_gz = $response->decoded_content;
    my $packages = '';

    gunzip \$packages_gz => \$packages or die "gunzip failed: $GunzipError\n";
    my @packages = split /\n\n/, $packages;
    return @packages;
}


sub parse_deps
{
    my ($control, $type) = @_;
    
    my @dep_texts;
    # если type == 1, то перед нами control-файл
    # если 2, то перед нами файл со списком пакетов из репозитория
    # если 3, то нам переданы название пакета с версией
    if ($type == 1) {
        @dep_texts = ($control =~ /Package: .*?Depends: (.*?)(?:Description:|Conflicts:)/gsm);
    } elsif ($type == 2) {
        @dep_texts = ($control =~ /Depends: (.*?)$/gsm);
    } elsif ($type == 3) {
        @dep_texts = $control;
    }
    my $dep_text = join "\n", @dep_texts;
    # массив отельных строчек вида 'libtemplate-perl (= 2.22-1)'
    my @deps = map { split /\s*\|\s*/ } grep {$_} split /,\s*/, join "\n", grep {!/^#/} split "\n", $dep_text;
    my @res;
    for my $dep ( @deps ){
        $dep =~ /^\s*([^ \(]+)\s*(.*)/;
        my ($package, $cond) = ($1, $2);
        $cond =~ s/^\((.*)\)$/$1/;
        my $parsed_dep = { package => $package, version => '', cmp => '' };
        next unless $cond;
        $cond =~ /^(=|<=|>=|<<?|>>?)\s*(\d.*)$/;
        my ($cmp, $required_version) = ($1, $2);
        next if !$cmp || !$required_version;
        #die "can't parse condition '$cond' from dependency '$dep'" if !$cmp || !$required_version;
        $parsed_dep->{version} = $required_version;
        $parsed_dep->{cmp} = $cmp;
        push @res, $parsed_dep;
    }

    return \@res;
}

sub parse_options
{
    my %O = (
    );
    my %defaults = (
        ignore => {
        },
        never_dmove => {},
    );

    GetOptions(
        "help" => sub {
            system("podselect -section NAME -section DESCRIPTION -section OPTIONS -section EXAMPLES $0 | pod2text-utf8"); 
            exit 0;
        },
        'c|conf=s' => \$O{conf_file},
        't|control-file=s' => \$O{control_file},
        'r|repository=s' => \$O{repository},
        'f|from=s' => \$O{from},
        'p|package=s' => \$O{package},
    );

    my $conf = defined $O{conf_file} ? YAML::LoadFile($O{conf_file}) : {};

    for my $param (qw/ignore repository control_file from never_dmove/){
        next if defined $O{$param};
        $O{$param} = $conf->{$param} || $defaults{$param};
    }
    
    die "undefined repository, stop\n" unless defined $O{repository};

    return \%O;
}

