#!/usr/bin/perl
use strict;
use warnings;
use Cwd;
use Data::Dumper;
use Getopt::Long;

my $default_src_deb = '';
my $distribution = 'trusty';
my $urgency = 'low';
GetOptions(
  'source-deb=s' => \$default_src_deb,
  'distribution=s' => \$distribution,
  'urgency=s' => \$urgency,
);

my @deb_files = @ARGV;
die "Usage: $0 [--source-deb <deb_file>] [--distribution <trusty|...>] [--urgency <low|...>] <deb_files>\n" if not @deb_files;

my $changes_files = get_changes_files(@deb_files);
for my $k (keys %$changes_files) {
    open(my $fh, '>', $k) or die "Can't open $k: $!";
    print $fh $changes_files->{$k};
    system(qq(debsign -m\$DEBEMAIL $k));
}

sub get_dpkg_info {
    my ($deb_file) = @_;
    my $info = qx(dpkg -I $deb_file);
    my %data = map { (/^(?:##)? ?(\w+): (\S+.*)$/) ? ($1, $2) : () } split /\n/s, $info;

    ($data{file}, my $deb_name, my $deb_version, my $deb_arch) = ($deb_file =~ m|(([^_/]+)_([^_/]+)_([^_/]+).deb)$|);

    ($data{size}) = ($info =~ m/size (\d+) bytes/);
    ($data{date} = qx(date '+%a, %d %b %Y %H:%M:%S %z')) =~ s/\n$//s;
    ($data{sha1} = qx(sha1sum $deb_file)) =~ s/\s.*\n$//s;
    ($data{sha256} = qx(sha256sum $deb_file)) =~ s/\s.*\n$//s;
    ($data{md5} = qx(md5sum $deb_file)) =~ s/\s.*\n$//s;
    ($data{Description}) =~ s/\n.*//s;
    $data{Package} //= $deb_name;
    $data{Source} //= $deb_name;
    $data{Architecture} //= $deb_arch;
    $data{Section} //= "misc";
    $data{Priority} //= "optional";

    return \%data;
}

sub get_changes_files {
    my (%src_to_info, %changes_to_info);

    for my $pkg (@_) {
        my $data = get_dpkg_info($pkg);
        $src_to_info{$data->{Source}} = [] if not exists $src_to_info{$data->{Source}}; 
        push @{$src_to_info{$data->{Source}}}, $data;
    }

    while (my ($src_pkg, $info) = each(%src_to_info)) {
        my @pkg_info = sort { $a->{Package} cmp $b->{Package} } @$info;

        my (@multisha1, @multisha256, @multifiles, @multidesc);
        for my $data (@pkg_info) {
            push @multisha1, " $data->{sha1} $data->{size} $data->{file}";
            push @multisha256, " $data->{sha256} $data->{size} $data->{file}";
            push @multifiles, " $data->{md5} $data->{size} $data->{Section} $data->{Priority} $data->{file}";
            push @multidesc, " $data->{Package} - " . substr($data->{Description}, 0, 65);
        }

        my @src_pkg_infos = grep { $_->{Package} eq $src_pkg || $_->{file} eq $default_src_deb } @pkg_info;
        if (scalar(@src_pkg_infos) != 1) {
            warn sprintf("\nNo source debs for %s found. You can specify it here from files listed below: $0 --source-deb <debfile>:\n%s\n",
                $src_pkg, join("\n", map { $_->{file} } @pkg_info));
            next;
        }
        my $src_pkg_info = shift @src_pkg_infos;
        warn sprintf("\nFound debs with required source %s:\n%s\n", $src_pkg, join("\n", map { $_->{file} } @pkg_info));

        $src_pkg_info->{Package} = join " ", map { $_->{Package} } @pkg_info;
        $src_pkg_info->{sha1} = join "\n", @multisha1;
        $src_pkg_info->{sha256} = join "\n", @multisha256;
        $src_pkg_info->{file} = join "\n", @multifiles;
        $src_pkg_info->{Description} = join "\n", @multidesc;
        my $changes_file = ( $src_pkg_info->{Source}."_".$src_pkg_info->{Version}."_".$src_pkg_info->{Architecture}.".changes");

        $changes_to_info{$changes_file} = get_changes_file($src_pkg_info);
        ##print $changes_to_info{$changes_file};
    }
   
    return \%changes_to_info;
}

sub get_changes_file {
    my ($data) = @_;

    my $changesfile = <<EOF;
Format: 1.8
Date: $data->{date}
Source: $data->{Source}
Binary: $data->{Package}
Architecture: $data->{Architecture}
Version: $data->{Version}
Distribution: $distribution 
Maintainer: $data->{Maintainer}
Changed-By: $ENV{DEBFULLNAME} <$ENV{DEBEMAIL}>
Urgency: $urgency
Description:
$data->{Description}
Changes:
 $data->{Package} ($data->{Version}) unstable; urgency=low
 .
   * Auto generated changes file for $data->{Package}
Checksums-Sha1:
$data->{sha1}
Checksums-Sha256:
$data->{sha256} 
Files:
$data->{file}
EOF
    $changesfile =~ s/\s+$//mg;
    return $changesfile;
}
