#!/usr/bin/perl

use strict;
use warnings;

=head1 DESCRIPTION

Скрипт для инициализации новых пустых БД Директа. 
Типичное (и проверенное) использование -- включение новых машин с тестовыми БД


./direct-init-db.pl -d ppclog --schema-dir ../../db_schema/ppclog --schema-dir ../../db_schema/ppclog_bs -g /tmp/direct-dev-flagman-docs/direct-test-mysql-grants.yaml

./direct-init-db.pl -d ppcpricelog --schema-dir ../../db_schema/ppcpricelog -g /tmp/direct-dev-flagman-docs/direct-test-mysql-grants.yaml 

./direct-init-db.pl -d ppcfiles --schema-dir ../../db_schema/ppcfiles -g /tmp/direct-dev-flagman-docs/direct-test-mysql-grants.yam

./direct-init-db.pl -d ppcstat --schema-dir ../../db_schema/ppcstat --schema-dir ../../db_schema/ppcstat_data --schema-dir ../../db_schema/ppcstat_phrases -g /tmp/direct-dev-flagman-docs/direct-test-mysql-grants.yaml

просто создать database в запущенном mysql:
./direct-init-db.pl -d ppcstat --schema-dir ../../db_schema/ppcstat --schema-dir ../../db_schema/ppcstat_data --schema-dir ../../db_schema/ppcstat_phrases -g /tmp/direct-dev-flagman-docs/direct-test-mysql-grants.yaml -f create_databases


=cut 

use YAML;
use Getopt::Long;
use Sys::Hostname;
use File::Find;


my %STEP = (
    init_datadir => {  
        code => \&init_datadir,
    },
    start_mysql => {  
        code => \&start_mysql,
    },
    grants => {  
        code => \&grants,
    },
    create_databases => {  
        code => \&create_databases,
    },
    create_tables => {  
        code => \&create_tables,
    },
    fill_data => {  
        code => \&fill_data,
    },
    scripts => {  
        code => \&scripts,
    },
);

my @ORDER = qw/
    init_datadir
    start_mysql
    grants
    create_databases
    create_tables
    fill_data
    scripts
/;

run() unless caller();

sub run 
{
    my %O = %{parse_options()};

    for my $st ( @{$O{steps}} ){
        print "step $st ...\n";
        $STEP{$st}->{code}->(%O);
        print " ... $st done\n\n";
    }

    exit 0;
}


sub parse_options
{   
    my %O = (
    );

    GetOptions(
        "help" => sub {
            system("podselect -section NAME -section DESCRIPTION -section OPTIONS -section EXAMPLES $0 | pod2text-utf8 | LESSCHARSET=utf-8 less");
            exit 0;
        },
        "schema-dir=s@" => \$O{schema_dirs},
        "f|first-step=s" => \$O{first_step},
        "l|last-step=s" => \$O{last_step},
        "s|step=s@" => \$O{steps},
        "d|dbname=s" => \$O{dbname},
        #"c|conf=s" => \$O{conf_file},
        "g|grants=s" => \$O{grants_rules_file},
    ) || die "can't parse options, stop";

    print YAML::Dump(\%O);

    if ( $O{steps} && ($O{first_step} || $O{last_step})) {
        die "steps and first_step/last_step are incompatible" ;
    } elsif ( $O{steps} ) {
        # nothing
    } else {
        $O{first_step} ||= $ORDER[0];
        $O{last_step} ||= $ORDER[-1];
        die "steps" unless exists $STEP{$O{first_step}} && exists $STEP{$O{last_step}};
        $O{steps} = [ grep { $_ eq $O{first_step} .. $_ eq $O{last_step} } @ORDER ];
    } 

    die "empty steps" unless @{$O{steps}||[]} > 0;

    $O{conf} = YAML::LoadFile($O{conf_file}) if $O{conf_file};
    #$O{dbname} = $O{conf}->{dbname} if exists $O{conf}->{dbname};
    die "bad dbname $O{dbname}" unless $O{dbname} =~ /^[a-z0-9]+$/i;

    return \%O;
}


sub init_datadir
{
    my (%O) = @_;

    my $dir = "/opt/mysql.$O{dbname}";
    mkdir $dir;
    my $err = $!;
    die "can't create $dir: $err" unless -d $dir;
    `mysqld --initialize-insecure --user=mysql --datadir=$dir`;
    die "mysql_install_db failed" unless $? == 0;
    `chown -R mysql:mysql $dir`;
    die "chown failed" unless $? == 0;
    `rm $dir/ib_logfile*`;
    die "removing ib_logfile* failed" unless $? == 0;

    return;
}


sub start_mysql
{
    my (%O) = @_;

    `/etc/init.d/mysql.$O{dbname} start`;
    for my $i (1 .. 20){
        last if -S "/var/run/mysqld.$O{dbname}/mysqld.sock";
        print "sleep #$i...\n";
        sleep 5;
    }
    die "start_mysql failed" unless -S "/var/run/mysqld.$O{dbname}/mysqld.sock"; 

    return;
}


sub grants
{
    my (%O) = @_;

    my $hostname = hostname();

    my $grants = qx!/usr/local/bin/generate-mysql-grant-statements.pl -i $O{dbname} --db-host $hostname --conf $O{grants_rules_file}!;
    die "generate-mysql-grant-statements.pl failed" unless $? == 0;

    my $status = system(qq!cat <<'EOF' | mysql -u root -S "/var/run/mysqld.$O{dbname}/mysqld.sock"\n\n$grants\nEOF\n!) || 0;
    print "database $O{dbname} status $status\n";
    die "grants failed" unless $status == 0;

    return;
}


sub create_databases
{
    my (%O) = @_;

    my $sql = join "\n", map { s!.*/([^/]*)$!create database if not exists $1 character set utf8 collate utf8_general_ci;!r } @{$O{schema_dirs}};

    my $status = system(qq!cat <<'EOF' | mysql -u root -S "/var/run/mysqld.$O{dbname}/mysqld.sock"\n\n$sql\nEOF\n!) || 0;
    die "create fatabases failed" unless $status == 0;

    return;
}


sub create_tables
{
    my (%O) = @_;

    for my $schema_dir ( @{$O{schema_dirs}} ){
        my $db = $schema_dir =~ s!.*/([^/]*)$!$1!r;
        find( {
                wanted => sub {
                    my $schema_file = $_;
                    return unless $schema_file =~ /\.schema\.sql$/;
                    return if $schema_file =~ /(YYYY|MM|DD|NN)/;
                    print "createing table according to $schema_file\n";
                    my $status = system(qq!cat $schema_file |sed 's/CREATE TABLE/CREATE TABLE IF NOT EXISTS/' | mysql -u root -S "/var/run/mysqld.$O{dbname}/mysqld.sock" $db!) || 0;
                    die "failed" unless $status == 0;
                },
            }
            , $schema_dir
        );
    }


    return;
}


sub fill_data
{
    my (%O) = @_;
}


sub scripts
{
    my (%O) = @_;
}


