#!/usr/bin/perl

use strict;
use warnings;

=head1 DESCRIPTION

    direct-populate-sandbox.pl -S /tmp/mysql_sandbox5623.sock  dir1 dir2 ..

    direct-populate-sandbox.pl -S /tmp/mysql_sandbox5623.sock /var/www/beta.lena-san.9001/db_schema/*

=cut

use YAML;
use Getopt::Long;
use Sys::Hostname;
use File::Find;

run() unless caller();

sub run
{
    my $opt = parse_options();

    create_databases($opt);
    create_tables($opt);
    create_procedures($opt);
    fill_data($opt);

    exit 0;
}

sub parse_options
{   
    my %O = (
    );

    GetOptions(
        "help" => sub {
            system("podselect -section NAME -section DESCRIPTION -section OPTIONS -section EXAMPLES $0 | pod2text-utf8 | LESSCHARSET=utf-8 less");
            exit 0;
        },
        "S|socket=s" => \$O{mysql_socket},
        "host=s"     => \$O{mysql_host},
        "port=i"     => \$O{mysql_port},
        "u|user=s"   => \$O{mysql_user},
        "p|password=s" => \$O{mysql_password},
    ) || die "can't parse options, stop";

    #print YAML::Dump(\%O);

    $O{schema_dirs} = [@ARGV];
    $ENV{MYSQL_PWD} = $O{mysql_password} if $O{mysql_password};
    $O{mysql_user} ||= 'root';
    if( !$O{mysql_port} && !$O{mysql_user} ){
        $O{mysql_socket} ||= "/var/run/mysqld/mysqld.sock";
    }

    $O{mysql_cmd} = "mysql -u $O{mysql_user}";
    if ( $O{mysql_socket} ){
        $O{mysql_cmd} .= " -S $O{mysql_socket}";
    }
    if ( $O{mysql_port} ){
        $O{mysql_cmd} .= " --port=$O{mysql_port}";
    }
    if ( $O{mysql_host} ){
        $O{mysql_cmd} .= " --host=$O{mysql_host}";
    }

    print "mysql_cmd: $O{mysql_cmd}\n";

    return \%O;
}


sub create_databases
{
    my ($opt) = @_;
    my $sql = join "\n", map { s!.*/([^/]*)$!create database if not exists $1 character set utf8 collate utf8_general_ci;!r } @{$opt->{schema_dirs}};

    my $status = system(qq!cat <<'EOF' | $opt->{mysql_cmd}\n\n$sql\nEOF\n!) || 0;
    die "create fatabases failed" unless $status == 0;

    return;
}


sub create_tables
{
    my ($opt) = @_;
    system(qq!$opt->{mysql_cmd} -e 'set global foreign_key_checks = 0'!) == 0 or die;
    for my $schema_dir ( sort @{$opt->{schema_dirs}} ){
        my $db = $schema_dir =~ s!.*/([^/]*)$!$1!r;
        print "creating database $db\n";
        find( {
                wanted => sub {
                    my $schema_file = $_;
                    return unless $schema_file =~ /\.schema\.sql$/;
                    return if $schema_file =~ /(YYYY|MM|DD|NN)/;
                    print "creating table according to $schema_file\n";
                    my $status = system(qq!cat $schema_file |sed 's/CREATE TABLE/CREATE TABLE IF NOT EXISTS/' | $opt->{mysql_cmd} $db!) || 0;
                    die "failed" unless $status == 0;
                },
            }
            , $schema_dir
        );
    }
    system(qq!$opt->{mysql_cmd} -e 'set global foreign_key_checks = 1'!) == 0 or die;

    return;
}

sub create_procedures {
    my ($opt) = @_;

    for my $schema_dir ( sort @{$opt->{schema_dirs}} ){
        my $db = $schema_dir =~ s!.*/([^/]*)$!$1!r;
        print "creating database $db\n";
        find( {
                wanted => sub {
                    my $schema_file = $_;
                    return unless $schema_file =~ /\.procedure\.sql$/;
                    print "creating procedure according to $schema_file\n";
                    my $status = system(qq!cat $schema_file | $opt->{mysql_cmd} --delimiter=// $db!) || 0;
                    die "failed" unless $status == 0;
                },
            }
            , $schema_dir
        );
    }

    return;
}

sub fill_data {
    my ($opt) = @_;

    system(qq!$opt->{mysql_cmd} -e 'set global foreign_key_checks = 0'!) == 0 or die;
    for my $schema_dir ( sort @{$opt->{schema_dirs}} ){
        my $db = $schema_dir =~ s!.*/([^/]*)$!$1!r;
        print "creating database $db\n";
        find( {
                wanted => sub {
                    my $schema_file = $_;
                    return unless $schema_file =~ /\.(data|init-data)\.sql$/;
                    print "filling data according to $schema_file\n";
                    my $status = system(qq!cat $schema_file | $opt->{mysql_cmd} $db!) || 0;
                    die "failed" unless $status == 0;
                },
            }
            , $schema_dir
        );
    }
    system(qq!$opt->{mysql_cmd} -e 'set global foreign_key_checks = 1'!) == 0 or die;

    return;
}
