# -* encoding: utf-8 -*-
from tools import get_data_from_juggler_api, nice_text
import datetime, re
import sys

JUGGLER_HOSTS_FOR_COMPLETE_EVENTS = (
    'checks.direct.yandex.ru',
    'checks_day.direct.yandex.ru',
    'direct.apache',
)

JUGGLER_HOSTS_FOR_RAW_EVENTS = (
    'ppcback01e.yandex.ru',
    'ppcback01f.yandex.ru',
    'ppcback01i.yandex.ru',
)

# "важность" статуса, используется для сортировки событий
EVENTS_PRIORITY = {
    'CRIT': 9,
    'WARN': 5,
    'INFO': 3,
    'OK': 1,
}

EVENTS_SEVERITY = {
    'INFO': -1,
    'OK': 0,
    'WARN': 1,
    'CRIT': 2,
}

# Типы объектов, которые могут иметь downtime в juggler
DOWNTIME_OBJECT_TYPES = ('HOST', 'CGROUP', 'CMS')


class RawEvent:
    def __init__(self, host, name, raw_event_data):
        data = raw_event_data.get('')
        self.host = host
        self.name = name
        self.description = data['description'][0]['description']
        self.status = data['status'][0]
        self.priority = EVENTS_PRIORITY.get(self.status)
        self.time = int(data['status'][1])
        self.datetime = datetime.datetime.fromtimestamp(self.time).strftime("%Y-%m-%d %H:%M:%S")

    def __unicode__(self):
        return self.name


class RawEvents:
    def __init__(self, with_obsolete=False, **kwargs):
        if with_obsolete:
            self.without_obsolete_flag = False;
        else:
            self.without_obsolete_flag = True;

        if 'from_list' in kwargs:
            self.events = kwargs['from_list']
        else: 
            self.events = list()
            self.load_from_api(**kwargs)

    def load_from_api(self, **kwargs):
        raw_data = get_data_from_juggler_api(type='raw_events',
                                             without_obsolete=self.without_obsolete_flag,
                                             **kwargs
                                             )
        for host, data in raw_data.iteritems():
            if data:
                for name, raw_event_data in data.items():
                    self.events.append(RawEvent(host, name, raw_event_data))


    def sort(self):
        # Сортировка по-умолчанию - два прохода, сначала по имени
        self.events = sorted(self.events, key=lambda x: x.name)
        # потом обратная по приоритету и времени
        self.events = sorted(self.events, key=lambda x: (x.priority, x.time), reverse=True)

    def get(self):
        return self.events


class CompleteEvent:
    def __init__(self, host, name, data):
        self.host = host
        self.name = name
        self.timestamp = int(data['check_time'])
        self.datetime = datetime.datetime.fromtimestamp(self.timestamp)
        self.date = self.datetime.strftime("%Y-%m-%d")
        self.time = self.datetime.strftime("%H:%M:%S")
        self.status = data['status'][0]
        self.status_begin_at = int(data['status'][1])
        self.priority = EVENTS_PRIORITY.get(self.status)
        self.description = nice_text(data['description'][0]['description'], field='description')
        summary = data['description'][0].get('summary', '—')
        self.summary = nice_text(summary, field='summary')
        self.failed_count = summary.replace(' from ', '/').replace(' failed', '')
        if ('aggregator' in data['description'][0]
            and data['description'][0]['aggregator']['name'] == 'logic_or'
            and 'children' in data['description'][0]
            and data['description'][0]['children']):
            self.details = []
            for childrens_host, childrens_data in data['description'][0]['children'].items():
                for children_name, one_children_data in childrens_data.items():
                    self.details.append({
                        'host': childrens_host,
                        'name': children_name,
                        'status': one_children_data['']['status'],
                        'summary': one_children_data[''].get('summary', '-')
                    })

    def __unicode__(self):
        return self.name

    def as_grafana_array(self):
        return [
            self.failed_count,
            self.name,
            EVENTS_SEVERITY.get(self.status, 9),
        ]


class CompleteEvents:
    def __init__(self, hosts, **kwargs):
        if 'from_list' in kwargs:
            self.events = kwargs['from_list']
        else: 
            self.events = list()
            self.load_from_api(hosts, **kwargs)

    def load_from_api(self, hosts, **kwargs):
        raw_data = get_data_from_juggler_api(type='complete_events',
                                             hosts=hosts,
                                             with_no_methods=True,
                                             old_juggler_format=False,
                                             **kwargs
                                             )

        for host, data in raw_data.iteritems():
            if data:
                for name, complete_event_data in data.items():
                    self.events.append(CompleteEvent(host, name, complete_event_data))

    def sort(self):
        # Сортировка по-умолчанию - два прохода, сначала по имени
        self.events = sorted(self.events, key=lambda x: x.name)
        # потом обратная по приоритету и времени начала текущего результата
        self.events = sorted(self.events, key=lambda x: (x.priority, x.status_begin_at), reverse=True)

    def get(self):
        return self.events

    def filter_list_by_params(self, host='', service=''):
        self.events = [new_event for new_event in self.events
                       if re.match(host, new_event.host, re.I) and
                       re.match(service, new_event.name, re.I)]


class Hosts:
    def __init__(self):
        self.hosts = dict()
        self.addDefaults()

    def addDefaults(self):
        for host in JUGGLER_HOSTS_FOR_RAW_EVENTS:
            self.hosts[host] = True

    def getAllHosts(self):
        return self.hosts

    def getCheckedHosts(self):
        result = []
        for host, checked in self.hosts.items():
            if checked:
                result.append(host)
        return result

    def applySessionData(self, data):
        if data and type(data).__name__ == 'dict':
            for host, checked in data.items():
                if checked:
                    self.hosts[host] = True
                else:
                    self.hosts[host] = False

    def applyUserData(self, data):
        for host in self.hosts.keys():
            self.hosts[host] = False
        if data:
            for host in set(data):
                if host:
                    self.hosts[host] = True
        if not True in self.hosts.values():
            self.addDefaults()


class Downtime:
    def __init__(self, object_type, object_name, service_name, raw_service_data):
        self.object_type = object_type
        self.object_name = object_name
        self.service_name = service_name
        self.start_time = int(raw_service_data['start_time'])
        self.start_datetime = datetime.datetime.fromtimestamp(self.start_time).strftime("%Y-%m-%d %H:%M:%S")
        if raw_service_data['end_time']:
            self.end_time = int(raw_service_data['end_time'])
            self.end_datetime = datetime.datetime.fromtimestamp(self.end_time).strftime("%Y-%m-%d %H:%M:%S")
        else:
            self.end_time = None
            self.end_datetime = '∞'
        self.local = raw_service_data['local']
        self.description = raw_service_data['description']

    def __unicode__(self):
        return '<{0}:{1}> {2}'.format(self.object_type, self.object_name, self.service_name)


class Downtimes:
    def __init__(self, filter_object=None, filter_service=None, expand_groups=False):
        self.downtimes = list()
        if expand_groups:
            self.expand_groups = True
        else:
            self.expand_groups = False

        self.object = filter_object
        self.service = filter_service

        self.load_from_api()

    def load_from_api(self):
        raw_data = get_data_from_juggler_api(type='downtimes',
                                             expand_groups=self.expand_groups,
                                             object=self.object,
                                             service=self.service,
                                             )

        for object_type in DOWNTIME_OBJECT_TYPES:
            if object_type in raw_data:
                for object_name, object_data in raw_data[object_type].items():
                    for service_name, raw_service_data in object_data.items():
                        self.downtimes.append(Downtime(object_type,
                                                       object_name,
                                                       service_name,
                                                       raw_service_data
                                                       ))
            else:
                pass

    def sort(self):
        # Сортировка в несколько проходов, сначала по времени начала (старые - внизу)
        self.downtimes = sorted(self.downtimes, key=lambda x: (x.start_time), reverse=True)
        # self.downtimes = sorted(self.downtimes, key=lambda x: x.service_name)

    def get(self):
        return self.downtimes
