#!/usr/bin/perl

=head1 SYNOPSIS

    arcadia-hotfix.pl intapi 1234567 1234444   # замержить r1234444 в релизный бранч direct/intapi/1234567
    arcadia-hotfix.pl intapi 1234567 1234444,1234555    # замержить r1234444, r1234555
    arcadia-hotfix.pl intapi 1234567 1234444 1234555 # то же самое, что и через запятую

=head1 DESCRIPTION

    Команда
    
    arcadia-hotfix.pl <app_name> <base_rev> <rev1>[,<rev2>,...]

    1. создаст, если нужно, релизную ветку direct/<app_name>/<base_rev> и скопирует туда поддиректорию arcadia из транка
    2. зачекаутит рабочую копию релизной ветки и замержит rev1,rev2,... в релизную ветку (точнее, в поддиректорию arcadia)
    3. закоммитит изменения
    4. запустит сборку в Sandbox

    Откат хотфиксов (svn merge -c -NNNNN ...) сейчас не поддерживается.

=cut

use strict;
use warnings;

use utf8;

use open ':std', ':encoding(UTF-8)';

use Path::Tiny;
use XML::LibXML;
use YAML;
use Yandex::Shell;
use Yandex::Svn;

our $ARCADIA_URL = 'svn+ssh://arcadia.yandex.ru/arc';
$ENV{'SVN_SSH'} = "ssh -S $ENV{'SSH_MASTER_CONN_ARCADIA'}" if $ENV{'SSH_MASTER_CONN_ARCADIA'};

run() unless caller();

sub run {
    if (grep { /^(-h|--help)$/ } @ARGV) {
        system("podselect -section SYNOPSIS -section DESCRIPTION $0 | pod2text-utf8 >&2");
        exit 0;
    }
    $Yandex::Shell::PRINT_COMMANDS = 1;
    $ENV{TMPDIR} = '/tmp/temp-ttl/ttl_1d' if (-d '/tmp/temp-ttl/ttl_1d');
    my ($app_full_name, $base_rev, @to_hotfix) = @ARGV;
    die "не указано приложение\n" unless $app_full_name;
    die "не указана базовая ревизия\n" unless $base_rev;
    die "не указаны ревизии, которые нужно захотфиксить\n" unless @to_hotfix;
    my @invalid_hf = grep { !/^[0-9]+(,[0-9]+)*$/ } @to_hotfix;
    die "некорректные параметры: " . join("\n", @invalid_hf, '') if @invalid_hf;
    @to_hotfix = map { split /,/, $_ } @to_hotfix;
    my $release_branch = "direct/release/$app_full_name/$base_rev";
    my $release_branch_path = "$ARCADIA_URL/branches/$release_branch";
    system 'svn', 'ls', "$release_branch_path/arcadia", '--depth', 'empty';
    if ($? != 0) {
        yash_system 'svn', 'mkdir', $release_branch_path, '--parents', '--message' => "RELEASE: Created $release_branch";
        yash_system 'svn', 'copy', "$ARCADIA_URL/trunk/arcadia\@$base_rev", "$release_branch_path/arcadia", '--message' => "RELEASE: Copied trunk\@$base_rev to $release_branch";
    }
    my $tempdir = Path::Tiny->tempdir('arcadia-hotfix-XXXXXXXX', CLEANUP => 0);
    warn "Checking out to $tempdir\n";
    yash_system "svn", "co", "$release_branch_path/arcadia", "$tempdir/arcadia", "--depth", "empty";
    my ($rev_to_checkout, ) = `svn info $tempdir/arcadia` =~ /^Revision:\s*(.*?)$/sm;
    yash_system "svn", "up", "-r", "$rev_to_checkout", "--set-depth", "infinity", "$tempdir/arcadia/direct";
    yash_system "svn", "up", "-r", "$rev_to_checkout", "--set-depth", "infinity", "$tempdir/arcadia/grut";
    my $commit_msg = "RELEASE: Merged " . join(',', @to_hotfix) . "\n\n";
    $commit_msg .= join "\n\n", map { my $r = svn_log_entry("$ARCADIA_URL/trunk/arcadia", $_); sprintf "r%d: %s, %s\n%s", @$r{qw/revision author date msg/} } @to_hotfix;
    # REVIEW в коммит-месседжах обрабатывается нежелательным для хотфиксов образом (коммиты с мержами не проходят). Маскируем.
    $commit_msg =~ s/REVIEW: ([0-9]+)/<rvw $1>/g;
    for my $c (@to_hotfix) {
        warn "Merging r$c\n";
        yash_system 'svn', 'merge', '--change' => $c, "$ARCADIA_URL/trunk/arcadia", "$tempdir/arcadia";
    }

    # коммитим с arcanum:check-skip=yes, чтобы избежать долгой проверки: https://wiki.yandex-team.ru/arcadia/svn/#svojjstvakommita
    yash_system 'svn', 'commit', "$tempdir/arcadia", '--message' => $commit_msg, '--with-revprop', 'arcanum:check-skip=yes';
    my $head_rev = svn_info("$tempdir/arcadia")->{revision};
    print "Коммит в Аркануме: https://a.yandex-team.ru/commit/$head_rev\n";
    my $version = "1.$base_rev.$head_rev-1";
    my $package_path = YAML::LoadFile("/etc/yandex-direct/direct-apps.conf.yaml")->{apps}->{$app_full_name}->{"package-json"};
    print "ready branch\@rev: $release_branch\@$head_rev, version-to-be: $version\n";
    print "to create sandbox task: ".join(" ", 'sandbox-ya-package.pl',
        '--branch' => "$release_branch",
        '--package' => $package_path,
        '--revision' => $head_rev,
        '--version' => $version,
        '--no-wait',
    );
#   
#    print "Команда для обновления релизного тикета:\n:java-release.pl --app $app_name --version $version --ticket \$DIRECT_JAVA_RELEASE_TICKET\n\n";
}

sub svn_log_entry {
    my ($path, $revision) = @_;
    my $xml = yash_qx("svn log --xml --change $revision $path");
    my $doc = XML::LibXML->load_xml(string => $xml);
    my $entry = $doc->findnodes('/log/logentry')->shift;
    die "Коммит $revision не найден внутри $path. Проверьте, что коммит находится в транке (а не в ветке, к примеру)." unless defined $entry;
    my $r = {
        revision => $entry->getAttribute('revision'),
        (map { $_ => $entry->findvalue($_) } qw/author date msg/),
    };
    $r->{date} =~ s/^([0-9]{4}-[0-9]{2}-[0-9]{2})T([0-9]{2}:[0-9]{2}:[0-9]{2}).*/$1 $2/;
    return $r;
}

1;
__END__
