#!/usr/bin/perl -w

=head1 NAME

    build-direct.pl

=head1 DESCRIPTION

    Собирает релиз НЕИНТЕРАКТИВНО

    Чекаутит чистую рабочую копию (например, транка) в /tmp/temp-ttl/ttl_7d/build-direct_XXXXXXX/
    Собирает пакет
    Подписывает
    Загружает на dist, если надо

    Параметры:
        --svn-url, url того, что собираем (например, svn+ssh://svn.yandex.ru/direct/trunk)
        --rev, номер ревизии
        --passphrase-file, файл с паролем для gpg-ключа для подписи
        --dupload, если выставлен, то загружает пакет на дист

    Пример запуска:
    build-direct.pl --svn-url svn+ssh://svn.yandex.ru/direct/trunk --rev 159671 --dupload --passphrase-file ~/pass

=head1 COMMENTS

=cut

use strict;
use warnings;

use File::Temp qw/ tempdir /;
use Getopt::Long;
use Yandex::Shell qw/ yash_qx yash_system/;

use open ':std' => ':utf8';
use utf8;

$ENV{PATH} = "/usr/local/sbin:/usr/local/bin:/usr/sbin:/usr/bin:/sbin:/bin:/usr/games:/usr/local/games";
$ENV{'SVN_SSH'} = "ssh -S $ENV{'SSH_MASTER_CONN_SVN'}" if $ENV{'SSH_MASTER_CONN_SVN'};

run() unless caller();


sub run
{
    my %OPT = parse_options();

    # чекаутим рабочую копию во временную папку
    my $temp_dir = tempdir('build-direct_XXXXXXX', CLEANUP => 0, DIR => '/tmp/temp-ttl/ttl_7d');
    yash_system('svn', 'co','-q', $OPT{svn_url}, '-r', $OPT{rev}, $temp_dir);

    # собираем пакет
    yash_system("cd $temp_dir/packages && make SKIP_SIGN=yes");

    # подписываем пакет
    if ( $OPT{use_gpg_agent} ){
        yash_system("cd $temp_dir/packages/yandex-direct && debsign -p'gpg --batch' --debs-dir ../");
    } else {
        yash_system("cd $temp_dir/packages/yandex-direct && debsign -p'gpg --batch --passphrase-file $OPT{pass_file} --no-use-agent' --debs-dir ../");
    }

    # загружаем на дист, если надо
    if ($OPT{dupload}) {
        my $dupload_cmd = "dupload -t direct-trusty --nomail";
        if ($ENV{'SSH_MASTER_CONN_DIST'}) {
            create_dupload_conf_master_connection("$temp_dir/packages");
            $dupload_cmd .= " -c";
        }
        if ($ENV{DIRECT_RELEASE_DUPLOAD_WITH_LOGIN}){
            create_dupload_conf_login("$temp_dir/packages");
            $dupload_cmd .= " -c";
        }
           
        yash_system("cd $temp_dir/packages && $dupload_cmd");
    }
}

sub create_dupload_conf_master_connection
{
    open(my $fh, '>', "$_[0]/dupload.conf");
    print $fh q(package config;
# temporary config, generated by build-direct.pl
$cfg{'direct-trusty'}{options} = "-o ControlPath=$ENV{'SSH_MASTER_CONN_DIST'}";
1;
);
    close $fh;
}


sub create_dupload_conf_login
{
    open(my $fh, '>', "$_[0]/dupload.conf");
    print $fh q(package config;
# temporary config, generated by build-direct.pl
$cfg{'direct-trusty'}{login} = "$ENV{DIRECT_RELEASE_DUPLOAD_WITH_LOGIN}";
1;
);
    close $fh;
}


sub parse_options
{
    my %O = (
        use_gpg_agent => 1,
    );

    GetOptions(
        'svn-url=s' => \$O{svn_url},
        'dupload' => \$O{dupload},
        'rev=s' => \$O{rev},
        'help' => \&usage,
        'passphrase-file=s' => \$O{pass_file},
        'gpg-agent!' => \$O{use_gpg_agent},
    ) or die "can't parse options";

    $O{rev} = "HEAD" unless $O{rev};
    die "expecting --svn-url" unless $O{svn_url};

    unless ($O{use_gpg_agent} || $O{pass_file} && -f $O{pass_file} ){
        die "expecting existing file in --passphrase-file (or use --gpg-agent)";
    }

    return %O;
}


sub usage
{
    system("podselect -section DESCRIPTION -section SYNOPSIS $0 | pod2text-utf8");
    exit(0);
}
