#!/usr/bin/python
# -*- coding: utf8 -*-

import re
import os
import sys
import shlex
import subprocess
import xml.etree.ElementTree as ET
import datetime
import time
import argparse
import yaml

"""
Фильтрует вывод svn log и удаляет коммиты в которых не затронуто заданное приложение или библиотеки.
"""

BEGIN_DATETIME = datetime.datetime(1970,1,1)
SECONDS_PER_DAY = 24 * 60 * 60
APPS = ('api5', 'intapi', 'jobs', 'logviewer', 'web')
SVN_ROOT = '/trunk/arcadia'
SVN_ROOT_DIRECT = '/trunk/arcadia/direct'

apps_conf_var = None

def get_apps_conf():
    global apps_conf_var
    if apps_conf_var == None:
        apps_conf_file = os.environ['APPS_CONF'] if 'APPS_CONF' in os.environ else '/etc/yandex-direct/direct-apps.conf.yaml'
        with open(apps_conf_file) as fh:
            apps_conf_var = yaml.load(fh)
    return apps_conf_var

def process_date(datime):
    dat, tim = datime.split('T')
    tim = tim.split('.')[0]
    y, m, d = map(int, dat.split('-'))
    hh, mm, ss = map(int, tim.split(':'))
    in_time = datetime.datetime(y, m, d, hh, mm, ss)
    
    delta = in_time - BEGIN_DATETIME
    total_seconds = SECONDS_PER_DAY * delta.days + delta.seconds
    time_tuple = time.localtime(total_seconds)[:6]
    return str(datetime.datetime(*time_tuple))


def create_useless_apps_list(name):
    return [app for app in APPS if app != name]


def get_log_text(begin, end):
    args = shlex.split(r'svn log -v -r ' + str(begin) + ':' + str(end) + ' --xml svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/direct')
    return subprocess.check_output(args)


def parse_log(txt):
    return ET.fromstring(txt)


def got_usefull_part(rev, foreign_dirs):
    """
    получает коммит и "бесполезные" пути (принадлежащие целиком другим приложениям)
    возвращает признак "Коммит полезен"
    """
    paths = rev.find('paths')
    for path_el in paths:
        path = path_el.text
        if not path.startswith(SVN_ROOT_DIRECT):
            continue
        is_foreign_dir = False
        for app_dir in foreign_dirs:
            regex = r"^" + SVN_ROOT + '/' + app_dir + r"($|/)"
            if re.match(regex, path):
                is_foreign_dir = True
                break
        if not is_foreign_dir:
            return True
    return False


def filter_commits(log, app_name):
    """
    получает нефильтрованный svn log и имя приложения
    возвращает лог, из которого удалены "чужеродные" коммиты
    """

    apps_conf = get_apps_conf()
    if not app_name in apps_conf['apps']:
        raise Exception("unknown app %s" % app_name)
    # список "чужих" директорий
    foreign_dirs = ['direct/packages', 'direct/monitorings', 'direct/infra']
    for app, conf in apps_conf['apps'].iteritems():
        if app == app_name:
            continue
        if 'primary-svn-dir' not in conf:
            continue
        foreign_dirs.append(conf['primary-svn-dir'])
    
    filtered = []
    for rev in log:
        if got_usefull_part(rev, foreign_dirs):
            filtered.append(rev)
    return filtered


def xml_log_to_text(root):
    text = ''
    if len(root):
        text += '-' * 72
        text += '\n'
    for rev in root:
        s = make_revision_output_text(rev)
        text += s
        text +='\n'
        text += '-' * 72
        text +='\n'
    return text


def make_revision_output_text(rev):
    s = 'r' + rev.attrib['revision'] + ' | ' + rev.find('author').text \
        + ' | ' + process_date(rev.find('date').text)
    msg = rev.find('msg')
    if msg is not None and msg.text is not None:
            msg = msg.text.encode('UTF8')
            cnt = msg.count('\n') + 1
    else:
        msg = ''
        cnt = 1
    s += ' | ' + str(cnt) + ' line' + ('s' if cnt > 1 else '')
    s += '\n\n'
    s += msg
    return s


def main():
    apps_conf = get_apps_conf()
    APPS = [a for a in apps_conf['apps'].keys() if a != 'direct']

    parser = argparse.ArgumentParser(description = 'Takes the logs related to the application or some libs in direct in the desired range of revisions')
    parser.add_argument('app_name', type = str, choices = APPS, help = 'app name')
    parser.add_argument('initial_revision', type = int, help = 'first revision in range')
    parser.add_argument('final_revision', type = int, help = 'last revision in range')
    args = parser.parse_args()

    if args.initial_revision < 0 or args.final_revision < 0:
        parser.print_usage(sys.stderr)
        sys.stderr.write("direct-arc-log: error: revision numbers can't be negative\n")
        return 1

    txt = get_log_text(args.initial_revision, args.final_revision)
    log = parse_log(txt)
    filtered_log = filter_commits(log, args.app_name)

    print xml_log_to_text(filtered_log)


if __name__ == '__main__':
    exit(main())
