#!/usr/bin/perl

=head1 NAME

    direct-release-tracker-perl.pl -- собрать релиз perl Директа

=head1 DESCRIPTION

    Скрипт для записи всего, что надо в релизные тикеты для большого Директа
    Собран copy-pase-ом из кусочков релизного шелла

=head1 EXAMPLES

    Создать релизный тикет:
    direct-release-tracker-perl.pl --version 1.161372-1
    direct-release-tracker-perl.pl --version 1.161372-1 -do
    direct-release-tracker-perl.pl --version 1.161372-1 --name 'Мега-фича'
    direct-release-tracker-perl.pl --version 1.161372-1 --name 'Мега-фича' -do

    Обновить релизный тикет (непротестированный режим!!!):
    direct-release-tracker-perl.pl --version 1.161372-1 --ticket DIRECT-NNNNN
    direct-release-tracker-perl.pl --version 1.161372-1 --ticket DIRECT-NNNNN -do

    Переименовать релизный тикет (непротестированный режим!!!):
    direct-release-tracker-perl.pl --ticket DIRECT-NNNNN --name 'Две мега-фичи' --rename
    direct-release-tracker-perl.pl --ticket DIRECT-NNNNN --name 'Две мега-фичи' --rename -do

=head1 TODO

    - проверить slide
    - проверить hotfix
    - проверить переименование
    - проверить downgrade: просто так не делается, с --downgrade делает разумное
    - доделать DirectRelease::Migrations и использовать

=cut

use strict;
use warnings;
use feature qw/state/;

use utf8;

use open ':std' => ':utf8';

use Getopt::Long;
use Startrek::Client::Easy;
use YAML;
use Yandex::Shell;
use XML::LibXML;
use Text::Diff;
use Text::Diff::Parser;
use POSIX;
use DirectRelease::Description qw(description2logs logs2description);
use ProjectSpecific qw/
    version2rev 
    svn_url 
    parse_release_summary 
    make_release_summary 
    change_release_summary
    tabula_testupdate_url
    /;
# TODO Доделать DirectRelease::Migrations, чтобы оно умело perl-Директ и использовать
#eval "use DirectRelease::Migrations qw(migration_details what_to_do_migrations)";
eval "use MigratorB::Parse";

=head1 INTERNAL

=cut

my %ST_NAME = (
    10009 => "New",
    10002 => "Testing",
    10040 => "RM Acceptance",
    10034 => "Need Acceptance",
    10035 => "Ready to deploy",
        6 => "Closed",
);

$ENV{'SVN_SSH'} = "ssh -S $ENV{'SSH_MASTER_CONN_SVN'}" if $ENV{'SSH_MASTER_CONN_SVN'};

my $DATE = strftime '%Y-%m-%d', localtime;
my $DEFAULT_RELEASE_NAME = 'Сборка от '.$DATE;
our $VERSION_RE = '1\.([0-9]{6,})(?:\.([0-9]{6,}))?-[0-9]+';     # ($base_rev, $head_rev) = ($1, $2);

run() unless caller();

sub parse_options {
    my %O = (
    );
    GetOptions(
        'v|version=s' => \$O{version},
        't|ticket=s' => \$O{ticket},
        'n|name=s' => \$O{name},
        'rename' => \$O{rename},
        'do' => \$O{do},
        'h|help' => sub { system("podselect -section SYNOPSIS -section DESCRIPTION $0 | pod2text-utf8 >&2"); exit 0 },
    ) || die "can't parse options, stop\n";

    defined && utf8::decode($_) for values(%O);
    die "no ticket or name given\n" if $O{rename} && (!$O{ticket} || !$O{name});
    die "nothing but ticket and name must be given to rename\n" if $O{rename} && ($O{revision} || $O{version} || $O{app_name});
    die "ticket key must be in format DIRECT-NNNNN" if $O{ticket} && $O{ticket} !~ /^DIRECT-[0-9]{5,}$/;
    if (! $O{version} && ! $O{rename}) {
        die "expecting one of params: --revision, --version, --rename, stop";
    }
    $O{ticket} ||= 'create';
    return \%O;
}

sub run
{
    my $opt = parse_options();
    my $argv = $opt->{argv};

    $ProjectSpecific::PROJECT = 'Direct';

    my %tr_opt = (
        release_issue_key => $opt->{ticket},
        version => $opt->{version},
        print_only => !$opt->{do},
        name => $opt->{name} || '',
        force => $opt->{force},
        downgrade => $opt->{downgrade},
    );
    $tr_opt{rename} = $opt->{rename} if $opt->{rename};
    my $tr_res = tracker_release( %tr_opt );
    print YAML::Dump($tr_res);
}


sub get_tracker_object {
    state $tracker;
    $tracker //= Startrek::Client::Easy->new();
    return $tracker;
}

=head2 get_prev_version

    найти предыдущую версию, записанную в трекере
    Если указан релиз -- из него, если нового релиза еще нет -- из предыдущего

=cut
sub get_prev_version
{
    my %O = @_;

    my $st = get_tracker_object();
    # В $r попадет либо текущий обрабатываемый релиз, либо предыдущий в этом приложении. Инвариант: предыдущая релизная версия записана в тикете $r
    my $r = $st->get(
        $O{current_release} eq 'create' ? (query => qq/Queue: DIRECT Type: Release Components: "Releases: Direct" "Sort by": key desc/) : (key => $O{current_release}),
        array => 1,
    )->[0];
    die "could not find release ticket $O{current_release} in tracker\n" if $O{current_release} && ! $r;
    my ($old_version, $old_base_rev, $old_head_rev) = $r->{summary} =~ /\b($VERSION_RE)\b/;
    return $old_version;
}


=head2 tracker_release

=cut

sub tracker_release
{
    my %OPT = @_;
    my $res = {};

    my $tracker = get_tracker_object();

    $OPT{name} ||= $DEFAULT_RELEASE_NAME;

    $OPT{prev_version} = get_prev_version(current_release => $OPT{release_issue_key});

    my $release = $OPT{release_issue_key} eq 'create' ? {} : $tracker->get( key => $OPT{release_issue_key} );
    my $to_do_struct = what_to_do_in_tracker($tracker, $release, %OPT);

    if( $OPT{print_only} ){
        print_updates($release, $to_do_struct);
        return;
    }

    my $release_queue = ProjectSpecific::get_project_data('default_tracker_queue');
    my $key;
    $key = $release->{key} if $OPT{release_issue_key} ne 'create';
    my @migrations_created;
    for my $type (qw/migrations release/){
        for my $to_do (@{$to_do_struct->{$type} || []}){
            if ( $type eq 'release' && scalar @migrations_created > 0 ){
                my $migrations_list = join "\n", @migrations_created;
                $to_do->{description} =~ s/__MIGRATIONS_PLACEHOLDER__/$migrations_list/;
            }
            my $tracker_result = $tracker->do(%$to_do);
            if ( $type eq 'migrations' && $to_do->{create} ) {
                push @migrations_created, $tracker_result;
            } elsif ( $type eq 'release' && $OPT{release_issue_key} eq 'create' && $tracker_result =~ /^$release_queue-/){
                die "2 issues in default queue: $key, $tracker_result; stop" if $key;
                $key = $tracker_result;
            }
        }
    }

    if ($OPT{release_issue_key} eq 'create') {
        print "new ticket: $key\n";
        $res->{created} = $key;
    }

    return $res;
}

=head2 what_to_do_in_tracker

=cut
sub what_to_do_in_tracker
{
    my ($tracker, $old_values, %O) = @_;
    my $to_do_struct = {release => [], migrations => [],};

    my $release_svn_url = svn_url('trunk');

    if ( $O{release_issue_key} eq 'create' ){
        my $rev_2 = version2rev($O{version}, name => 'new', rev => 'last');
        my $rev_1 = version2rev($O{prev_version}, name => 'prev', rev => 'base');

        my $desc_parts = {};
        $desc_parts->{changelog} = log_text(start_rev => $rev_1 + 1, end_rev => $rev_2, svn_url => $release_svn_url );
        # hash_merge
        my $migration_details = migration_details(start_rev => $rev_1, end_rev => $rev_2, deploy_svn_url => "$release_svn_url/deploy" );
        # to_do на миграции
        # 1. Создаем
        for my $migr_file ( sort keys %{$migration_details->{migration_files}||{}} ){
            my $migr_params = {
                create => 1,
                type => 'task',
                description => $migration_details->{migration_files}->{$migr_file},
                summary => "direct: Миграция $migr_file",
                queue => "DIRECTMIGR",
            };
            push @{$to_do_struct->{migrations}}, $migr_params;
        }

        # 2. Обновляем
        for my $migr_ticket ( sort keys %{$migration_details->{migration_tickets_to_update}||{}} ){
            #print "migr_ticket $migr_ticket\n$migration_details->{migration_tickets_to_update}->{$migr_ticket}\n\n";
            my $migr_params = {
                key => $migr_ticket,
                description => "$migration_details->{migration_tickets_to_update}->{$migr_ticket}",
            };
            push @{$to_do_struct->{migrations}}, $migr_params;
        }

        # to_do на создание релиза
        for ( qw/migration_tickets_all migration_warning/ ){
            next unless exists $migration_details->{$_};
            $desc_parts->{$_} = $migration_details->{$_}; 
        }
        $desc_parts->{need_create_migration_tickets} = 1 if keys %{$migration_details->{migration_files}||{}};

        my $description = logs2description(%$desc_parts);

        my $summary = make_release_summary(name => $O{name}, version => $O{version} ); 

        my $release_params = {
            create => 1, 
            assignee => [getpwuid($<)]->[0],
            description => $description,
            summary => $summary,
            type => 'release',
            queue => ProjectSpecific::get_project_data('default_tracker_queue'),
        };
        my $release_component;
        $release_component = eval { ProjectSpecific::get_project_data('startrek_release_component') };
        if ($release_component) {
            $release_params->{components} = $release_component;
        }

        if ($O{testserver}) {
            $release_params->{tags} = ["testserver:$O{testserver}" ];
        }

        push @{$to_do_struct->{release}}, $release_params;
    } else {
        my $release_params = {};
        $release_params->{key} = $old_values->{key};

        my $summary_details = parse_release_summary($old_values->{summary}); 
        my $prev_version = $summary_details->{version};
        $release_params->{summary} = $old_values->{summary};

        # если надо -- переименуем релиз...
        $release_params->{summary} = change_release_summary($release_params->{summary}, name => $O{name}) if ($O{rename});

        # если сменилась версия -- поменяем заголовок, описание, статусы + прокомментируем
        if ( $O{version} && $O{version} ne $prev_version || $O{force}){
            if ( $O{version} lt $prev_version && ! $O{downgrade} ){
                print "VERY suspicious: new version of packages less than previous one. Use --downgrade if absolutely indispensable.";
                exit(1);
            }

            my $rev_1 = version2rev($prev_version, name => 'prev', rev => 'last');
            my $rev_2 = version2rev($O{version},   name => 'new', rev => 'last');
            my $base_rev = version2rev($O{version}, name => 'new', rev => 'base');
            # откуда брать диффы: для транковой сборки -- в транке, для бранчевой -- в бранче 
            my $release_svn_url = $base_rev eq $rev_2 ? svn_url('trunk') : svn_url(release => $base_rev);
            $release_params->{summary} = change_release_summary($release_params->{summary}, version => $O{version});

            my $desc_parts = description2logs($old_values->{description});

            # каждый раз полностью пересоставляем migration-сообщение
            # ??? что должно быть, если модифицирован старый-старый migration-файл?
            my $rev_0;
            ($rev_0) = $desc_parts->{changelog} =~ m/^.*?r(\d+)/sm or die "can't determine base revision for release";
            # 2 релиза перешли через переезд: были собраны из старого репозитория, а хотфиксились из нового
            # для них фиксим стартовую ревизию, чтобы в релиз не записались миграции за всю историю (но в DIRECTMIGR они успели создаться :( )
            $rev_0 = 4927041 if $rev_0 == 173819;
            $rev_0 = 4926910 if $rev_0 == 173690;

            #hash_merge
            # где искать миграции: для транковых версий -- в транке, для бранчевых -- в бранче
            my $migration_details = migration_details(start_rev => $rev_0 - 1, end_rev => $rev_2, deploy_svn_url => "$release_svn_url/deploy" );
            # to_do на миграции
            # 1. Создаем
            for my $migr_file ( sort keys %{$migration_details->{migration_files}||{}} ){
                my $migr_params = {
                    create => 1, 
                    type => 'task',
                    description => $migration_details->{migration_files}->{$migr_file},
                    summary => "direct: Миграция $migr_file",
                    queue => "DIRECTMIGR",
                };
                push @{$to_do_struct->{migrations}}, $migr_params;
            }
            # 2. Обновляем
            for my $migr_ticket ( sort keys %{$migration_details->{migration_tickets_to_update}||{}} ){
                #print "migr_ticket $migr_ticket\n$migration_details->{migration_tickets_to_update}->{$migr_ticket}\n\n";
                my $migr_params = {
                    key => $migr_ticket,
                    description => "$migration_details->{migration_tickets_to_update}->{$migr_ticket}",
                    comment => 'текст миграции обновлен',
                };
                push @{$to_do_struct->{migrations}}, $migr_params;
            }
            for ( qw/migration_tickets_all migration_warning/ ){
                next unless exists $migration_details->{$_};
                $desc_parts->{$_} = $migration_details->{$_}; 
            }
            $desc_parts->{need_create_migration_tickets} = 1 if keys %{$migration_details->{migration_files}||{}};

            my $migration_svn_changed = qx!svn diff -r$rev_1:$rev_2 $release_svn_url/deploy!;

            # приписываем куда следует дополнения к changelog -- для нового тикета в description, для тестирующегося -- в комментарий
            my $addition_log_text = log_text(start_rev => $rev_1 + 1, end_rev => $rev_2, svn_url => $release_svn_url );
            if ( $addition_log_text ){
                if ($old_values->{status} eq 'new' || $old_values->{status} eq 'readyForTest') {
                    $desc_parts->{changelog} = $desc_parts->{changelog}."\n".$addition_log_text;
                } else {
                    my $testing = $old_values->{status} eq 'testing';
                    my $testupdate_message = $testing ? "\n".test_update_message_for_tracker( testupdate_url => tabula_testupdate_url($O{version})) : "";
                    $release_params->{comment} = 
                    ($testing ? "В релиз включен новый код": "Добавлены хотфиксы")
                    . ", версия: $prev_version --> $O{version}"
                    . $testupdate_message
                    ."\n$addition_log_text\n"
                    .''; 
                }
            }

            # Собираем полный description
            if (1){
                $release_params->{description} = logs2description(%$desc_parts);
            }

            # если у тестирующегося тикета меняется migration-сообщение -- отдельно отмечаем это в комментарии
            if ($migration_svn_changed && $old_values->{status} ne 'new') { 
                $release_params->{comment} = $release_params->{comment}."\nВНИМАНИЕ!!! Обновлена инструкция по выкладке!"
            }

            # Если релиз уже выложен и закрыт -- переоткрываем, игнорируем тестирование; раньше еще делали accept, но потом его разрешили не всем
            if( ($O{version} ne $prev_version) && $old_values->{status} eq 'closed') {
                $release_params->{actions} = [ qw/reopen ignore_testing/ ]; 
            }
        }

        push @{$to_do_struct->{release}}, $release_params;
    }

    return $to_do_struct;
}

sub migration_to_ticket_description {
    my ($from_file) = @_;

    return "%%\n$from_file\n%%\n"
    . "----\n"
    . "!!Внимание!!! Данный тикет нельзя править руками. См. ((https://wiki.yandex-team.ru/direct/development/howto/directmigr/ инструкцию))\n";
}

# отдает текст svn-лога начиная с ревизии $O{start_rev} и до ревизии $O{end_rev}
# копия логки из direct/packages/Makefile, таргет release_log
sub log_text 
{
    my %O = @_;
    die("bad parameters: \$O{start_rev} = $O{start_rev}, \$O{end_rev} = $O{end_rev}, \$O{svn_url} = $O{svn_url}") if !$O{start_rev} || !$O{end_rev} || !$O{svn_url};
    # TODO работать без рабочей копии

    return '' unless $O{start_rev} <= $O{end_rev};
    #my @log = qx/make release_log START=$O{start_rev} END=$O{end_rev}/;
    # svn log -r ${START}:${END} $(SVN_ROOT) | perl -pe 's/^----+.*\n//;s/^\s+$$//;s/^(r\d+\s)/\n$$1/;'
    my @log = qx/svn log -r $O{start_rev}:$O{end_rev} $O{svn_url}/;
    shift @log;
    for (@log){
        s/^----+.*\n//;
        s/^\s+$//;
        s/^(r\d+\s)/\n$1/;
    }

    my $log_text = join "", @log;

    # Для хотфиксов: подчищаем комментарии от ненужного
    $log_text =~ s/^(r\d+\s)[^\n]*\n+(RELEASE: )?(Merged|Created)[^\n]*//gsm; 
    # Для релизных бранчей -- убираем маркер "RELEASE"
    $log_text =~ s/^RELEASE: *\n//gsm; 
    # И просто убираем лишние пустые строки
    $log_text=~s/\n{3,}/\n\n/gsm;

    return $log_text;
}

sub migration_details
{
    my %O = @_;
    die unless $O{deploy_svn_url};

    # filterdiff используем на случай, если в deploy у файлов менялись svn-properties: Text::Diff::Parser не умеет такое разбирать
    my @log = qx!svn diff --depth=files --no-diff-deleted -r $O{start_rev}:$O{end_rev} $O{deploy_svn_url} |filterdiff --clean!;
    my $raw_log_text = join '', @log;
    my $diff_parser = Text::Diff::Parser->new( Diff=>$raw_log_text );

    my $warning = '';
    my %modified;
    #die qq!svn diff -r $O{start_rev}:$O{end_rev} --summarize $O{deploy_svn_url} | "grep" '^M' --color=never!;
    my $M = qx!svn diff -r $O{start_rev}:$O{end_rev} --summarize $O{deploy_svn_url} | "grep" '^M' --color=never!;
    if ($M) {
        # модифицирована (не добавлена, а именно модифицирована) migration-запись
        $warning = "ВНИМАНИЕ!!!\nМодифицирована migration-запись, необходима проверка:\n$M\n";
        %modified = map { s!.*\s.*/(\S+)!$1!; $_ => 1} split /\n/, $M;
    }

    my $files = {};
    foreach my $change ( $diff_parser->changes ) {
        my $filename = $change->filename2;
        # Соглашение: файлы *.data не включаем в список
        next if $filename =~ m/\.data$/;
        # бывают файлы *.csv (по смыслу те же *.data), их тоже пропускаем
        next if $filename =~ m/\.csv$/;
        # shell-ных миграций не бывает, но разрешаем скрипты складывать в /deploy как данные (инструкция по запуску будет в отдельном .migr)
        next if $filename =~ m/\.sh$/;
        # модифицированные (не добавленные, а отредактированные) файлы пропускем
        next if $modified{$filename};

        my $size = $change->size;
        my @lines = map { $change->text( $_ ) } 0..($size-1);

        my $text=join "\n", @lines;
        # YAML::Load не работает, если текст не заканчивается переводом строки
        $text .= "\n" unless $text =~ /\n$/;
        $files->{ $filename } = migration_to_ticket_description(MigratorB::Parse::to_text($filename => $text))
    }

    # миграции, что уже созданы тикетами (захотфиксились, например), записываем в отдельный список
    my $migr_tickets_all = [];
    my $migr_tickets_to_update = {};
    for my $migr_file ( sort keys %$files ){
        my $ticket = find_migration_ticket($migr_file);
        if ( $ticket ){
            my $migr_text = delete $files->{$migr_file};
            push @$migr_tickets_all, $ticket->{key};
            ### если миграция есть, но текст неактуальный -- надо обновлять текст
            if ( $ticket->{description} ne $migr_text ){
                $migr_tickets_to_update->{$ticket->{key}} = $migr_text; 
            }
        }
    }

    return {
        migration_files => $files,
        migration_tickets_all => $migr_tickets_all,
        migration_tickets_to_update => $migr_tickets_to_update,
        migration_warning => $warning,
    };
}

sub test_update_message_for_tracker
{
    my %O = @_;

    my $message = sprintf "обновление ТС доступно по ссылке %s", $O{testupdate_url};

    return $message;
}

sub find_migration_ticket
{
    my ($migr_file) = @_;
    my $issues = get_tracker_object()->get( query => qq!Queue: DIRECTMIGR Summary: #"direct: Миграция $migr_file"!, array => 1);
    if (scalar @$issues == 0){
        return '';
    } elsif ( scalar @$issues == 1 ) {
        return $issues->[0];
    } elsif (scalar @$issues > 1){
        die "only one ticket should be created for a migration ($migr_file), found ".(scalar @$issues).": ".join(", ", (map {$_->{key}} @$issues));
    } else {
        die "can't be";
    }
}

# печатает сводку по запланированным изменениям в релиз-тикете
sub print_updates
{
    my ($release, $to_do_struct) = @_;

    for my $type (qw/migrations release/){
        for my $to_do (@{$to_do_struct->{$type} || []}){
            print "\n";
            if( $to_do->{create} ){ 
                my $warning_msg = '';
                $warning_msg = "\nWARNING: DEFAULT RELEASE NAME! Use -n <description> to name release more meaningful\n" if $type eq 'release' && parse_release_summary($to_do->{summary})->{name} eq $DEFAULT_RELEASE_NAME;

                print "===To create===\n$to_do->{summary}\n$warning_msg\n\n";
                print "Description:\n$to_do->{description}\n"
            } else {
                print "===To update: $to_do->{key}===\nSummary: ".(exists $to_do->{summary} ? $to_do->{summary} : "no changes")."\n\n"; 
                if (exists $to_do->{description} && $type eq 'release') {
                    my $old_desc = "$release->{description}\n";
                    my $new_desc = "$to_do->{description}\n";
                    my $desc_diff = diff \$old_desc, \$new_desc;
                    print "Description diff:\n$desc_diff\n";
                } elsif ( exists $to_do->{description} ) {
                    ### TODO дифф хорошо показывается только для релиза, а надо и для миграций тоже
                    print "New description:\n$to_do->{description}\n";
                } else {
                    print "Description: no changes\n";
                }
            }
            print "Comment:\n$to_do->{comment}\n" if $to_do->{comment};
            print "Actions:\n".join(', ', @{$to_do->{actions}})."\n" if exists $to_do->{actions};
            print "Add followers:\n" . join(', ', @{$to_do->{followers}}) . "\n" if exists $to_do->{followers};
        }
    }
}
