#!/usr/bin/python
# -*- coding: utf-8 -*-

import os
import sys

sys.path.insert(0, '/opt/direct-py/startrek-python-client-sni-fix')
from startrek_client import Startrek

import json
import re
import datetime
import requests
import subprocess
import time
import yaml
import logging
logging.getLogger("startrek_client.collections").addHandler(logging.NullHandler())

from kazoo.client import KazooClient
logging.getLogger("kazoo.client").addHandler(logging.NullHandler())

STARTREK_TOKEN_FILE = '~/.startrek_client_token'
SCRIPT_NAME = os.path.basename(__file__)
APPS_CONF_FILE = "/etc/yandex-direct/direct-apps.conf.yaml"
APPS_CONFIG = None

startrek_token = None
startrek_client = None

def send_message(message, quiet=False):
    test_chat_id = '-1001359667477'
    release_chat_id = '-1001072994247'
    url_message = 'http://sparkle-deploy.yandex.net/plugin-telegram-bot'
    if quiet:
        url_message += '?quiet'

    json_message = {
        'chat-id': release_chat_id,
        'message': message
    }
    r = requests.post(url_message, json=json_message)
    print r.text
    #r.raise_for_status()

    return

def send_metrics(app, duration, success):
    """
    отправляем метрики в metrics.log, потом их можно смотреть в logviewer, пример: https://direct.yandex.ru/logviewer/short/3xFM3dxO3YLp8L
    """
    metrics_url = 'http://intapi.direct.yandex.ru/metrics/add'
    name = 'autorelease.create.%s.sec' % app
    data = {
        'metrics': [{
            'name': name,
            'value': duration,
            'context': {'app': app, 'success': 1 if success else 0}
        }]
    }
    headers = {'Content-Type': 'application/json', 'Accept': 'application/json'}
    requests.post(metrics_url, json=data, headers=headers, timeout=2),
    return


def run_create_release(app, assignee):
    os.environ['PATH'] = "%s:/usr/local/bin" % os.environ.get("PATH", "")
    if app == 'dna':
        cmd = [
                "direct-dna-release",
                "--new",
                "--robot-ppc",
                ]
    else:
        os.environ['SANDBOX_TOKEN_FILE'] = "/etc/direct-tokens/sandbox_oauth_token_ppc"
        os.environ['DIRECT_RELEASE_ALLOW_AUTO_RFT'] = "1"
        os.environ['DIRECT_RELEASE_ASSIGNEE'] = assignee
        os.environ['DIRECT_RELEASE_QA_ENGINEER'] = assignee
        os.environ['DIRECT_RELEASE_NO_TMUX'] = "1"
        os.environ['DIRECT_RELEASE_NO_MASTER_CONNECTION'] = "1"
        os.environ['DIRECT_RELEASE_NO_NOTIFICATIONS'] = "1"
        os.environ['DIRECT_RELEASE_SANDBOX_TASK_TIMEOUT'] = "4800"

        if app == 'direct':
            os.environ['DIRECT_RELEASE_DUPLOAD_WITH_LOGIN'] = "ppc"
            os.environ['DIRECT_RELEASE_GPG_PASSPHRASE_FILE'] = "/home/ppc/.gnupg/gpg-pass"
            os.environ['DIRECT_RELEASE_NO_GPG_AGENT'] = "1"
            os.environ['DEBEMAIL'] = "ppc@yandex-team.ru"
            os.environ['DEBFULLNAME'] = "Robot ppc"
            os.environ['DIRECT_RELEASE_ALLOW_AUTO_RFT'] = ""

        cmd = [
                "direct-release",
                "-a",
                app,
                "-s",
                "testing",
                "create-release"
        ]

    now_str = datetime.datetime.now().strftime("%Y-%m-%d-%H-%M-%S")
    base_filename = '/var/log/yandex/autoreleases/direct-release-%s-%s' % (app, now_str)
    file_stdout = base_filename + '.stdout'
    file_stderr = base_filename + '.stderr'
    with open(file_stdout, "w") as fout, open(file_stderr, "w") as ferr:
        exit_code = subprocess.call(cmd, stdout=fout, stderr=ferr)

    return exit_code


def find_releases_in_statuses(app, statuses):
    component = APPS_CONFIG[app]['tracker-component']
    statuses_str = ", ".join( [ '"%s"' % s for s in statuses] )
    ST_QUERY = u'Queue: DIRECT Type: Release Components: "%s" Status: %s' % (component, statuses_str)
    tickets = startrek_client.issues.find(ST_QUERY)
    return tickets


def find_new_releases(app):
    # сразу после создания тикета он не находится через API (репликация?), поэтому повторяем поиск несколько раз с паузами
    releases = []
    for i in range(0, 5):
        releases = find_releases_in_statuses(app, ['New', 'Ready For Test'])
        if len(releases) > 0:
            break
        time.sleep(15)
    return releases


def ok_to_create_release(app, insistence):
    # проверяем, что нет тестирующихся релизов текущего приложения
    releases_to_check = [ {'app': app, 'statuses': ['Testing', 'New', 'Ready For Test'] } ]

    # java-web и dna собираем парами и только если оба предыдущих релиза дотестировались
    # но с настойчивостью (insistence) 2 и больше -- не проверяем
    if app == 'java-web' and insistence <= 1:
        releases_to_check.append({'app': 'dna', 'statuses': [ 'Testing' ] } )
    if app == 'dna' and insistence <= 1:
        releases_to_check.append({'app': 'java-web', 'statuses': [ 'Testing' ] } )

    # steps -- особое приложение
    if app == 'steps':
        return True

    ok = True
    for c in releases_to_check:
        tickets = find_releases_in_statuses(c['app'], c['statuses'])
        if len(tickets) > 0:
            ok = False
    if not ok:
        return ok

    if app == 'dna':
        return ok

    # здесь проверяем, что нет запущенной сборки. Если есть -- sleep упадет об блокировку
    os.environ['DIRECT_RELEASE_NO_NOTIFICATIONS'] = "1"
    to_run = ['direct-release', '-a', app, '-s', 'testing', 'sleep', '1']
    with open(os.devnull, 'w') as FNULL:
        exit_code = subprocess.call(to_run, stdout=FNULL, stderr=FNULL)

    if exit_code != 0:
        ok = False

    return ok


def check_and_set_last_try(zkh, app, limit, dry_run):
    now = datetime.datetime.now()

    zk_path = "/direct/autorelease-ctl/%s/current" % app
    now = datetime.datetime.now()

    last_try_json, stat = zkh.get(zk_path)
    try:
        last_try = json.loads(last_try_json)
        last_try_time = datetime.datetime.strptime(last_try['start'], '%Y-%m-%d %H:%M:%S')
        delta = (now - last_try_time).total_seconds()
    except:
        # если не получилось получить последнюю сборку из ZK, то считаем, что была давно
        delta = 86400

    if delta < limit:
        return False

    new_last_try = {
            "start": now.strftime('%Y-%m-%d %H:%M:%S'),
            }

    new_last_try_json = json.dumps(new_last_try, sort_keys=True)
    if not dry_run:
        zkh.set(zk_path, new_last_try_json)

    return True

def check_release_schedule(app):
    """
    Проверяем, подходит ли текущее время под расписание сборки
    """
    cmd = [
            'dt-autorelease-ctl',
            '-a',
            app,
            'check',
            ]
    p = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
    release_qa = ''
    insistence = 1
    for line in p.stdout:
        m = re.search(r'release_qa:\s*(\S+)', line)
        if m:
            release_qa = m.group(1)
        m = re.search(r'insistence:\s*(\S+)', line)
        if m:
            insistence = int(m.group(1))
    p.communicate()
    ok = 1 if p.returncode == 0 else 0
    return (ok, release_qa, insistence)


def ready_for_test(releases):
    for issue in releases:
        if issue.status.key != 'new':
            continue
        issue.transitions['ready_for_test'].execute(comment=u"RFT")
    return


def set_release_qa_assignee(releases, release_qa):
    """
    устанавливаем исполнителя и QA-engineer в релизах
    """
    for issue in releases:
        try:
            issue.update(assignee=release_qa, qaEngineer=release_qa)
        except Exception as e:
            # считаем некритичным, если не получилось проставить QA-инженера
            print "failed to set assignee or qaEngineer for %s\n%s" % (issue.key, str(e))
    return


def run():
    global APPS_CONFIG
    global startrek_token
    global startrek_client

    dry_run = False

    os.environ['PATH'] = "%s:/usr/local/bin" % os.environ.get("PATH", "")
    os.environ['ABC_OAUTH_TOKEN_FILE'] = '/etc/direct-tokens/abc_oauth_token_ppc'

    if len(sys.argv) != 2:
        sys.exit("expecting exactly 1 parameter(s), stop")

    app = sys.argv[1]

    with open(APPS_CONF_FILE) as apps_fd:
        APPS_CONFIG = yaml.load(apps_fd)['apps']

    startrek_token = open(os.path.expanduser(STARTREK_TOKEN_FILE)).readline().strip()
    startrek_client = Startrek(token=startrek_token, useragent=SCRIPT_NAME)

    ZK_HOSTS = ['ppctest-zookeeper01i.sas.yp-c.yandex.net:2181', 'ppctest-zookeeper01f.myt.yp-c.yandex.net:2181', 'ppctest-zookeeper01v.vla.yp-c.yandex.net:2181']
    zkh = KazooClient(','.join(ZK_HOSTS))
    zkh.start()

    (schedule_ok, release_qa, insistence) = check_release_schedule(app)
    if not schedule_ok:
        exit(0)

    ok = ok_to_create_release(app, insistence)
    if not ok:
        exit(0)

    last_try_ok = check_and_set_last_try(zkh, app, limit=6*3600, dry_run=dry_run)
    if not last_try_ok:
        exit(0)

    if dry_run:
        print "dry_run: Ok to start create-release for app '%s'" % app
        exit(0)

    send_message(u"Начинается автосборка релиза %s" % app, quiet=True)
    start_time = time.time()
    exit_code = run_create_release(app, release_qa)
    end_time = time.time()
    success = exit_code == 0

    send_metrics( app, int( end_time - start_time ), success )

    if not success:
        send_message(u"❌ Автосборка релиза %s закончилась неудачно" % app, quiet=True)
        sys.exit("create-release finished with code %s" % exit_code)

    new_releases = find_new_releases(app)
    set_release_qa_assignee(new_releases, release_qa)
    if app == "dna":
        ready_for_test(new_releases)

    new_releases_str = "\n".join(["(%s) https://st.yandex-team.ru/%s" % (r.status.name, r.key) for r in new_releases])
    send_message(u"✅ Автосборка релиза %s закончилась успешно\n%s" % (app, new_releases_str), quiet=True)
    zkh.stop()
    exit(0)


if __name__ == '__main__':
    run()
