#!/usr/bin/perl

=head1 NAME

    java-release.pl -- собрать релиз java-приложения Директа

=head1 SYNOPSIS

    # создать релизный тикет, когда есть уже собранные пакеты
    java-release.pl --app logviewer --name '<название релиза>' --version 1.12345678-1   # вывести, что получается
    java-release.pl --app logviewer --name '<название релиза>' --version 1.12345678-1 --do  # сделать всё в стартреке

    # собрать пакеты в sandbox, есть небольшие неудобства
    java-release.pl --app logviewer --name '<название релиза>' --revision 12345678
    # или просто из свежего транка
    java-release.pl --app logviewer --name '<название релиза>'
    # если повторить эту команду с --do, снова запустится сборка пакетов, сейчас лучше указывать явно версию.

    # добавить хотфикс в релиз DIRECT-12345
    java-release.pl --app logviewer --ticket DIRECT-12345 --version 1.12346000-1
    # добавить хотфикс в релиз DIRECT-12345 и переоткрыть тикет
    java-release.pl --app logviewer --ticket DIRECT-12345 --reopen --version 1.12346000-1

    # переименовать релиз DIRECT-12345
    java-release.pl --ticket DIRECT-12345 --name '<новое название релиза>' --rename

=cut

use strict;
use warnings;

use utf8;

use open qw/:std :encoding(UTF-8)/;

use Getopt::Long;
use Startrek::Client::Easy;
use YAML;
use Yandex::Shell;
use XML::LibXML;
use DirectRelease::Description qw(description2logs logs2description);
eval "use DirectRelease::Migrations qw(migration_details what_to_do_migrations)";

$ENV{'SVN_SSH'} = "ssh -S $ENV{'SSH_MASTER_CONN_ARCADIA'}" if $ENV{'SSH_MASTER_CONN_ARCADIA'};

our $ARCADIA_URL = 'svn+ssh://arcadia.yandex.ru/arc';
our $VERSION_RE = '1\.([0-9]{7,})(?:\.([0-9]{7,}))?-[0-9]+';     # ($base_rev, $head_rev) = ($1, $2);
$Yandex::Shell::PRINT_COMMANDS = 1;
our $st;

run() unless caller();

sub run {
    my $opt = parse_options();
    $st = Startrek::Client::Easy->new();

    my ($changes, $to_do_migrations) = what_to_do_in_tracker($opt);

    my @migrations_created;
    
    if ($opt->{do}) {
        for my $to_do (@{$to_do_migrations->{migrations} || []}) {
            my $tracker_result = $st->do(%$to_do);
            if ($to_do->{create}) {
                push @migrations_created, $tracker_result;
            }
        }
    } else {
        print Dump($to_do_migrations->{migrations});
    }

    if (scalar @migrations_created > 0) {
        my $migrations_list = join "\n", @migrations_created;
        $changes->{description} =~ s/__MIGRATIONS_PLACEHOLDER__/$migrations_list/;
    }

    if ($opt->{do}) {
        my $res = $st->do(%$changes);
        if ($changes->{create}) {   # в этом случае в $res ключ тикета
            print "\nnew ticket: $res\nlink: https://st.yandex-team.ru/$res\n";
        }
    } else {
        print Dump($changes);
    }
    exit 0;
}

sub parse_options {
    my %O;
    GetOptions(
        'r|revision=i' => \$O{revision},
        'version=s' => \$O{version},
        'app=s' => \$O{app_full_name},
        'ticket=s' => \$O{ticket},
        'name=s' => \$O{name},
        'rename' => \$O{rename},
        'test-passed' => \$O{test_passed},
        'reopen' => \$O{reopen},
        'changelog-url=s' => \$O{changelog_url},
        'sandbox-task=s' => \$O{sandbox_task},
        'do' => \$O{do},
        'h|help' => sub { system("podselect -section SYNOPSIS -section DESCRIPTION $0 | pod2text-utf8 >&2"); exit 0 },
    ) || die "can't parse options, stop\n";
    defined && utf8::decode($_) for values(%O);

    die "no ticket given\n" if ($O{reopen} || $O{test_passed}) && !$O{ticket};
    die "no ticket or name given\n" if $O{rename} && (!$O{ticket} || !$O{name});
    die "nothing but ticket and name must be given to rename\n" if $O{rename} && ($O{revision} || $O{version} || $O{app_full_name});
    die "nothing but ticket to test-passed\n" if $O{test_passed} && ($O{revision} || $O{version} || $O{app_full_name});
    die "no app name given\n" if !$O{app_full_name} && !$O{rename} && !$O{test_passed};
    die "revision and version are mutually exclusive\n" if $O{revision} && $O{version};
    die "ticket key must be in format DIRECT-NNNNN" if $O{ticket} && $O{ticket} !~ /^DIRECT-[0-9]{5,}$/;
    die "no name for new release\n" if !$O{ticket} && !$O{name};
    if (! $O{revision} && ! $O{version} && ! $O{rename} && ! $O{test_passed}) {
        die "expecting one of params: --revision, --version, --rename, --test-passed, stop";
    }
    if( $O{revision} ){
        $O{version} = "1.$O{revision}-1";
        delete $O{revision};
    }
    return \%O;
}


sub what_to_do_in_tracker
{
    my ($opt) = @_;
    my %changes = ();
    my $to_do_migrations = {};

    my $version = $opt->{version};

    # Если только переименование тикета -- сразу же возвращаем результат (не надо ничего делать с миграциям, не надо знать приложение и т.д.)
    if ( $opt->{ticket} && $opt->{rename} && !$version ){
        # релизный тикет есть, новой версии нет, переименовываем
        my $r = $st->get( key => $opt->{ticket}, array => 1 )->[0];
        %changes = (
            (
                key => $opt->{ticket},
                summary => ($r->{summary} =~ s/(^.*?: )(.*)( - выложить .*)/$1$opt->{name}$3/r),
            ),
        );
        return (\%changes, $to_do_migrations);
    }

    if ( $opt->{ticket} && $opt->{test_passed}) {
        my $r = $st->get( key => $opt->{ticket}, array => 1 )->[0];
        my $status = $r->{status};
        if (grep { $status eq $_ } qw/new readyForTest/) { # в остальных состояниях считаем релиз протестированным, можно уточнять при необходимости
            die "error: release is not yet in testing (status is '$status'), can't finish testing\n";
        }
        my @allowed_transitions = map { $_->{id} } @{ $st->request('GET', "/issues/$opt->{ticket}/transitions") };
        %changes = (
        key => $opt->{ticket},);
        if (grep { $_ eq 'passed'} @allowed_transitions) {
            $changes{actions} = ['passed'];
        } else {
            warn "status for ticket $opt->{ticket} is '$status', skipping transition\n"; 
        }
        return (\%changes, $to_do_migrations);
    }

    my $app_full_name = $opt->{app_full_name} or die "empty app name";
    my $app_conf = YAML::LoadFile("/etc/yandex-direct/direct-apps.conf.yaml")->{apps};
    die "can't find app='$app_full_name' in apps-config" unless exists $app_conf->{"$app_full_name"};
    
    $app_conf = $app_conf->{"$app_full_name"};
    my $primary_tracker_component = $app_conf->{'tracker-component'};

    # В $r попадет либо текущий обрабатываемый релиз, либо предыдущий в этом приложении. Инвариант: changelog считаем от версии, записанной в тикете $r
    my $r = $st->get(
        $opt->{ticket} ? (key => $opt->{ticket}) : (query => qq/Queue: DIRECT Type: Release Components: "$primary_tracker_component" "Sort by": key desc/),
        array => 1,
    )->[0];
    die "could not find release ticket $opt->{ticket} in tracker\n" if $opt->{ticket} && ! $r;

    # Выясняем предыдущую версию и changelog после нее. 
    # Если это самый первый релиз (!$r) -- оставляем то и другое пустым
    # Если хотим только переименовать релиз -- оставляем то и другое пустым тоже
    my ($old_version, $changelog, $migration_details) = ( '', '' );
    if( $r && !$opt->{rename} ){
        $r->{summary} =~ /\b($VERSION_RE)\b/;
        $old_version = $1;

        if ( $opt->{version} lt $old_version ){
            print "VERY suspicious: new version of packages ($opt->{version}) less than previous one ($old_version). Downgrades aren't supported at the moment.";
            exit(1);
        }

        if ($opt->{changelog_url}) {
            $changelog = yash_qx("svn cat $opt->{changelog_url}");
        } else {
            my $ticket_param = $opt->{ticket} ? "-t $opt->{ticket}": "";
            $changelog = yash_qx("release-changelog $opt->{app_full_name} $opt->{version} $ticket_param");
        }
        utf8::decode($changelog);

        $migration_details = migration_details(
            release_issue => $r,
            version => $version,
            ticket => $opt->{ticket},
            app_full_name => $opt->{app_full_name},
        );
    }

    my $desc_parts = {};

    $to_do_migrations = what_to_do_migrations($migration_details, $opt->{app_full_name});

    # to_do на создание релиза
    for ( qw/migration_tickets_all migration_warning/ ){
        next unless exists $migration_details->{$_};
        $desc_parts->{$_} = $migration_details->{$_};
    }
    $desc_parts->{need_create_migration_tickets} = 1 if keys %{$migration_details->{migration_files}||{}};

    # Что надо сделать в Стартреке
    if ( $opt->{ticket} && !$opt->{rename} && $version ) {
        $desc_parts->{changelog} = description2logs($r->{description})->{changelog};

        # релизный тикет есть, меняем версию
        %changes = 
        (
            key => $opt->{ticket},
            comment => "В релиз включен новый код, версия: $old_version --> $version\n\n$changelog",
            summary => ($r->{summary} =~ s/\Q$old_version\E/$version/r),
            description => logs2description(%$desc_parts),
            '6061b65f6669ed0a30e9e041--sandboxBuildTask' => $opt->{sandbox_task},
            ($opt->{reopen} && $r->{status} eq 'closed' ? (actions => [ qw/reopen ignore_testing/ ]) : ()),
        );
        
        if (scalar @{$to_do_migrations->{migrations}} || $migration_details->{migration_warning}) {
            $changes{comment} = $changes{comment}."\nВНИМАНИЕ!!! Обновлена инструкция по выкладке!";
        }
    } elsif (! $opt->{ticket} && !$opt->{rename} && $version) {
        $desc_parts->{changelog} = $changelog;

        # создаем новый релизный тикет
        %changes = ( create => 1,
            queue => 'DIRECT',
            type => 'release',
            description => logs2description(%$desc_parts),
            components => [$primary_tracker_component, "Releases: JavaDirect", ],
            assignee => $ENV{DIRECT_RELEASE_ASSIGNEE} || (getpwuid($<))[0],
            summary => "релиз $opt->{app_full_name}: $opt->{name} - выложить $version",
            '6061b65f6669ed0a30e9e041--sandboxBuildTask' => $opt->{sandbox_task},
        );
    } else {
        die "can't be: \$opt->{ticket}=$opt->{ticket} \$version=$version, \$opt->{rename}=$opt->{rename}";
    }
    
    return (\%changes, $to_do_migrations);
}

1;
