#!/usr/bin/python
# -*- encoding: utf-8 -*-

import difflib
import os
import re
import socket
import sys

sys.path.insert(0, '/opt/direct-py/startrek-python-client-sni-fix')
from startrek_client import Startrek
import direct_juggler.juggler as dj

YABS_ROBOT_ID = 'robot-yabs-autosupbs'
TAG = 'show_auto_supbs_diff'
STARTREK_TOKEN_FILE = '/etc/direct-tokens/startrek'
SCRIPT_NAME = os.path.basename(__file__)
HOSTNAME = socket.getfqdn()
SIGN = u"----\nСкрипт %s с машины %s" % (SCRIPT_NAME, HOSTNAME)
SERVICE_NAME = 'scripts.show-auto-supbs-diff-in-issues.working'

DESCRIPTION_URL = "https://wiki.yandex-team.ru/direct/Development/howto/show-auto-supbs-diff-in-issues/"
IGNORE_BLOCK_KEY = u"Также вам могут пригодиться следующие запросы"
with open(STARTREK_TOKEN_FILE) as st_fd:
    startrek_token = st_fd.readline().strip()


def get_blocks(text):
    """
    Генератор, который находит и возвращает последовательно блоки для дальнейшего сравнения
    :return: ключ блока, содержимое блока (ключ включен в содержимое)
    """
    iterator = iter(text.splitlines())
    current_line = next(iterator, None)
    while current_line is not None:
        if current_line.startswith("**"):
            block_key = current_line.replace("**", "")
            yield block_key, current_line
        elif current_line.startswith("<{"):
            block_key = current_line.replace("<{", "")
            # блок со справочной информацией в конце. не нужно сравнивать - пропускаем и заканчиваем поиск
            if block_key == IGNORE_BLOCK_KEY:
                return

            collapse_block = ""
            while current_line is not None:
                collapse_block += current_line + "\n"
                if current_line.endswith("}>"):
                    break
                current_line = next(iterator, None)
            if not current_line.endswith("}>"):
                raise RuntimeError("Could not find end of collapse block")

            yield block_key, collapse_block
        else:
            raise RuntimeError("Unknown type of block: " + current_line)

        current_line = next(iterator, None)


def get_data_of_collapse_block(block):
    """
    Возвращает список строк содержимого блока в удобном формате для сравнения. Удаляет спец. символы форматирования и
    экранирует строки начинающиеся на +, чтобы в diff'е не отображались как новые строки
    """
    block = re.sub("^<{.*\n", "", block)
    block = re.sub("^%%\n", "", block)
    block = re.sub("}>$", "", block)
    block = re.sub("%%$", "", block)
    block = re.sub("^\+", "\\+", block, flags=re.M)
    return block.splitlines(True)


def get_diff_of_collapse_block(key, expected_block, actual_block):
    """
    Вычисляет дифф содержимого collapse-блоков
    :return: строку в формате collapse-блока, внутри которого будет дифф
    """
    expected_data = get_data_of_collapse_block(expected_block)
    actual_data = get_data_of_collapse_block(actual_block)

    diff = ''.join(difflib.ndiff(expected_data, actual_data))
    return u"<{%s\n%%%%(diff)\n%s%%%%}>" % (key, diff)


def compare(expected_text, actual_text):
    expected_blocks = {key: block_text for key, block_text in get_blocks(expected_text)}

    result = ""
    actual_blocks_keys = set()
    for block_key, block_text in get_blocks(actual_text):
        actual_blocks_keys.add(block_key)

        if block_key in expected_blocks and expected_blocks[block_key] != block_text:
            diff = get_diff_of_collapse_block(block_key, expected_blocks[block_key], block_text)
            result += u"**!!(крас)есть разница в данных:!!** {}\n".format(diff)
        elif block_key not in expected_blocks:
            result += u"**!!(зел)добавлено:!!** {}\n".format(block_text)

    for block_key in expected_blocks.keys():
        if block_key not in actual_blocks_keys:
            result += u"**!!(крас)удалено:!!** {}\n".format(expected_blocks[block_key])

    return result


def generate_link_for_comment(ticket_key, comment_id):
    return "https://st.yandex-team.ru/{0}#{1}".format(ticket_key, comment_id)


def get_summonees(ticket):
    summonees = None
    if ticket.qaEngineer:
        summonees = [ticket.qaEngineer.id]
    elif ticket.assignee:
        summonees = [ticket.assignee.id]
    elif ticket.createdBy and ticket.createdBy.id != 'ppc':
        summonees = [ticket.createdBy.id]

    return summonees


def get_comments_to_compare(comments):
    first_comment = None
    last_comment = None
    for comment in comments:
        if comment.createdBy.id != YABS_ROBOT_ID:
            continue

        if first_comment is None:
            first_comment = comment
        else:
            last_comment = comment

    return (first_comment, last_comment)


def compare_and_get_result_for_human(ticket_key, first_comment, last_comment):
    result = compare(first_comment.text, last_comment.text)
    link_to_first_comment = generate_link_for_comment(ticket_key, first_comment.longId)
    link_to_last_comment = generate_link_for_comment(ticket_key, last_comment.longId)
    comments_description = u"""\n----\n**(({0} Что это?))**
                                    \n(({1} Комментарий с ожидаемыми параметрами)).
                                   \n(({2} Комментарий с актуальными параметрами)).""". \
        format(DESCRIPTION_URL, link_to_first_comment, link_to_last_comment)

    if not result:
        return u"**Различий не обнаружено.**" + comments_description
    else:
        return u"**Результат сравнения:**\n{0}\n{1}".format(result, comments_description)


def process_ticket(startrek_client, ticket):
    comments = startrek_client.issues[ticket.key].comments.get_all()
    first_comment, last_comment = get_comments_to_compare(comments)

    summonees = get_summonees(ticket)
    if first_comment is None or last_comment is None:
        startrek_client.issues[ticket.key].comments.create(
            text=u"В тикете не нашлось двух комментариев от робота %%{}%% для сравнения".format(
                YABS_ROBOT_ID) + SIGN,
            summonees=summonees,
        )
        return

    result = compare_and_get_result_for_human(ticket.key, first_comment, last_comment)
    startrek_client.issues[ticket.key].comments.create(
        text=result + SIGN,
        summonees=summonees,
    )


def run():
    startrek_client = Startrek(token=startrek_token, useragent=SCRIPT_NAME)

    tickets = startrek_client.issues.find(u'Queue: DIRECT Status: !Closed Tags: %s' % TAG)
    tickets_to_remove_tag = []

    for ticket in tickets:
        try:
            process_ticket(startrek_client, ticket)
            tickets_to_remove_tag.append(ticket)
        except Exception as e:
            print e, type(e)

    if tickets_to_remove_tag:
        startrek_client.bulkchange.update(
            list(tickets_to_remove_tag),
            tags={'remove': [TAG]}
        ).wait()
    return


if __name__ == '__main__':
    try:
        run()
        dj.queue_events([{
            'service': SERVICE_NAME,
            'status': 'OK',
            'description': 'OK',
        }])
    except Exception as e:
        dj.queue_events([{
            'service': SERVICE_NAME,
            'status': 'CRIT',
            'description': "unexpected exception (%s) %s" % (type(e), e)
        }])
