#!/usr/bin/python
# -*- coding: utf-8 -*-

import sys
sys.path.insert(0, '/opt/direct-py/startrek-python-client-sni-fix')

import argparse
import json
import logging
import os
import pipes
import re
import requests
import subprocess
import tempfile
import yaml

from kazoo.client import KazooClient
from startrek_client import Startrek

SCRIPT_NAME = None
HOSTNAME = None
ST = None
APP_CONFIG = None
SANDBOX_EXPECTED_END_STATES = ['SUCCESS', 'FAILURE']
SANDBOX_UNEXPECTED_END_STATES = ['STOPPED', 'EXCEPTION']
SANDBOX_END_STATES = SANDBOX_EXPECTED_END_STATES + SANDBOX_UNEXPECTED_END_STATES
MAX_LAUNCH_COUNT = 3

class RestarterState:
    def __init__(self, zk, zk_node_path):
        self.zk = zk
        self.zk_node_path = zk_node_path
        self.clear()

    def clear(self):
        self.regression_ticket_key = None
        self.release_ticket_key = None
        self.task_id = None
        self.commented_on_start = False
        self.commented_on_finish = False
        self.finished = False
        self.launch_count = 0
        self.need_restart = True

    def load(self):
        state_json, stat = self.zk.get(self.zk_node_path)
        if state_json:
            state_struct = json.loads(state_json)
        else:
            state_struct = {}

        self.zk_node_version = stat.version
        self.finished = state_struct.get('finished', False)
        self.task_id = state_struct.get('task_id', None)
        self.commented_on_start = state_struct.get('commented_on_start', False)
        self.commented_on_finish = state_struct.get('commented_on_finish', False)
        self.release_ticket_key = state_struct.get('release_ticket', None)
        self.regression_ticket_key = state_struct.get('regression_ticket', None)
        self.launch_count = int(state_struct.get('launch_count', 0))
        self.need_restart = state_struct.get('need_restart', True)

    def save(self):
        state_json = json.dumps({
            'finished': self.finished,
            'task_id': self.task_id,
            'commented_on_start': self.commented_on_start,
            'commented_on_finish': self.commented_on_finish,
            'release_ticket': self.release_ticket_key,
            'regression_ticket': self.regression_ticket_key,
            'launch_count': self.launch_count,
            'need_restart': self.need_restart,
        })
        self.zk.set(self.zk_node_path, state_json, self.zk_node_version)
        _, stat = self.zk.get(self.zk_node_path)
        self.zk_node_version = stat.version


def get_last_release_ticket():
    global APP_CONFIG
    global ST
    release_tickets = list(ST.issues.find('Queue: DIRECT Type: Release Components: "%s" (Status: "Ready for test" or Status: "Testing") "Sort by": key desc' % APP_CONFIG['tracker-component']))
    if len(release_tickets) == 0:
        return None
    else:
        return release_tickets[0]

def get_dna_version():
    return subprocess.check_output(['/usr/local/bin/dna-utils', 'get-latest-release-package']).rstrip()

def get_dna_release_branch_from_version(version):
    version_re = r'^[0-9]+.[0-9]+.[0-9]+'
    if not re.match(version_re, version):
        raise ValueError('Version should match "{}", but was "{}"'.format(version_re, version))
    return 'releases/direct/dna/' + version.split('-')[0]

def sign_comment(comment):
    global SCRIPT_NAME
    global HOSTNAME
    comment += u'----\nСкрипт {} с машины {}'.format(SCRIPT_NAME, HOSTNAME)
    return comment

def run():
    global SCRIPT_NAME
    SCRIPT_NAME = os.path.basename(__file__)

    global HOSTNAME
    HOSTNAME = subprocess.check_output(['hostname', '--fqdn'])

    parser = argparse.ArgumentParser()
    supported_apps = ['java-web', 'dna']
    supported_apps_str = ', '.join(supported_apps)
    parser.add_argument('app', help=u'имя приложения из apps-конфига. Поддерживаются: {}'.format(supported_apps_str))
    args = parser.parse_args()
    if args.app not in supported_apps:
        sys.exit('error: unsupported app "{}", must be one of: {}'.format(args.app, supported_apps))

    logging.basicConfig(stream=sys.stderr, level=logging.INFO, format='[%(asctime)s]\t%(message)s')

    global ST
    with open('/etc/direct-tokens/startrek', 'r') as f:
        token = f.read().rstrip()
        ST = Startrek(token=token, useragent='Direct/{}'.format(SCRIPT_NAME))

    global APP_CONFIG
    with open('/etc/yandex-direct/direct-apps.conf.yaml', 'r') as f:
        apps_config = yaml.load(f)
        APP_CONFIG = apps_config['apps'][args.app]

    zk_node_path = '/direct/dna-release-tests-launch/' + args.app
    zk_servers = ['ppctest-zookeeper01i.sas.yp-c.yandex.net:2181', 'ppctest-zookeeper01f.myt.yp-c.yandex.net:2181', 'ppctest-zookeeper01v.vla.yp-c.yandex.net:2181']
    zk = KazooClient(','.join(zk_servers))
    zk_lock = zk.Lock(zk_node_path, "lock")
    zk.start()
    zk.ensure_path(zk_node_path)
    if not zk_lock.acquire(blocking=False):
        logging.info("couldn't acquire lock on {}, exiting".format(zk_node_path))
        sys.exit(0)

    state = RestarterState(zk, zk_node_path)
    state.load()

    release_ticket = None
    if state.release_ticket_key:
        release_ticket = ST.issues[state.release_ticket_key]
    if not state.release_ticket_key or state.finished:
        release_ticket = get_last_release_ticket()
        if not release_ticket:
            logging.info('no release ticket found')
            sys.exit(0)
        logging.info('found release ticket: {}'.format(release_ticket.key))
        if state.finished and release_ticket.key != state.release_ticket_key:
            state.clear()
            state.release_ticket_key = release_ticket.key
            state.save()
        if state.finished and release_ticket.key == state.release_ticket_key:
            logging.info('tests for release {} already finished, exiting'.format(release_ticket.key))
            sys.exit(0)
    assert release_ticket

    dna_tests_ticket = None
    if not state.regression_ticket_key:
        for link in release_ticket.links:
            t = link.object
            if re.match(u'^Запуск.* автотестов +DNA +в +релизе ', t.display, re.I):
                dna_tests_ticket = t
                state.regression_ticket_key = dna_tests_ticket.key
                break
        if not dna_tests_ticket:
            logging.info('no regression ticket found for release {}'.format(release_ticket.key))
            sys.exit(0)
        logging.info('found regression ticket for release {}: {}'.format(release_ticket.key, dna_tests_ticket.key))
        state.save()
    else:
        dna_tests_ticket = ST.issues[state.regression_ticket_key]
    if not state.finished and dna_tests_ticket.status.key == 'closed':
        logging.info('regression ticket {} already closed, finishing'.format(dna_tests_ticket.key))
        state.finished = True
        state.save()
        sys.exit(0)
    assert dna_tests_ticket

    global MAX_LAUNCH_COUNT
    dna_release_branch = get_dna_release_branch_from_version(get_dna_version())
    cmd = ['/usr/local/bin/dna-utils', 'tests', '--tests-branch', dna_release_branch, '--beta-port', 'TS']
    cmd_str = ' '.join([pipes.quote(s) for s in cmd])
    if not state.task_id or (state.task_id and state.commented_on_finish and state.need_restart):
        cmd = ['/usr/local/bin/dna-utils', 'tests', '--tests-branch', dna_release_branch, '--beta-port', 'TS']
        if state.task_id:
            cmd = ['/usr/local/bin/dna-utils', 'rerun-failed', state.task_id]
        cmd_str = ' '.join([pipes.quote(s) for s in cmd])
        logging.info('running: ' + cmd_str)
        try:
            p = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
            for line in p.stdout:
                m = re.search(r'(https://sandbox\.yandex-team\.ru/task/([0-9]+))', line)
                if m:
                    task_url = m.group(1)
                    state.task_id = m.group(2)
                    state.commented_on_start = False
                    state.commented_on_finish = False
                    logging.info('started task in Sandbox: {}'.format(task_url))
                    state.launch_count += 1
                    break
        except OSError as e:
            if e.errno == 7:    # Argument list too long
                state.tests = None
                logging.warn('argument list is too long, resetting tests list')
                state.save()
                sys.exit(0)
            else:
                raise
        state.save()
    assert state.task_id

    if not state.commented_on_start:
        comment = u'Запущены тесты в Sandbox: https://sandbox.yandex-team.ru/task/{}\n'.format(state.task_id)
        comment += u'<{Команда для ручного запуска с ppcdev\n%%' + cmd_str + '\n%%}>\n'
        comment += u'По завершении запуска должен появиться комментарий.\n'
        dna_tests_ticket.comments.create(text=sign_comment(comment))
        state.commented_on_start = True
        state.commented_on_finish = False
        state.save()
    assert state.commented_on_start

    task_info = requests.get('https://sandbox.yandex-team.ru/api/v1.0/task/{}'.format(state.task_id)).json()
    task_status = task_info['status']

    if task_status in SANDBOX_END_STATES:
        state.need_restart = task_status != 'SUCCESS' and task_status in SANDBOX_EXPECTED_END_STATES and state.launch_count < MAX_LAUNCH_COUNT
        report_url = None

        if task_status in SANDBOX_EXPECTED_END_STATES:
            task_resources = requests.get('https://sandbox.yandex-team.ru/api/v1.0/task/{}/resources'.format(state.task_id)).json().get('items', [])
            try:
                report_resource = [r for r in task_resources if r['type'] == 'SANDBOX_CI_ARTIFACT'][0]
                report_url = report_resource['http']['proxy'] + '/report-hermione:ci/index.html'
            except IndexError:
                pass

        if not state.commented_on_finish:
            comment = u'((https://sandbox.yandex-team.ru/task/{} Запуск в Sandbox)) завершился со статусом {}\n'.format(state.task_id, task_status)
            if report_url:
                comment += u'Отчёт: {}\n'.format(report_url)
            if state.need_restart:
                comment += u'**Тесты будут перезапущены (осталось попыток: {} из {})**\n'.format(MAX_LAUNCH_COUNT - state.launch_count, MAX_LAUNCH_COUNT)
            dna_tests_ticket.comments.create(text=sign_comment(comment))
            state.commented_on_finish = True
        state.save()
    else:
        logging.info('task {} is still running, exiting'.format(state.task_id))
        sys.exit(0)

    if not state.need_restart:
        if 'dont_remind' in dna_tests_ticket.tags:
            dna_tests_ticket.update(tags=[t for t in dna_tests_ticket.tags if t != 'dont_remind'])
        state.finished = True
        state.save()

if __name__ == '__main__':
    run()
