# -*- coding: utf8 -*-

import os
import re
import sys
import shlex
import subprocess
import tempfile
import shutil
import time
import xml.etree.ElementTree as ET
import json
import yaml
import dateutil.parser


APPS_CONFIG_PATH = '/etc/yandex-direct/direct-apps.conf.yaml'
with open(APPS_CONFIG_PATH, 'r') as fh:
    APPS_CONFIG = yaml.load(fh)['apps']

GLOBAL_PATH = '/trunk/arcadia/direct'

os.environ['ARC_TOKEN_PATH'] = '/etc/direct-tokens/oauth_arc_robot-direct-arc-p'


def get_command_output(command):
    return subprocess.check_output(shlex.split(command), stderr=subprocess.PIPE)


def is_commit_in_app(rev, dependencies):
    """
    Проверяем, принадлежит ли коммит приложению (входит в зависимости приложения)
    """
    paths = rev.find('paths')
    for path_el in paths:
        path = path_el.text
        for dep in dependencies:
            if (path.startswith(dep, len(GLOBAL_PATH) + 1) and (len(GLOBAL_PATH) + len(dep) + 1 == len(path)
                    or path[len(GLOBAL_PATH) + len(dep) + 1] == '/')):
                return True
    return False


def get_app_dependencies(app, working_dir):
    """
    Получить список зависимостей приложения
    """
    # если не переопределить директорию кэша для ya, то dump dir-graph выдает неправильные результаты
    # при параллельном запуске для разных приложений
    ya_cache_dir = tempfile.mkdtemp(dir='/tmp/temp-ttl/ttl_1d')
    try:
        os.environ['YA_CACHE_DIR'] = ya_cache_dir
        dir_graph_output = get_command_output('%s dump dir-graph --plain --split %s' % (
            os.path.join(working_dir, "ya"),
            os.path.join(working_dir, APPS_CONFIG[app]['primary-svn-dir']),
        ))

        dir_graph = json.loads(dir_graph_output)

        common_path = 'direct'
        return [
            dep[len(common_path) + 1:]
            for dep in dir_graph
            if dep.startswith(common_path) and (len(dep) == len(common_path) or dep[len(common_path)] == '/')
        ]
    finally:
        os.environ.pop('YA_CACHE_DIR', None)
        shutil.rmtree(ya_cache_dir, ignore_errors=True)


def checkout_working_copy_svn(app, rev=None):
    """
    Создаем рабочую копию с помощью svn
    """
    working_dir = os.path.join(tempfile.mkdtemp(dir='/tmp/temp-ttl/ttl_1d'), 'arcadia')
    try:
        ya = get_command_output('svn cat svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/ya')
        clone = subprocess.Popen(
            ['python', '-', 'clone', working_dir], stdout=subprocess.PIPE, stdin=subprocess.PIPE
        )
        clone.communicate(input=ya)[0]
        ya_path = os.path.join(working_dir, "ya")
        if clone.returncode:
            raise ValueError, "python clone has non-zero return code: %d" % clone.returncode
        if rev:
            get_command_output('%s svn update --quiet --revision %s %s' % (ya_path, str(rev), working_dir))

        get_command_output('%s make -j0 --checkout %s' % (
            ya_path,
            os.path.join(working_dir, APPS_CONFIG[app]['primary-svn-dir'])
        ))
    except Exception as e:
        remove_working_copy_svn(working_dir)
        raise

    return working_dir


def remove_working_copy_svn(working_dir):
    """
    Просто удаляем svn-каталог
    """
    shutil.rmtree(working_dir)


def checkout_working_copy_arc(rev=None):
    """
    Создаем рабочую копию с помощью arc
    """
    tmp_dir = tempfile.mkdtemp(dir='/tmp/temp-ttl/ttl_1d')

    arcadia_dir = os.path.join(tmp_dir, 'arcadia')
    store_dir = os.path.join(tmp_dir, 'store')
    os.makedirs(arcadia_dir)
    os.makedirs(store_dir)

    get_command_output('arc mount -m %s -S %s' % (arcadia_dir, store_dir))
    if rev:
        old_wd = os.getcwd()
        os.chdir(arcadia_dir)
        get_command_output('arc checkout r%s' % str(rev))
        os.chdir(old_wd)
    return arcadia_dir


def remove_working_copy_arc(working_dir):
    """
    Отмаунчиваем arc и удаляем используемый каталог
    Делаем ретраи для удаления, так как arc не всегда успевает отпустить все файлы
    """
    n = 30
    unmounted = False
    for i in xrange(n):
        try:
            if not unmounted:
                get_command_output('arc unmount %s' % working_dir)
                unmounted = True
            shutil.rmtree(os.path.dirname(working_dir))
            break
        except:
            # если так и не получилось удалить, то не продолжаем выполнение
            if i + 1 == n:
                raise
            time.sleep(1)


def checkout_working_copy(app, vcs, rev=None):
    if vcs == 'arc':
        return checkout_working_copy_arc(rev=rev)
    elif vcs == 'svn':
        return checkout_working_copy_svn(app, rev=rev)
    else:
        raise ValueError("unknown vcs: '%s'" % vcs)


def remove_working_copy(working_dir, vcs):
    if vcs == 'arc':
        return remove_working_copy_arc(working_dir)
    elif vcs == 'svn':
        return remove_working_copy_svn(working_dir)
    else:
        raise ValueError("unknown vcs: '%s'" % vcs)


def get_app_dependencies_with_checkout(app, vcs='arc', rev=None):
    working_dir = checkout_working_copy(app, vcs, rev=rev)
    try:
        dependencies = get_app_dependencies(app, working_dir)
    finally:
        remove_working_copy(working_dir, vcs)

    return dependencies


def get_svnlog_with_deps(app, l_rev, r_rev, working_copy=None, dependencies=None, vcs='arc'):
    """
    Получить для заданного приложения svnlog с ревизии l_rev по r_rev
    Можно передать уже полученные зависимости
    Можно передать путь до рабочей копии аркадии, если нет, то чекаутится заданное приложение
    """
    if not dependencies:
        if not working_copy:
            dependencies = get_app_dependencies_with_checkout(app, vcs)
        else:
            working_dir = os.path.expanduser(working_copy)
            if not os.path.exists(working_dir):
                sys.exit("Path to working copy '%s' doesn't exist\n" % working_dir)

            dependencies = get_app_dependencies(app, working_dir)

    svnlog = read_svnlog_from_xml(
        get_command_output('svn log -v -r %s:%s --xml svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/direct' % (
            str(l_rev), str(r_rev)
        ))
    )

    svnlog = [rev for rev in svnlog if is_commit_in_app(rev, dependencies)]
    return svnlog


def get_svnlog_from_branch(app, base_rev, l_rev, r_rev):
    """
    Получаем svnlog хотфикса или слайда релиза
    """
    if app == 'direct':
        merge_source_url = 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/direct/perl'
        branch_url = 'svn+ssh://arcadia.yandex.ru/arc/branches/direct/release/perl/release-%s' % (base_rev)
    else:
        merge_source_url = 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia'
        branch_url = 'svn+ssh://arcadia.yandex.ru/arc/branches/direct/release/%s/%s/arcadia' % (app, base_rev)

    hotfix_revisions_output = subprocess.check_output([
        'svn', 'mergeinfo', '-r', 'r%s:r%s' % (l_rev, r_rev), '--show-revs', 'merged',
        merge_source_url,
        branch_url
    ])

    hotfix_revisions = [rev for rev in hotfix_revisions_output.split('\n') if rev]
    if not hotfix_revisions:
        return []

    hotfix_log_xml = subprocess.check_output(
        ['svn', 'log', '--xml'] +
        [opt for rev in hotfix_revisions for opt in ['-r', rev.replace("*", "")]] +
        ['svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia']
    )
    svnlog = read_svnlog_from_xml(hotfix_log_xml)
    return svnlog


def prettify_log_row(rev):
    """
    Формируем changelog
    """
    result = 'r%s | %s | %s' % (
        rev.attrib['revision'],
        rev.find('author').text,
        dateutil.parser.parse(rev.find('date').text).strftime('%Y-%m-%d %H:%M:%S')
    )
    msg = rev.find('msg')
    msg = (msg.text.encode('utf-8')
           if msg is not None and msg.text is not None
           else '')
    cnt = msg.count('\n') + 1

    result += ' | %d line%s\n%s' % (cnt, 's' if cnt > 1 else '', msg)
    return result


def prettify_log(svnlog):
    return "\n\n".join([prettify_log_row(rev) for rev in svnlog])


def get_revisions_with_tickets(svnlog):
    """
    Достаем из svnlog пары (revision, ticket)
    """
    result = []
    for commit in svnlog:
        rev = commit.attrib['revision']
        ticket = ''

        msg = commit.find('msg')
        if msg is not None and msg.text is not None:
            msg = msg.text.encode('utf-8')
            ticket = re.search(ur'^(DIRECT-[0-9]+)', msg, re.M | re.U)
            if ticket:
                result.append((rev, ticket.group(1)))

    return result


def read_svnlog_from_xml(xml):
    return ET.fromstring(xml)

