use strict;
use warnings;

use FindBin qw/$Bin/;
use Test::More;
use Test::Exception;
use YAML;

use lib "$Bin/lib";
use DirectReleaseTesting;

my $dir = prepare_tmp_dir("app1");

my @tests = (
   {
        title => '"testing-done" 3x and look at overview ==> 5 releases',
        cmds => [
            {cmd => 'create-release',
             stage => 'testing'},
            {cmd => 'testing-done',
             stage => 'testing'},
            {cmd => 'open-release D-712',
             stage => 'testing'},
            {cmd => 'testing-done',
             stage => 'testing'},
            {cmd => 'open-release D-1312',
             stage => 'testing'},
            {cmd => 'testing-done',
             stage => 'testing'},
            {cmd => 'overview',
             stage => '',
             check => sub {
                 my $output = shift;
                 my $number = () = ($output =~ m!^app: app1!gm);
                 return $number == 5;
             }
            },
        ],
    },
    {
        title => '"testing-done", then "next-waiting" ==> 1 finished release in testing and 1 in stable',
        cmds => [
            {cmd => 'create-release',
             stage => 'testing'},
            {cmd => 'testing-done',
             stage => 'testing'},
            {cmd => 'finish',
             stage => 'stable'},
            {cmd => 'next-waiting',
             stage => 'stable'},
            {cmd => 'overview',
             stage => '',
             check => sub { 
                 my $output = shift;
                 my $number = () = ($output =~ m!^app: app1!gm);
                 return $number == 2 && $output =~ m!^app: app1, stage: testing, FINISHED!gm && $output =~ m!^app: app1, stage: stable!gm;
             }
            },
        ],
    },
    {
        title => '"testing-done" 3x, "next-waiting" 3x ==> empty queue',
        cmds => [
            {cmd => 'create-release',
             stage => 'testing'},
            {cmd => 'testing-done',
             stage => 'testing'},
            {cmd => 'open-release D-712',
             stage => 'testing'},
            {cmd => 'testing-done',
             stage => 'testing'},
            {cmd => 'open-release D-1312',
             stage => 'testing'},
            {cmd => 'testing-done',
             stage => 'testing'},
            {cmd => 'finish',
             stage => 'stable'},
            {cmd => 'next-waiting',
             stage => 'stable'},
            {cmd => 'finish',
             stage => 'stable'},
            {cmd => 'next-waiting',
             stage => 'stable'},
            {cmd => 'finish',
             stage => 'stable'},
            {cmd => 'next-waiting',
             stage => 'stable'},
            {cmd => 'overview',
             stage => '',
             check => sub {
                 my $output = shift;
                 my $number = () = ($output =~ m!^app: app1!gm);
                 return $number == 2 && $output =~ m!^app: app1, stage: testing, FINISHED!m && $output =~ m!^app: app1, stage: stable!m;
             }
            },
        ],
    },
    {
        title => '"testing-done" 3x, "next-waiting" 4x ==> exception',
        cmds => [
            {cmd => 'create-release',
             stage => 'testing'},
            {cmd => 'testing-done',
             stage => 'testing'},
            {cmd => 'open-release D-712',
             stage => 'testing'},
            {cmd => 'testing-done',
             stage => 'testing'},
            {cmd => 'open-release D-1312',
             stage => 'testing'},
            {cmd => 'testing-done',
             stage => 'testing'},
            {cmd => 'finish',
             stage => 'stable'},
            {cmd => 'next-waiting',
             stage => 'stable'},
            {cmd => 'finish',
             stage => 'stable'},
            {cmd => 'next-waiting',
             stage => 'stable'},
            {cmd => 'finish',
             stage => 'stable'},
            {cmd => 'next-waiting',
             stage => 'stable'},
            {cmd => 'finish',
             stage => 'stable'},
            {cmd => 'next-waiting',
             stage => 'stable',
             error => 1},
        ],
    },
    {
        title => '"open release" -> "rollback" -> "next-waiting"',
        cmds => [
            {cmd => 'open-release D-712',
             stage => 'stable'},
            {cmd => 'rollback D-1312',
             stage => 'stable'},
            {cmd => 'overview',
             stage => '',
             check => sub {
                 my $output = shift;
                 diag($output);
                 my $number = () = ($output =~ m!^app: app1!gm);
                 return $number == 3 && 
                        $output =~ m!^app: app1, stage: waiting-1!m &&
                        $output =~ m!^app: app1, stage: stable!m &&
                        $output =~ m!^app: app1, stage: testing!m &&
                        $output =~ m!D-712! && $output =~ m!D-1312!;
             }
            },
            {cmd => 'rollback D-1312',
             stage => 'stable'},
            {cmd => 'next-waiting',
             stage => 'stable'},
            {cmd => 'next-waiting',
             stage => 'stable'},
            {cmd => 'overview',
             stage => '',
             check => sub {
                 my $output = shift;
                 my $number = () = ($output =~ m!^app: app1!gm);
                 return $number == 2 && $output =~ m!^app: app1, stage: stable!gm && $output =~ m!^app: app1, stage: testing!m;;
             }
            },
        ],
    },
    {
        title => 'same "next-waiting" 2x ==> exception',
        cmds => [
            {cmd => 'open-release D-712',
             stage => 'testing'},
            {cmd => 'testing-done',
             stage => 'testing'},
            {cmd => 'testing-done',
             stage => 'testing',
             error => 1},
            {cmd => 'open-release D-712',
             stage => 'testing'},
            {cmd => 'testing-done',
             stage => 'testing',
             error => 1},
        ],
    },
    {
        title => 'changelog must be done before track',
        cmds => [
            {cmd => 'open-release D-712',
             stage => 'testing'},
            {cmd => 'create-sandbox-task',
             stage => 'testing'},
            {cmd => 'get-changelog',
             stage => 'testing'},
            {cmd => 'wait-packages',
             stage => 'testing'},
            {cmd => 'track',
             stage => 'testing',
             error => 1},
            {cmd => 'wait-changelog',
             stage => 'testing'},
            {cmd => 'track',
             stage => 'testing'},
        ],
    },
);

# для команды ensure-screen-tmux
$ENV{'TMUX'} = "for_testing";

for my $tc (@tests) {
    clear_state($dir, "app1");

    my $title = $tc->{title};
    $title ||= join(", ", @{$tc->{actions}});
    for my $cmd (@{$tc->{cmds}}) {
        my $stage = "";
        if ($cmd->{stage}) {
            $stage = "-s $cmd->{stage}";
        }
        my $cmd_to_run = "$Bin/../bin/direct-release $stage -a app1 -t $dir/test.conf.yaml $cmd->{cmd}";
        my $output = `$cmd_to_run`;
        my $check_result = 1;
        if ($cmd->{check}) {
            $check_result = $cmd->{check}->($output);
        }
        diag("$output");
        ok(($cmd->{error} ? $? != 0 : $? == 0) && $check_result, "$title: $cmd->{cmd}");
   }
}

done_testing();
