#!/usr/bin/perl

use strict;
use warnings;

=head1 NAME

=encoding UTF-8

    check-mysql-int-types.pl

=head1 DESCRIPTION

    Проверка на то,  что все числовые поля в базе достаточно далеки от 
    своего максимального значения.
    Близкие к максимуму столбцы нужно поальтерить, иначе переполнение и беда-беда-беда.
    (столбцы с хэшами обычно альтерить не надо)

    В качестве агрументов скрипты передаются имена баз, скрипт
    - получает список таблиц,
    - для каждой таблицы получает список целочисленных столбцов
    - делает запросы вида
        SELECT max(int_field1), max(int_field2), ... FROM table
    - сравниваются полученные значения и максимально допустимое значение типа, результат выводится в таблицу

    Запросы в худшем случае делают full-scan, поэтому лучше запускать на наименьшем репрезентативном шарде

    Для ускорения можно передать параметр --sample-size N - просматривать только N последний строк в каждой таблицу
    (ориентируясь по первичному ключу)

=head1 EXAMPLES

    check-mysql-int-types.pl --host=ppctest-ts1-mysql.ppc.yandex.ru --port=3348 --db ppc -u ... -p ... --sample 10000

=head1 TODO

    - автосемплирование для больших таблиц
    - проверка отдельных таблиц/полей 
    + прокидывать --sample через dbs

=cut

use Getopt::Long qw(:config no_ignore_case);
use Math::BigInt;

use Yandex::DBTools;

$|++;

our $SAMPLE_SIZE;

run() unless caller();

sub run
{
    my %O = %{parse_options()};
    $SAMPLE_SIZE = $O{sample_size};

    my $db = $O{mysql_database};

    $Yandex::DBTools::DONT_SEND_LETTERS = 1;
    %Yandex::DBTools::DB_CONFIG = (
        CHILDS => {
            $db => {
                'AutoCommit' => '1',
                'CHILDS' => {
                    '_' => {},
                },
                'connect_timeout' => '4',
                'host' => $O{mysql_host},
                'pass' => $O{mysql_password},
                'port' => $O{mysql_port},
                'user' => $O{mysql_user},
                'utf8' => '1',
                'db'   => $O{mysql_database},
            },
        },
    );

    check_db($db, $O{exclude} // {});
}

sub parse_options
{
    my %O = (
    );

    GetOptions(
        "h|help" => sub {
            system("podselect -section NAME -section DESCRIPTION -section OPTIONS -section EXAMPLES $0 | pod2text"); 
            exit 0;
        },
        "H|host=s" => \$O{mysql_host},
        "P|port=s" => \$O{mysql_port},
        "u|user=s" => \$O{mysql_user},
        "p|password=s" => \$O{mysql_password},
        "db|database=s" => \$O{mysql_database},
        'sample-size=i' => \$O{sample_size},
        'exclude-file=s' => \$O{exclude_file},
    ) || die "can't parse options, stop";

    if ( $O{exclude_file} ){
        open(my $fh, '<', $O{exclude_file}) or die "can't open file '$O{exclude_file}': $!; stop\n";
        # строки, начинающиеся с решетки пропускаем -- это комментарии
        # остальное склеиваем и делим по пробельным символам
        $O{exclude} = { map { $_ => 1 } split /\s+/, join " ", grep { !/^\s*#/ } <$fh> };
        delete $O{exclude_file};
    }

    return \%O;
}


=head1 check_db($db)

    проверить базу

=cut
sub check_db {
    my ($db, $exclude) = @_;
    my @tables = @{get_one_column_sql($db, "show tables")};
    for my $table (@tables) {
        check_table($db, $table, $exclude);
    }
}


=head2 check_table($db, $tbl)
    
    проверить таблицу

=cut
sub check_table {
    my ($db, $table, $exclude) = @_;
    my @cols = 
        @{get_all_sql($db, "
                    SELECT column_name, column_type
                      FROM information_schema.columns
                     WHERE table_name = ?
                       AND table_schema = database()
                    ", $table)};
    $_->{type_info} = parse_column_type($_->{column_type}) for @cols;
    my @int_cols = grep {$_->{type_info}{is_int}} @cols;

    if (!@int_cols) {
        return {int_cols => []};
    }

    my $from = $table;
    if ($SAMPLE_SIZE) {
        my $pk_prefix = get_one_field_sql($db, "
                                SELECT column_name
                                  FROM information_schema.key_column_usage
                                 WHERE table_name = ?
                                   AND table_schema = database()
                                   AND constraint_name = 'PRIMARY'
                                   AND ordinal_position = 1
                                ", $table);
        if ($pk_prefix) {
            $from = "(SELECT ".( join", ", map {"`$_->{column_name}` as `$_->{column_name}`"} @int_cols )." FROM $table ORDER BY `$pk_prefix` DESC LIMIT $SAMPLE_SIZE) as t";
        }
    }
    my $SQL = "SELECT ".join(", ", map {"max(`$_->{column_name}`) as `$_->{column_name}`"} @int_cols).
        " FROM $from";
    my $max_vals = get_one_line_sql($db, $SQL);

    for my $int_col (@int_cols) {
        my $full_name = "$db.$table.$int_col->{column_name}";
        next if $exclude->{$full_name};
        my $val = $max_vals->{$int_col->{column_name}} // 0;
        my $max_val = $int_col->{type_info}{int_max_value};
        print join " ", (
            sprintf("%-67s", "$full_name"),
            sprintf("%-20s", $int_col->{column_type} =~ s/\s/_/gr), 
            sprintf("%-8s", int(10_000 * $val / $max_val)/100 . " %"),
            "$val / $max_val", 
        );
        print "\n";
        #printf "%s\t%0.2f%%\t(%s of %s)\n", "$db.$table.$int_col->{column_name}", 100*($val//0)/$max_val, $val//'null', $max_val;
    }
}


=head2 parse_column_type($type)

    попрарсить mysql-ный тип столбца, вернуть хэш со знаниями о типе

=cut
sub parse_column_type {
    my ($type) = @_;
    my $ret = {
        type => $type,
    };
    if ($type =~ /^(tiny|small||big)int/i) {
        $ret->{is_int} = 1;
        $ret->{int_modifier} = $1 // '';
        $ret->{is_int_unsigned} = $type =~ /unsigned/i ? 1 : 0;
        $ret->{size} = {tiny => 1, small => 2, '' => 4, big => 8}->{$ret->{int_modifier}};
        $ret->{int_max_value} = Math::BigInt->new(2)->bpow($ret->{size} * 8 - ($ret->{is_int_unsigned} ? 0 : 1))->bsub(1)->numify;
    }
    return $ret;
}
