#!/usr/bin/perl

use strict;
use warnings;

=head1 NAME

=encoding UTF-8

    direct-analyze-alter.pl

=head1 DESCRIPTION

    Получает контакты БД и запрос "alter table". Показывает, что альтер поменяет в своей таблице: типы, знаковость, дефолты, not null и т.д.

    Ранний прототип, на настоящих таблицах и альтерах может ломаться. 

    cat alter_campaigns_1.sql | ./bin/direct-analyze-alter.pl -H ppctest-mysql01k.yandex.ru --port=3306 --user=adiuser --db=ppc -p <...> -

=head1 TODO 

  + интеграция с dbs
  + юнит-тесты на разборщик
  * юнит-тесты на my_diff

=cut

use Getopt::Long qw/:config no_ignore_case/;
use YAML;
use Carp::Always;

use Yandex::DBTools;

use Direct::Homemade::SqlParser;

use utf8;
use open ':std' => ':utf8';

my $NOT_SET = '~~~ NOT SET ~~~';

run() unless caller();

sub run
{
    my %O = %{parse_options()};

    my $db = $O{mysql_database};

    $Yandex::DBTools::DONT_SEND_LETTERS = 1;
    %Yandex::DBTools::DB_CONFIG = (
        CHILDS => {
            $db => {
                'AutoCommit' => '1',
                'CHILDS' => {
                    '_' => {},
                },
                'connect_timeout' => '4',
                'host' => $O{mysql_host},
                'pass' => $O{mysql_password},
                'port' => $O{mysql_port},
                'user' => $O{mysql_user},
                'utf8' => '1',
                'db'   => $O{mysql_database},
            },
        },
    );

    # предварительно заглядываем в альтер и находим имя таблицы, чтобы взять ее create table из БД
    $O{query} =~ /^\s*alter\s*table\s*`?([a-z0-9_\.]*)/i;
    my $table = $1;

    $table !~ /[;-]/ or die;
    my $tables = get_one_column_sql($db, "show tables like '$table'");
    if ( @$tables < 1 ){
        die "can't find table '$table', stop\n";
    }
    my $create_table_sql = get_one_line_sql($db, "show create table $table")->{"Create Table"};

    analyze_alter(create_table_sql => $create_table_sql, alter_table_sql => $O{query});

}


sub analyze_alter
{
    my %O = @_;

    my $alter_ast = parse_alter_table_statement( $O{alter_table_sql} );
    my $create_ast = parse_create_table_statement($O{create_table_sql});

    if ( $alter_ast->{table} ne $create_ast->{table} ){
        die "different tables: '$alter_ast->{table}' ne '$create_ast->{table}'";
    }

    analyze_column_modifications( alter_ast => $alter_ast, create_ast => $create_ast );
    analyze_column_additions( alter_ast => $alter_ast, create_ast => $create_ast );
    analyze_index_modifications( alter_ast => $alter_ast, create_ast => $create_ast );
    analyze_index_additions( alter_ast => $alter_ast, create_ast => $create_ast );
    analyze_unknowns( alter_ast => $alter_ast, create_ast => $create_ast );

    #print YAML::Dump($alter_ast, $create_ast);
}


# сокращенная копия analyze_column_modifications
sub analyze_column_additions
{ 
    my %O = @_;

    print "### Column additions\n";
    my $cnt = 0;
    for my $spec( @{$O{alter_ast}->{alter_specifications}} ){
        next unless $spec->{spec_type} =~ /^add_column$/;
        $cnt++;

        my $column = $spec->{column};
        print "\n#### $column\n";
        my @current_definitions = grep { $_->{column} &&  $_->{column} eq $column} @{$O{create_ast}->{create_definitions}}; 
        if (@current_definitions > 0){
            print "Error: column '$column' already present in orig table\n";
            next;
        } 

        my $new_def = $spec->{new_column_definition};

        # TODO можно сделать отключение ворнингов
        if ( 1 ){
            for my $msg ( @{$new_def->{warnings} || []} ){
                print "Warning: $msg\n";
            }
        }

        for my $d ( ($new_def) ){
            next unless exists $d->{generated_definition};
            $d->{Generated_def_tmp_str} = join " ", @{$d->{generated_definition}};
        }

        for my $f (qw/type length unsigned values not_null default on_update Generated_def_tmp_str/){
            if (!exists $new_def->{$f}){
                # атрибута у новой колонки нет
                next;
            }
            $new_def->{$f} = $NOT_SET unless exists $new_def->{$f};
            if ( $f eq 'values' ){
                print "* values:\n";
                # для enum-ов и set-ов выводим построчечный дифф
                my $diff = my_diff([], $new_def->{values});
                my_format_diff($diff, "   ");
            } elsif ( $f eq 'Generated_def_tmp_str' ){
                print "* generated:\n   + $new_def->{$f}\n";
            } else {
                print "* $f:\n   + $new_def->{$f}\n";
            }
        }
    }
    print "total: $cnt additions\n\n";

    return;
}


sub analyze_column_modifications
{ 
    my %O = @_;

    print "### Column modifications\n";
    my $cnt = 0;
    for my $spec( @{$O{alter_ast}->{alter_specifications}} ){
        # TODO обрабатывать change column
        next unless $spec->{spec_type} =~ /^modify_column$/;
        $cnt++;

        my $column = $spec->{column};
        if ($spec->{column_new_name}){
            print "\n#### $column ==(RENAME)==> $spec->{column_new_name}\n";
        } else {
            print "\n#### $column\n";
        }
        my @current_definitions = grep { $_->{column} &&  $_->{column} eq $column} @{$O{create_ast}->{create_definitions}}; 
        if (@current_definitions >= 2){
            die "multiple column '$column' in orig table, stop";
        } elsif ( @current_definitions < 1 ){
            print "Error: no column '$column' found in orig table\n";
            next;
        } 
        my $old_def = $current_definitions[0]->{column_definition};
        my $new_def = $spec->{new_column_definition};

        # TODO можно сделать отключение ворнингов
        if ( 1 ){
            for my $msg ( @{$new_def->{warnings} || []} ){
                print "Warning: $msg\n";
            }
        }

        for my $d ( ($old_def, $new_def) ){
            next unless exists $d->{values};
            $d->{Values_tmp_str} = join ", ", @{$d->{values}};
        }
        
        for my $d ( ($old_def, $new_def) ){
            next unless exists $d->{generated_definition};
            $d->{Generated_def_tmp_str} = join " ", @{$d->{generated_definition}};
        }
        
        for my $f (qw/type length unsigned Values_tmp_str not_null default on_update auto_increment Generated_def_tmp_str /){
            if (!exists $old_def->{$f} && !exists $new_def->{$f}){
                # атрибута у колонки нет и он не меняется
                next;
            }
            $new_def->{$f} = $NOT_SET unless exists $new_def->{$f};
            $old_def->{$f} = $NOT_SET unless exists $old_def->{$f};
            if ( $new_def->{$f} eq $old_def->{$f} ){
                if ( $f eq 'Values_tmp_str' ){
                    print "  values: $old_def->{$f}\n";
                } elsif ( $f eq 'Generated_def_tmp_str' ){
                    print "  generated: $old_def->{$f}\n";
                } else {
                    print "  $f: $old_def->{$f}\n";
                }
            } else {
                if ( $f eq 'Values_tmp_str' ){
                    print "* values:\n";
                    # для enum-ов и set-ов выводим построчечный дифф
                    my $diff = my_diff($old_def->{values}, $new_def->{values});
                    my_format_diff($diff, "   ");
                } elsif ( $f eq 'Generated_def_tmp_str' ){
                    print "* generated:\n   - $old_def->{$f}\n   + $new_def->{$f}\n";
                } else {
                    print "* $f:\n   - $old_def->{$f}\n   + $new_def->{$f}\n";
                }
            }
        }
        delete $old_def->{Values_tmp_str};
        delete $new_def->{Values_tmp_str};
    }
    print "total: $cnt modifications\n\n";

    return;
}


sub analyze_index_additions
{ 
    my %O = @_;

    print "### Index additions\n";
    my $cnt = 0;
    for my $spec( @{$O{alter_ast}->{alter_specifications}} ){
        next unless $spec->{spec_type} =~ /^add_index$/;
        $cnt++;

        my @title_parts = ();
        push @title_parts, 'New'; 
        push @title_parts, 'UNIQUE' if $spec->{unique}; 
        push @title_parts, 'PRIMARY' if $spec->{primary}; 
        push @title_parts, 'index'; 

        print "\n#### ".join(' ', @title_parts)."\n";
        print "* columns: ".join(", ", @{$spec->{index_columns}||[]})."\n";

    }
    print "\ntotal: $cnt additions\n\n";

    return;
}


sub analyze_index_modifications
{ 
    my %O = @_;

    print "### Index modifications\n";
    print "Not supported yet\n\n";

    return;
}


sub analyze_unknowns
{
    my %O = @_;

    print "### Unknowns\n";
    my $cnt = 0;
    for my $spec( @{$O{alter_ast}->{alter_specifications}} ){
        next unless $spec->{spec_type} =~ /^unknown$/;
        $cnt++;

        print "\n#### Unknown #$cnt\n";
        print "* text: $spec->{text}\n";

    }
    print "\ntotal: $cnt unknowns\n\n";

    return;
}



sub parse_options
{
    my %O = (
    );

    GetOptions(
        "h|help" => sub {
            system("podselect -section NAME -section DESCRIPTION -section OPTIONS -section EXAMPLES $0 | pod2text"); 
            exit 0;
        },
        "H|host=s" => \$O{mysql_host},
        "P|port=s" => \$O{mysql_port},
        "u|user=s" => \$O{mysql_user},
        "p|password=s" => \$O{mysql_password},
        "db|database=s" => \$O{mysql_database},
        'q|query=s'     => \$O{query},
    ) || die "can't parse options, stop";

    if( @ARGV && !$O{query} ){
        $O{query} = shift @ARGV;
    }

    if ( $O{query} eq '-' ){
        $O{query} = join "\n", <STDIN>;
    }

    return \%O;
}




=head2 my_lcs

    Получает две ссылки на массивы строк, возвращает максимальную общую подпоследовательность 
    (надо для my_diff)

=cut
{
# кеш промежуточных результатов. Не state -- вдруг захотим запускать на старых perl-ах
my $cache;

sub my_lcs
{
    my ($s1, $s2) = @_;

    my $l1 = scalar(@$s1);
    my $l2 = scalar(@$s2);

    $cache = [];
    my $lcs = _my_lcs($s1, $l1, $s2, $l2);

    return $lcs;
}

sub _my_lcs
{
    my ($s1, $l1, $s2, $l2) = @_;

    if ( defined $cache->[$l1]->[$l2] ){
        # pass
    } elsif ( $l1 <= 0 || $l2 <= 0 ){
        $cache->[$l1]->[$l2] = [];
    } elsif ( $s1->[$l1-1] eq $s2->[$l2-1] ){
        my $sub_lcs = _my_lcs($s1, $l1 - 1, $s2, $l2 - 1);
        $cache->[$l1]->[$l2] = [ @$sub_lcs, $s1->[$l1-1] ];
    } else {
        my $sub_lcs_1 = _my_lcs($s1, $l1 - 1, $s2, $l2);
        my $sub_lcs_2 = _my_lcs($s1, $l1, $s2, $l2 - 1);
        if ( @$sub_lcs_1 >= @$sub_lcs_2  ){
            $cache->[$l1]->[$l2] = [ @$sub_lcs_1 ];
        } else {
            $cache->[$l1]->[$l2] = [ @$sub_lcs_2 ];
        }
    }

    return $cache->[$l1]->[$l2];
}
}

=head2 my_diff

    Получает две ссылки на массив строк, возвращает их дифф.
    Дифф -- ссылка на массив статусов строк. 
    Статус строки -- ссылка на массив из двух элементов: первый -- 'common', '1' или '2', второй -- собственно строка.

    Для исходных массивов ['aaa', 'bbb', 'ccc'], ['aaa', 'ccc', 'ddd] вернет 
    [['common', 'aaa'], 
    ['1', 'bbb'],
    ['common', 'ccc'],
    ['2', 'ddd']]

=cut
sub my_diff
{
    my ($s1, $s2) = @_; 
    my $lcs_ref = my_lcs($s1, $s2);
    my @arr1 = @$s1;
    my @arr2 = @$s2;
    my @lcs  = @$lcs_ref;

    my $diff = []; 

    while( @arr1 > 0 || @arr2 > 0 ){
        print "";
        if ( @arr1 > 0 && @arr2 > 0 && @lcs > 0 && $arr1[0] eq $lcs[0] && $arr2[0] eq $lcs[0] ){
            push @$diff, ['common', $lcs[0]];
            shift @arr1;
            shift @arr2;
            shift @lcs;
            next;
        } elsif ( @arr1 > 0 && ( @lcs <= 0 || $arr1[0] ne $lcs[0]) ){
            push @$diff, ['1', $arr1[0]];
            shift @arr1;
            next;
        } elsif ( @arr2 > 0 && ( @lcs <= 0 || $arr2[0] ne $lcs[0]) ){
            push @$diff, ['2', $arr2[0]];
            shift @arr2;
            next
        } else {
            die;
        }
    }

    return $diff;
}

=head2 my_format_diff
    
    красиво печатает дифф, возвращенный my_diff

=cut
sub my_format_diff
{
    my ($diff, $indent) = @_;
    $indent ||= '';

    for my $h ( @$diff ){
        if ($h->[0] eq 'common' ){
            print "$indent  $h->[1]\n";
        } elsif ( $h->[0] eq '1' ){
            print "$indent- $h->[1]\n";
        } elsif ( $h->[0] eq '2' ){
            print "$indent+ $h->[1]\n";
        } else {
            die "unrecognizable diff, '$h->[0]'";
        }
    }
}

1;
