#!/usr/bin/perl

=head1 NAME

    tag-db-columns.pl -- в списке колонок отметить исключения из теста db_schema/unsigned_ids.t и, возможно, ещё какие-то колонки

=head1 SYNOPSIS

    tag-db-columns.pl pr:ppc:1
    tag-db-columns.pl dt:ppcdict
    tag-db-columns.pl pr:ppc:1 -a mytag=cols.txt

    # подсветить протеганное
    tag-db-columns.pl pr:ppc:1 -a mytag=cols.txt | perl -MTerm::ANSIColor -lane 'print((grep {/^#/} @F) ? colored($_, "red") : $_)'
    # или даже так
    tag-db-columns.pl pr:ppc:1 -a mytag=cols.txt | grep -E '.*#.*|$' --color    
    

=head1 DESCRIPTION

    Вывести список колонок всех таблиц, возможно с тегами.
    По умолчанию проставляются теги для исключений из теста unit_tests/db_schema/unsigned_ids.t:
        - ut:unsigned -- поле типа bigint unsigned
        - ut:signed -- поле типа int signed
        - ut:fk -- поле, которые логически связано с полем в другой таблице, но имеет другой тип (по смыслу -- внешний ключ, не отражённый в mysql; определяется наличием записи FK(<another_table>.<col>) в $DT/db_schema/<db>/<table>.text)

    Также можно добавить любой тег, определяемый списком колонок в формате <db>.<table>.<column_name> из файла. Формат файла аналогичен $DT/unit_tests/db_schema/unsigned_ids.exceptions.unsigned.data@136651
    Скрипт предназначен для вывода информации об особенностях структуры таблиц в виде, пригодном для обработки стандартными инструментами (grep, awk, sort и т. д.)
    Основное предназначение -- собрать информацию по колонкам, которые нужно поальтерить.

=head1 EXAMPLE

    Например, пусть мы хотим везде поменять размер ClientID.
    Готовим список колонок:
        direct-sql pr:ppc:1 'select table_schema, table_name, column_name from information_schema.columns where column_name = "ClientID"' -B | sed -e 1d -e 's/\s\+/./g' > clientids.txt
    Помечаем их в общем списке:
        tag-db-columns.pl pr:ppc:1 -a alter_clientid=clientids.txt
   
    Примерный фрагмент вывода:
        ppc users uid
        ppc users ClientID #ut:fk #alter_clientid #ut:signed
        ppc users rep_type
        ppc users email
        ppc users valid #ut:signed
        ppc users LastChange
        ppc users FIO

=cut

use strict;
use warnings;

use utf8;

use open ':std', ':encoding(UTF-8)';

use Getopt::Long;

use ProjectSpecific;
use Yandex::DBTools;

run() unless caller();

sub run {
    my $opt = parse_options();
    my $db_conf = shift @ARGV or die "no db specified\n";
    my ($conf_name, $db) = $db_conf =~ /^([^:]+):(.*)$/;
    $Yandex::DBTools::DONT_SEND_LETTERS = 1;
    $Yandex::DBTools::CONFIG_FILE = ProjectSpecific::get_db_conf_file($conf_name);
    my %cols_for_tag;
    for my $exception_list (qw/unsigned signed fk/) {
        $cols_for_tag{"ut:$exception_list"} = get_exceptions_from_ut($exception_list);
    }
    if ($opt->{file_for_tag}) {
        my %file_for_tag = %{ $opt->{file_for_tag} };
        for my $tag (keys %file_for_tag) {
            $cols_for_tag{$tag} = get_cols_from_file($file_for_tag{$tag});
        }
    }
    my %tags_for_col;
    for my $tag (keys %cols_for_tag) {
        push @{ $tags_for_col{$_} }, $tag for @{ $cols_for_tag{$tag} };
    }
    for my $col (@{ get_cols_from_db($db) }) {
        my $col_full_name = join '.', @{$col}{qw/table_schema table_name column_name/};
        my @tags = @{ $tags_for_col{$col_full_name} || [] };
        print join " ", @{$col}{qw/table_schema table_name column_name/}, (map {"#$_"} @tags);
        print "\n";
    }
}

sub parse_options {
    my %O = (
    );
    GetOptions(
        'a|add-tags-from-file=s%' => \$O{file_for_tag},
        'h|help' => sub { system "podselect -section NAME -section SYNOPSIS -section DESCRIPTION $0 | pod2text-utf8 >&2" ; exit 0 },
    ) || die "can't parse options, stop\n";
    return \%O;
}

sub get_exceptions_from_ut {
    my ($list_name) = @_;
    my $svn_path = "svn+ssh://svn.yandex.ru/direct/trunk/unit_tests/db_schema/unsigned_ids.exceptions.${list_name}.data";
    my $error;
    my @cmd = ('svn', 'cat', $svn_path);
    open(my $fh, '-|', @cmd) or die "couldn't run svn cat " . join(' ', @cmd) . ": $!";
    return get_cols_from_file($fh)
}

sub get_cols_from_file {
    my $fh;
    if (ref $_[0] eq "") {
        open $fh, '<', $_[0] or die "couldn't open $_[0]: $!";
    } elsif (ref $_[0] eq "GLOB") {
        $fh = $_[0];
    }
    my @res;
    ### скопировано из $DT/unit_tests/db_schema/unsigned_ids.t (sub list_from_file)
    while (my $l = <$fh>){
        chomp($l);
        $l =~ s/#.*$//;
        next if $l =~ /^\s*$/;
        my @values = split /\s+/, $l;
        push @res, @values;
    }
    ###
    return \@res;
}

sub get_cols_from_db {
    my ($db) = @_;
    my @cols = @{ get_all_sql($db, "
                        SELECT table_schema, table_name, column_name
                          FROM information_schema.columns
                         WHERE table_schema = database()
                        ") };
    return \@cols;
}

1;
__END__
