use strict;
use warnings;

package Direct::Homemade::SqlParser;

=head1 DESCRIPTION

Простой и очень специализированный парсер запросов 'create table' и 'alter table'.
Пользоваться за пределами direct-analyze-alter.pl с осторожностью, 
за пределами yandex-du-direct-sql -- вообще не надо, обратная совместимость не обещается.

=cut

use base qw/Exporter/;


our @EXPORT = qw/
    parse_alter_table_statement
    parse_create_table_statement
/;


sub parse_alter_table_statement
{
    my ($q) = @_;
    
    $q = normalyze_alter($q);

    my $res;

    $q =~ s/^alter table ([^ ]+) +//i || die "can't parse query (1), stop\n";
    $res->{table} = $1;

    # делим запрос на слова по пробелам; скобки отделены от текста пробелами при нормализации
    my @w = split / +/, $q;

    my $alter_specifications = shift_specifications(\@w);
    $res->{alter_specifications} = [map { parse_alter_specification($_) } @$alter_specifications];

    return $res;
}
        
sub parse_create_table_statement
{
    my ($q) = @_;
    $q = normalyze_create_table($q);
    
    my $res;

    $q =~ s/^create table ([^ ]+) +//i || die "can't parse query (1), stop\n";
    $res->{table} = $1;
    $q =~ s/^ *\( *//;
    $q =~ s/ *\) *([^()]*)$//;
    $res->{table_options} = $1;

    # делим запрос на слова по пробелам; скобки отделены от текста пробелами при нормализации
    my @w = split / +/, $q;

    my $specifications = shift_specifications(\@w);
    $res->{create_definitions} = [map { parse_create_definition($_) } @$specifications];

    return $res;
}

sub normalyze_alter
{
    my ($q) = @_;

    $q =~ s/^\s+//gsm;
    $q =~ s/[;\s]+$//gsm;
    $q =~ s/[`]//gsm;
    $q =~ s/([\(\),])/ $1 /gsm;
    $q =~ s/\s+/ /gsm;

    return $q;
}

sub normalyze_create_table
{
    my ($q) = @_;

    $q =~ s/^\s+//gsm;
    $q =~ s/[;\s]+$//gsm;
    $q =~ s/[`]//gsm;
    $q =~ s/([\(\),])/ $1 /gsm;
    $q =~ s/\s+/ /gsm;

    # Комментарии /**/ выкидываем
    $q =~ s!/\*.*?\*/?!!g;

    return $q;
}

sub parse_alter_specification
{
    my ($ref) = @_;
    my @w = @$ref;
    my $res = {};

    if ( check_prefix(\@w, [qr/^add$/i, qr/^(index|key)$/i]) ){
        shift @w for 1 .. 2;
        $res->{spec_type} = "add_index";
        shift_index_def($res, \@w);
        ensure_eoi(@w);
    } elsif ( check_prefix(\@w, [qr/^add$/i, qr/^unique$/i, qr/^(index|key)$/i]) ) {
        shift @w for 1 .. 3;
        $res->{spec_type} = 'add_index';
        $res->{unique} = 1;
        shift_index_def($res, \@w);
        ensure_eoi(@w);
    } elsif ( check_prefix(\@w, [qr/^add$/i, qr/^primary$/i, qr/^key$/i]) ) {
        shift @w for 1 .. 3;
        $res->{spec_type} = 'add_index';
        $res->{primary} = 1;
        shift_index_def($res, \@w);
        ensure_eoi(@w);
    } elsif ( check_prefix(\@w, [qr/^modify$/i]) ){
        shift @w;
        shift @w if $w[0] =~ /^column$/i;
        $res->{spec_type} = "modify_column";
        $res->{column} = shift @w;
        $res->{new_column_definition} = parse_column_definition(@w);
    } elsif ( check_prefix(\@w, [qr/^change$/i]) ){
        # почти повторяется обработка modify, дополнительно сохраняем новое имя колонки
        shift @w;
        shift @w if $w[0] =~ /^column$/i;
        $res->{spec_type} = "modify_column";
        $res->{column} = shift @w;
        $res->{column_new_name} = shift @w;
        $res->{new_column_definition} = parse_column_definition(@w);
    } elsif ( check_prefix(\@w, [qr/^add$/i]) ){
        shift @w;
        shift @w if $w[0] =~ /^column$/i;
        $res->{spec_type} = "add_column";
        $res->{column} = shift @w;
        $res->{new_column_definition} = parse_column_definition(@w);
    } elsif ( check_prefix(\@w, [qr/^(ALGORITHM=INPLACE|LOCK=NONE)$/i]) ){
        $res->{spec_type} = "pass";
        $res->{text} = join " ", @w;
    } else {
        $res->{spec_type} = "unknown";
        $res->{text} = join " ", @w;
    }

    return $res;
}

sub parse_create_definition
{
    my ($ref) = @_;
    my @w = @$ref;

    my $res = {};

    if ( $w[0] =~ /^(index|key)$/i ) {
        shift @w;
        $res->{type} = 'index';
        shift_index_def($res, \@w);
        ensure_eoi();
    } elsif ( check_prefix(\@w, [qr/^unique$/i, qr/^(index|key)$/i]) ) {
        shift @w for 1 .. 2;
        $res->{type} = 'index';
        $res->{unique} = 1;
        shift_index_def($res, \@w);
        ensure_eoi();
    } elsif ( $w[0] =~ /^primary$/i ){ 
        shift @w for 1 .. 2;
        $res->{type} = 'primary_index';
        $res->{index_columns} = shift_list(\@w);
        ensure_eoi();
    } elsif ( check_prefix(\@w, [qr/constraint/i]) ){
        shift @w;
        $res->{type} = 'constraint';
        $res->{definition} = join " ", splice(@w, 0, scalar(@w));
        ensure_eoi();
    } else {
        $res->{type} = "column";
        $res->{column} = shift @w;
        $res->{column_definition} = parse_column_definition(@w);
    }

    return $res;
}

sub parse_column_definition
{
    my (@w) = @_;

    my $res = {};

    if ( $w[0] =~ /^(tinyint|smallint|int|bigint|float|double|decimal)$/i ){
        $res->{type} = lc shift @w;
        # возможно, указана "длина" поля
        if(check_prefix(\@w, ['('])){
            shift @w;
            my @len;
            while(@w && $w[0] ne ')'){
                push @len, shift @w;
            }
            if (!@w || $w[0] ne ')'){
                die "cant't parse column definition (2), stop";
            }
            shift @w;
            $res->{length} = join "", @len;
        }
        # знаковость и беззнаковость
        if (check_prefix(\@w, [qr/^unsigned$/i])){
            shift @w;
            $res->{unsigned} = 1;
        } else {
            $res->{unsigned} = 0;
        }
    } elsif ( $w[0] =~ /^char|varchar|text|varbinary$/i){
        $res->{type} = lc shift @w;
        # возможно, указана "длина" поля
        if( check_prefix(\@w, ['(']) ){
            my $len = shift_list(\@w);
            if ( @$len != 1 ){
                die "can't parse varchar length, stop\n".join " ", @$len;
            }
            $res->{length} = shift @$len;
        }
        # caracter set
        if ( check_prefix(\@w, [qr/^character$/i, qr/^set$/i]) ){
            shift @w for 1 .. 2;
            $res->{character_set} = shift @w;
        }
        if ( check_prefix(\@w, [qr/^collate$/i,]) ){
            shift @w;
            $res->{collate} = shift @w;
        }

    } elsif ( $w[0] =~ /^timestamp|date|datetime$/i) {
        $res->{type} = lc shift @w;
    } elsif ( $w[0] =~ /^(enum|set)$/i) {
        $res->{type} = lc shift @w;
        $res->{values} = shift_list(\@w);
        my_unquote(@{$res->{values}});
        die "can't parse enum/set values" unless defined $res->{values};
    } elsif ( $w[0] =~ /^(bool|boolean)$/i ) {
        $res->{type} = lc shift @w;
        $res->{warnings} = ['Вместо boolean лучше явно использовать tinyint, в mysql boolean -- алиас для совместимости'];
    } elsif ( $w[0] =~ /^(blob|mediumblob|longblob|json)$/i) {
        $res->{type} = lc shift @w;
    } else {
        die "can't parse type $w[0], stop";
    }

    $res->{not_null} = 'null';
    # not null и default могут идти в разном порядке, разбираем их, пока есть
    while(1){
        if ( check_prefix(\@w, [qr/^not$/i, qr/^null$/i]) ){
            # not null
            shift @w for 1 .. 2;
            $res->{not_null} = 'not null';
        } elsif ( check_prefix(\@w, [qr/^null$/i]) ){
            # null
            shift @w;
            $res->{not_null} = 'null';
        } elsif ( check_prefix(\@w, [qr/^default$/i]) ){
            # default
            shift @w;
            $res->{default} = shift @w;
            my_unquote($res->{default});
        } elsif ( check_prefix(\@w, [qr/^on$/i, qr/^update$/i]) ) {
            # on update
            shift @w for 1 .. 2;
            $res->{on_update} = shift @w;
        } elsif ( check_prefix(\@w, [qr/^auto_increment$/i] ) ){
            shift @w;
            $res->{auto_increment} = 1;
        } else {
            last;
        }
    }
    # После column definition может идти описание позиции, где колонка должна быть
    # формат: [FIRST | AFTER col_name]
    if ( check_prefix(\@w, [qr/^first$/i]) ){
        $res->{position}->{direction} = shift @w;
    }
    if ( check_prefix(\@w, [qr/^after$/i]) ){
        $res->{position}->{direction} = shift @w;
        $res->{position}->{ref_column} = shift @w;
    }
    # Колонка может быть генерируемая
    # синтаксис: https://dev.mysql.com/doc/refman/5.7/en/create-table-generated-columns.html
    if ( check_prefix(\@w, [qr/^generated$/i]) || check_prefix(\@w, [qr/^as$/i]) ){
        $res->{generated_definition} = shift_generated_definition(\@w);;
    }

    my $comment = shift_comment(\@w);
    if (defined $comment){
        $res->{comment} = $comment;
    }

    ensure_eoi(@w);
    return $res;
}


=head2 

=cut
sub check_prefix
{
    my ($arr, $prefix) = @_;

    if ( @$arr < @$prefix ){
        return 0;
    }

    for my $i ( 0 .. scalar(@$prefix) - 1 ){
        my $pattern = $prefix->[$i];
        my $value = $arr->[$i];
        if ( ref $pattern eq '' ){ 
            return 0 if $value ne $pattern;
        } elsif ( ref $pattern eq 'Regexp' ){
            return 0 if $value !~ $pattern;
        } else {
            die "unsupported pattern '$pattern'";
        }
    }
    return 1;
}

=head2 shift_index_def(\%res, \@w_ref)

Добавляет в %res описание индекса из @w_ref.
Например ['i_client_id_to', '(', 'client_id_to', ')']
переносится в %res в виде {name => 'i_client_id_to', index_columns => ['client_id_to']}

Массив @w_ref модифицируется!

=cut
sub shift_index_def {
    my ($res, $w_ref) = @_;

    if ($w_ref->[0] ne '(') {
        $res->{name} = shift @$w_ref;
    }
    $res->{index_columns} = shift_list($w_ref);
    die "can't parse index definition" unless defined $res->{index_columns};
}


=head2 shift_list

Получает ссылку на массив, shift-ит из него все, похожее на список в скобках через запятую, возвращает ссылку на массив значений в списке
Исходный массив модифицируется!

Если в исходный массив не начинается с откр. скобки -- возвращает undef

=cut
sub shift_list
{
    my ($w_ref) = (@_);

    if ( $w_ref->[0] eq '(' ){
        shift @$w_ref;
    } else {
        return undef;
    }

    my @values;
    while( @$w_ref && $w_ref->[0] ne ')' ){
        my $w = shift @$w_ref;
        next if $w eq ',';
        push @values, $w;
    }
    if (!@$w_ref || $w_ref->[0] ne ')'){
        die "cant't parse list (5), stop";
    }
    shift @$w_ref;
    return \@values;
}

sub ensure_eoi
{
    if(@_){
        die "unexpected input, stop\n".(join " ", @_);
    }
}

=head2 shift_specifications

    Получает ссылку на массив слов, shift-ит из него все, что похоже на "список спецификаций/объявлений"
    Возвращает ссылку на массив спецификаций; каждая спецификация -- ссылка на массив слов.
    Исходный массив модифицируется!

=cut
sub shift_specifications
{
    my ($w_ref) = @_;

    my @specifications;
    my @cur_spec; 
    my $st = 'start';
    my $cur_quote;
    my @cur_str;
    # В generated объявлениях бывают сколько угодно вложенные скобки и запятые между ними
    # Чтобы правильно понять, когда объявление закончилось -- поддерживаем стек скобок и сверяем по нему 
    # (классический алгоритм проверки скобочной последовательности)
    my @brackets_stack;
    while ( scalar @$w_ref ){
        my $w = shift @$w_ref;
        if ($st eq 'start' && $w !~ /^\(|,$/ && $w !~ /^['"]/ ){
            push @cur_spec, $w;
        } elsif ( $st eq 'start' && $w =~ /^(['"])/ ){
            $cur_quote = $1;
            if ( $w =~ /$cur_quote$/ ){
                push @cur_spec, $w;
            } else {
                $st = 'quote';
                @cur_str = ($w);
            }
        } elsif ( $st =~ /^(start|bracket)$/ && $w eq '(' ){
            $st = 'bracket';
            push @cur_spec, $w;
            push @brackets_stack, $w;
        } elsif ($st eq 'start' && $w eq ',' ){
            push @specifications, [@cur_spec];
            @cur_spec = ();
        } elsif ( $st eq 'bracket' && $w ne ')' ){
            push @cur_spec, $w;
        } elsif ( $st eq 'bracket' && $w eq ')' ){
            push @cur_spec, $w;
            if ( scalar @brackets_stack <= 0 ){
                die "unmatched '$w'";
            } elsif ( $brackets_stack[-1] ne '(' ) {
                die "'$w' doesn't match $brackets_stack[-1]";
            } else {
                pop @brackets_stack;
            }
            if ( scalar @brackets_stack > 0 ){
                $st = 'bracket';
            } else {
                $st = "start";
            }
        } elsif ( $st eq 'quote' && $w =~ /$cur_quote$/ ){
            $st = "start";
            push @cur_str, $w;
            push @cur_spec, join " ", @cur_str;
        } elsif ( $st eq 'quote' && $w !~ /$cur_quote$/ ){
            push @cur_str, $w;
        } else {
            die "can't parse specifications, stop (st=$st, w=$w)";
        }
    }
    die "unexpected state '$st', stop" unless $st eq 'start';
    if(@cur_spec > 0){
        push @specifications, [@cur_spec];
        @cur_spec = ();
    }

    return \@specifications;
}

sub shift_comment
{
    my ($w_ref) = @_;
    return undef unless @$w_ref && $w_ref->[0] =~ /^comment$/i;
    shift @$w_ref;
    my $comment = shift @$w_ref;
    my_unquote($comment);
    return $comment;
}


# в тексте точно идет определение генерации колонки, шифтим его целиком
sub shift_generated_definition
{
    my ($w_ref) = @_;
    
    my $def = [];
    while ( @$w_ref && $w_ref->[0] !~ /^comment$/i ){
        push @$def, shift @$w_ref;
    }

    return $def;
}


sub my_unquote
{
    for (@_) {
        s/^'(.*)'$/$1/ and next;
        s/^"(.*)"$/$1/ and next;
    }
}


1;
