use strict;
use warnings;

package Direct::Sql;

use YAML;
use Data::Dumper;
use Time::HiRes qw(gettimeofday tv_interval);

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Shell;
use ProjectSpecific qw/get_db_conf_file/;



sub new
{
    my ($class, %O) = @_;

    $Yandex::DBTools::CONFIG_FILE = $class->conf_file($O{name});
    return bless { 
        name => $O{name} || '',
    };
}


sub conf_file
{
    my ($self, $name) = @_;
    return get_db_conf_file($name) || die "can't find conf file for '$name'";
}


sub compile_actions
{
    my ($self, %param) = @_;

    my @shards = ();
    if ( $param{db} =~ /^([^:]+):all(:.*)?$/ ){
        my ($db, $slave) = ($1, $2);
        $slave ||= '';
        @shards = map {"$_$slave"} @{Yandex::DBShards::get_shard_dbnames($db,"", shard => "all")->{dbnames}}
    } else {
        @shards = ($param{db});
    }
    die "empty shard list, stop" unless @shards > 0; 

    my @actions;
    my $skip_column_names = $param{skip_column_names};
    for my $shard (@shards){
        push @actions, {
            shell_cmd => $self->_compile_single_action(
                db => $shard, 
                query => $param{query},
                batch => $param{batch},
                quick => $param{quick},
                mysql_user => $param{mysql_user},
                mysql_password => $param{mysql_password},
                skip_column_names => $skip_column_names,
            ), 
            comment => "$shard",
        };
        $skip_column_names = 1  if $param{batch};
    }

    return \@actions;
}


sub _is_clickhouse($) {
    my ($cfg) = @_;

    return $cfg->{engine} && $cfg->{engine} eq 'clickhouse' ? 1 : 0;
}

sub _clh_args($%) {
    my ($cfg, %param) = @_;

    my $host = ref $cfg->{host} ? $cfg->{host}->[0] : $cfg->{host};
    my $user = $param{mysql_user} || $cfg->{user};
    my $pass = $param{mysql_password} || $cfg->{pass};

    return (
        "--user=$user", 
        (($cfg->{pass}//'') ne '' ? ("--password=$cfg->{pass}") : ()),
        "--host=$host", 
        ($cfg->{port} == 8123 ? "--port=9000" : "--port=$cfg->{port}"),
        "--database=$cfg->{db}",
        "--max_block_size=1024",
       );
}


sub _compile_single_action 
{
    my ($self, %param) = @_;

    my $cfg = get_db_config($param{db});

    if (_is_clickhouse($cfg)) {
        return $self->_compile_single_clh_action(%param);
    }

    # default args
    my $host = ref $cfg->{host} ? $cfg->{host}->[0] : $cfg->{host};
    my $user = $param{mysql_user} || $cfg->{user};
    my $pass = $param{mysql_password} || $cfg->{pass};
    my @mysql_args = (
        ($param{batch}) ? ('--batch') : ('--table'),
        ($param{quick}) ? ('--quick') : (),
        "--default-character-set=utf8",
        "--user=$user",
        "--host=$host", "--port=$cfg->{port}",
        ($param{skip_column_names} ? "--skip-column-names" : ()),
        $cfg->{db},
        "--comments",
    );

    # add user args TODO ??? --mysql-args
    # push @mysql_args, @ARGV;

    my $cmd =  'mysql';

    my $env_cmd = 'env MYSQL_PWD=' . yash_quote($pass) . ' ';

    my $command = "cat <<'EOF' |"
        . $env_cmd . join(" ", $cmd, map {yash_quote($_)} @mysql_args)
        . "\n"
        . $param{query}
        . "\nEOF\n";

    return $command;
}


sub _compile_single_clh_action 
{
    my ($self, %param) = @_;

    my $cfg = get_db_config($param{db});

    my $format = !$param{batch} ? 'PrettyCompactMonoBlock'
        : $param{skip_column_names} ? 'TabSeparated'
        : 'TabSeparatedWithNames';

    my @args = (
        _clh_args($cfg, %param),
        "--format=$format",
        );

    my $cmd =  'clickhouse-client';
    my $command = "cat <<'EOF' |"
        . join(" ", $cmd, map {yash_quote($_)} @args)
        . "\n"
        . $param{query}
        . "\nEOF\n";

    return $command;
}


sub perform_actions
{
    my ($self, $actions, %opt) = @_;

    my $fail_count = 0;
    for my $act ( @$actions ){
        print "\n### $act->{comment}\n" unless $opt{batch};
        print $act->{shell_cmd}."\n" if $opt{verbose};
        my $t0 = [gettimeofday];
        my $exit_code = system $act->{shell_cmd};
        $fail_count++ if $exit_code != 0;
        my $elapsed = tv_interval ( $t0, [gettimeofday]);
        $elapsed = int($elapsed*100)/100;
        print "...done in $elapsed seconds\n" unless $opt{batch};
    }
    return $fail_count;
}


sub print_actions
{
    my ($self, $actions) = @_;

    print "\n### $_->{comment}\n$_->{shell_cmd}\n" for @$actions;
}

# можно переопределить в классах-наследниках, 
# чтобы для разных баз добавлять особые парамтеры типа -A
sub _additional_params_for_mysql_shell
{
    my ($self, $cfg) = @_;
    return $cfg->{db} =~ /^ppc.*log$/ ? ("-A") : ();
}

sub run_shell 
{
    my ($self, %param) = @_;

    my $cfg = get_db_config($param{db});

    if (_is_clickhouse($cfg)) {
        return $self->run_clickhouse_shell(%param);
    } else {
        return $self->run_mysql_shell(%param);
    }
}

sub run_mysql_shell 
{
    my ($self, %param) = @_;

    my $cfg = get_db_config($param{db});

    # default args
    my $host = ref $cfg->{host} ? $cfg->{host}->[0] : $cfg->{host};
    my $prompt = $self->{name}.":$param{db}> ";
    my @mysql_args = (
        "--prompt=$prompt",
        "-t",
        "--default-character-set=utf8",
        "--user=$cfg->{user}",
        "--host=$host", "--port=$cfg->{port}",
        $self->_additional_params_for_mysql_shell($cfg),
        $cfg->{db},
    );

    # add user args TODO ??? --mysql-args
    # push @mysql_args, @ARGV;

    $ENV{MYSQL_PWD} = $cfg->{pass};

    my $cmd =  'mysql';

    my $command = join(" ", $cmd, map {yash_quote($_)} @mysql_args);

    exec $command;
}


sub run_clickhouse_shell 
{
    my ($self, %param) = @_;

    my $cfg = get_db_config($param{db});

    my $cmd =  'clickhouse-client';
    my $command = join(" ", $cmd, map {yash_quote($_)} _clh_args($cfg, %param));

    exec $command;
}


1;
