#!/usr/bin/perl

use strict;
use Config::General qw(ParseConfig);

my $internal_check_name = 'p2p_mon';
my $external_check_name = 'p2p_client'; # check name for external hosts

my $p2p_client = "/usr/bin/yabs_p2p_client -h localhost";
my $pid_file = "/var/run/yabs_p2p/yabs_p2p.pid";

my $problem_hosts_file = "/var/tmp/p2p_mon_problem_hosts.txt"; # file for hosts for which critical or warning was sent to juggler

my $config_file="/home/monitor/agents/etc/p2p_mon.conf";

my $memory_limit = 6144; # default memory limit
my (%ignore_hostlist, %ignore_params) = ();

my %critical_values = ();

if ( -e $config_file ) {
	my %config = ParseConfig($config_file);

	foreach my $param (keys %config) {
		$critical_values{$param} = $config{$param} if ($param =~ /^(MIN_|MAX_)/);
	}

	$memory_limit = $config{"memory_limit"} if (exists $config{"memory_limit"});
	%ignore_hostlist = map { $_ => 1 } split(/,\s*/, $config{"ignore_hostlist"}); # fill ignore hosts hash
	%ignore_params = map { $_ => 1 } split(/,\s*/, $config{"ignore_params"}); # fill ignore params hash
}

my %param_description = (
	NO_SPACE		=> 'no free space',
	LAST_TASK_ADDED		=> 'send was long ago',
	LAST_TASK_DONE		=> 'receive was long ago',
	OLD_GET_TASKS		=> 'old get tasks',
	ONLINE_HOSTS		=> 'few online hosts',
	TOTAL_TASKS_COUNT	=> 'many active tasks',
);

my $hostname = `hostname`;
chomp $hostname;

my $status = 0;
my @description = ();

# send p2p_client=ok for self (at postinst)
if ($ARGV[0] =~ /p2p_client_ok/) {
	print "send to juggler $external_check_name=OK for $hostname\n";
	send_to_juggler($hostname, $external_check_name, 0, "OK");
	exit 0;
}

if (my $p2p_version = installed_version()) { # check if yabs-p2p installed
	if (my $pid = get_pid()) { # check if p2p daemon is running
		# check memory usage
		my $memory = memory_usage($pid);
		if ($memory > $memory_limit) {
			$status = 2;
			push(@description, "memory usage too high (${memory}Mb > ${memory_limit}Mb)");
		}

		# check installed and running version
		my %daemon_state = get_daemon_state();
		if ($p2p_version ne $daemon_state{VERSION}) {
			$status = 2;
			push(@description, "installed ($p2p_version) and running ($daemon_state{VERSION}) versions differ");
		}

		# other critical params from daemon state
		foreach my $param (@{ $daemon_state{PARAMS} }) {
			my ($param, $value) = split(':', $param);
			# beware! looong if condition below! short version: if ( not ignore and ( >max or <min or no defined limits ) ) => critical
			if (not $ignore_params{$param} and (# if param is not in ignorelist
					(exists $critical_values{"MAX_$param"} and $value > $critical_values{"MAX_$param"}) or	# and bigger then MAX threshold
					(exists $critical_values{"MIN_$param"} and $value < $critical_values{"MIN_$param"}) or	# or smaller than MIN threshold
					(not exists $critical_values{"MAX_$param"} and not exists $critical_values{"MIN_$param"})	# or thresholds not defined, but param present => CRITICAL
				)
			   ) {
				$status = 2;
				$param = $param_description{$param} ? $param_description{$param} : $param; # param descrition, if exist
				$value = defined($value) ? "($value)" : ""; # param value (ex. tasks count), if exist
				push(@description, $param.$value);
			}
		}

		# check tasks for hosts
		my %problem_hosts = ();
		if($daemon_state{HOSTS}) {
			foreach my $host (@{ $daemon_state{HOSTS} }) {
				$host =~ s/\((\d+)\)//;
				$problem_hosts{$host} = $1; # tasks count

				my $state = host_state($host);

				# send state to juggler
				send_to_juggler($host, $external_check_name, host_state($host), "$problem_hosts{$host} tasks for me on $hostname");
			}
		}

		if (%problem_hosts) {
			# push problem hosts to local description (for warning)
			push(@description, "old tasks for [".scalar(keys %problem_hosts)."]: ".join(', ', keys %problem_hosts));
			$status = 1 if (not $status); # warning on local host if other checks oks
		}

		# read previous critical hosts from file
		my @problem_hosts_prev = ();
		open(PROBLEM_HOSTS, "< $problem_hosts_file");
		@problem_hosts_prev = <PROBLEM_HOSTS>;
		chomp @problem_hosts_prev;
		close(PROBLEM_HOSTS);

		# send oki for good hosts
		foreach my $host (@problem_hosts_prev) {
			if (not $problem_hosts{$host}) {
				send_to_juggler($host, $external_check_name, 0, "OK");
			}
		}

		# rewrite file with current problem hosts
		open(PROBLEM_HOSTS, "> $problem_hosts_file");
		print PROBLEM_HOSTS join("\n", keys %problem_hosts);
		close(PROBLEM_HOSTS);

		# OK if OK
		push(@description, "OK") unless($status);
	} else {
		$status = 2;
		push(@description, 'not running');
	}
} else {
	$status = 1;
	push(@description, 'not installed');
}

print "PASSIVE-CHECK:$internal_check_name;$status;".join(', ', sort @description)."\n";
exit $status;

# subchecks
#
sub installed_version {
	my $version = `dpkg -l yabs-p2p-daemon | grep '^ii  yabs-p2p-daemon' | awk '{print \$3}'`;
	chomp $version;
	return $version;
}

sub get_pid {
	if (-f $pid_file) {
		my $pid = `cat $pid_file`;
		chomp $pid;
		return $pid if ($pid and `ps --no-headers -o comm -p $pid` =~ /^yabs_p2p$/); # check if process name is yabs_p2p
	}
}

sub memory_usage {
	my $pid = shift;
	my $rss = int((`ps --no-headers -o rss -p $pid`) / 1024); # rss in mb
	chomp $rss;
	return $rss;
}

sub get_daemon_state {
	my $state = `$p2p_client mon_state | tail -n1`;
	#$state = "r025.7 OLD_TASKS_FOR:bsut01d(1),bsut02d(2)";
	chomp($state);

	my ($version, @params) = split(' ', $state);

	my %state = (VERSION => $version);
	foreach my $param (@params) {
		if ($param =~ /^OLD_TASKS_FOR:(.*)$/) {
			@{ $state{HOSTS} } = split(',', $1);
		} elsif ($param =~ /^(LAST_TASK_ADDED|LAST_TASK_DONE):(.*)$/) {
			push(@{ $state{PARAMS} }, $1.":".(time() - $2)); # calculate passed time
		} else {
			push(@{ $state{PARAMS} }, $param);
		}
	}
	return %state;
}

sub host_state {
	my $host = shift;
	if ($ignore_hostlist{$host}) {
		return 1; # warning
	}
	return 2; # critical
}

sub send_to_juggler {
	my ($host, $check, $status, $desc) = @_;
#	print "$status\t$host\t$desc\n";
	system("/usr/bin/juggler_queue_event --host=$host.yandex.ru --service=$check --status=$status --description=\"$desc\"");
}

