package main

import (
	click "github.com/roistat/go-clickhouse"
	"fmt"
	"bytes"
	"encoding/json"
	//"os"
	consulapi "github.com/hashicorp/consul/api"
	"strings"
	"errors"
)

const (
	consulHost = "localhost:8500"
)

type serverHost struct {
	ShardName string
	ShardAddr string
}

func (s serverHost) ServerName() string {
	return fmt.Sprintf("%s", strings.Split(s.ShardAddr, ":")[0])
}

func (s serverHosts) AllShards() []string {
	shards := make(map[string]int)
	var keys []string
	for _, i := range s {
		shards[i.ShardName] += 1
	}
	for k, _ := range shards {
		keys = append(keys, k)
	}
	return keys
}

type clickTable struct {
	Partition int
	NamePart string
	Bytes bytesSize
	Database string
	Table string
	Engine string
}

func (t clickTable) Name() string {
	return fmt.Sprintf("%s.%s", t.Database, t.Table)
}

type clickTables []clickTable

func (c clickTables) SizeTables() map[string]bytesSize {
	result := make(map[string]bytesSize)
	for _, data := range c {
		result[data.Name()] += data.Bytes
	}
	return result
}

type serverHosts []serverHost
type bytesSize int

var (
	pv int
	npv string
	bv int
	dv string
	tv string
	ev string

)

type clickMeta struct {
	serverHost
	MetaData clickTables
}

type typeChanks struct {
	Good clickTables
	Bad clickTables
}

type clickMetas []clickMeta

func (c *clickMetas) getEqualChanks(shard string) (*typeChanks, error) {
	shardServers := make(map[clickTable]int)
	var chanks typeChanks
	var err error
	numServer := 0

	for _, i := range *c {
		if i.ShardName != shard { continue }
		numServer += 1
		for _, j := range i.MetaData {
			shardServers[j] += 1
		}
	}

	for key, val := range shardServers {
		if val != numServer {
			chanks.Bad = append(chanks.Bad, key)
			continue
		}
		chanks.Good = append(chanks.Good, key)
	}

	if len(chanks.Bad) != 0 {
		err = fmt.Errorf("not found complect chanks")
	}
	return &chanks, err
}

func writeConsul(key string, jdata *[]byte) (err error) {
	config := consulapi.DefaultConfig()
	config.Address = consulHost
	consul, err := consulapi.NewClient(config)
	if err != nil {
		err = fmt.Errorf("error connect to consul: %s", err)
		return
	}

	kv := consul.KV()
	d := &consulapi.KVPair{Key: key, Value: *jdata}
	_, err = kv.Put(d, nil)
	if err != nil {
		err = fmt.Errorf("error put to consul: %s", err)
	}
	return
}

/* Получаем данные о расположении чанков с данными на машине. Результат кодируем в JSON.
   Пример структуры:
   	[{"Partition":201612,"NamePart":"20161215_20161215_1946302_1946302_0",
   	 "Bytes":16901,"Database":"system","Table":"query_log","Engine":"MergeTree"}]
 */
func getListPartition(server serverHost) (*[]byte, error) {
	var result clickTables
	var err error
	var jdata []byte
	conn := click.NewConn(server.ShardAddr, click.NewHttpTransport())
	query := click.NewQuery("SELECT partition, name, sum(bytes), database, table, engine FROM system.parts " +
				"group by partition, name, database, table, engine ")
	iter := query.Iter(conn)
	if err := iter.Error(); err != nil {
		return nil, fmt.Errorf("error connect: %s", err)
	}

	for iter.Scan(&pv, &npv, &bv, &dv, &tv, &ev) {
		result = append(result, clickTable{Partition: pv, NamePart: npv, Bytes: bytesSize(bv), Database: dv,
			Table: tv, Engine: ev})
	}

	jdata, err = json.Marshal(result)
	if err != nil {
		err = fmt.Errorf("error marshaling: %s", err)
	}
	return &jdata, err
}

/* Сохраняем в консул информацию о расположении чанков в каждом из кластеров.
   Стурктура пути: <тип машин>/<имя шарда>/<имя сервера>/chanks. Данные закодированы в JSON.
   Пример:
   	ppchouse/shard2/ppchouse02i.yandex.ru/chanks
   	ppchouse/shard2/ppchouse02k.yandex.ru/chanks
*/
func updateConsul(hosts serverHosts) error {
	errBuff := bytes.NewBuffer(nil)
	for _, h := range hosts {
		bytesData, err := getListPartition(h)
		if err != nil {
			errBuff.WriteString(fmt.Sprintln(err))
			continue
		}
		key := fmt.Sprintf("ppchouse/%s/%s/chanks", h.ShardName, h.ServerName())
		writeConsul(key, bytesData)
	}
	if errBuff.Len() != 0 {
		return errors.New(errBuff.String())
	}
	return nil
}

func readConsul(hosts serverHosts, meta *clickMetas) error {
	errBuff := bytes.NewBuffer(nil)
	config := consulapi.DefaultConfig()
	config.Address = consulHost
	consul, err := consulapi.NewClient(config)
	if err != nil {
		return fmt.Errorf("error connect to consul: %s", err)
	}

	kv := consul.KV()
	for _, h := range hosts {
		var jdata clickTables
		key := fmt.Sprintf("ppchouse/%s/%s/chanks", h.ShardName, h.ServerName())
		kvp, _, err := kv.Get(key, nil)
		if err != nil {
			errBuff.WriteString(fmt.Sprintln(err))
			continue
		}
		// Если ключа нет, то в3озвращается nil :(
		if kvp == nil {
			errBuff.WriteString(fmt.Sprintf("not found key %s", key))
			continue
		}

		err = json.Unmarshal(kvp.Value, &jdata)
		if err != nil {
			errBuff.WriteString(fmt.Sprintln(err))
			continue
		}

		m := clickMeta{serverHost{h.ShardName, h.ServerName()}, jdata}
		*meta = append(*meta, m)
	}
	return nil
}



func main() {
	var err error
	var hosts serverHosts
	var metaData clickMetas //Полный комплект данных по шардам всех машин
	var chanks *typeChanks
	mapChanks := make(map[string]typeChanks) //Задублированные и незадублированные чанки для шарда

	hosts = append(hosts, serverHost{ShardName: "shard2", ShardAddr: "ppchouse02i.yandex.ru:8123"})
	hosts = append(hosts, serverHost{ShardName: "shard2", ShardAddr: "ppchouse02e.yandex.ru:8123"})
	hosts = append(hosts, serverHost{ShardName: "shard2", ShardAddr: "ppchouse02k.yandex.ru:8123"})
	hosts = append(hosts, serverHost{ShardName: "shard2", ShardAddr: "localhost:8124"})
	err = updateConsul(hosts) //ЗПолучаем данные о чанках и записываем в консул
	if err != nil {
		fmt.Println(err)
	}
	readConsul(hosts, &metaData)
	for _, shard := range hosts.AllShards() {
		chanks, err = metaData.getEqualChanks(shard)
		if err != nil {
			fmt.Println(err)
		}
		mapChanks[shard] = *chanks
	}
	fmt.Println("#%v", mapChanks)
}