package main

import (
    "os"
    "flag"
    "fmt"
    "regexp"
    "github.com/go-ini/ini"
    "net/http"
    "io/ioutil"
    "net"
    "net/url"
    "encoding/json"
    "crypto/tls"
    "strings"
    "sort"
    "bytes"
    "time"
)

type MyConfig struct {
    ServerAddr string
    ServerPort int
    Ignore []string
    Replicated []string
    Databases []string
    ConductorAPI string
    ConductorGroups []string
    CacheDir string
    IgnoreSettings []string
}

type MyError struct {
    Value string
}

type JsonMembers []JsonData

type JsonData struct {
    Database string `json: database`
    Name string `json: name`
    Engine string `json: engine`
}

type JsonDict struct {
    Meta JsonMembers `json meta`
    Data JsonMembers `json: data`
    Rows int32 `json: rows`
}

type VariableMembers []VariableData

type VariableData struct {
    Name string `json: name`
    Value string `json: value`
    Changed int `json: changed`
}

type VariableDict struct {
    Meta VariableMembers `json: meta`
    Data VariableMembers `json: data`
    Rowns int32 `json: rows`
}

func (v VariableMembers) getNames() (keys []string) {
    map1 := make(map[string]int)
    for _, raw1 := range v {
        map1[raw1.Name] += 1
    }
    for key1, _ := range map1 {
        keys = append(keys, key1)
    }
    return keys
}

func (v VariableMembers) Len() int { return len(v) }
func (v VariableMembers) Swap(i, j int) {
    v[i], v[j] = v[j], v[i]
}
func (v VariableMembers) Less(i, j int) bool {
    return v[i].Name < v[j].Name
}

const (
    cacheTime = 600
)

var (
    myConfig *string = flag.String("c", "/etc/default/clickTables.ini", "Путь до конфигурационного файла.")
    debug *bool = flag.Bool("d", false, "Включить debug режим.") 
    chRepl *bool = flag.Bool("r", true, "Проверить, что таблицы из конфига(replicated или все, исключая ignore) реплицируемы.")
    chStruct *bool = flag.Bool("st", false, "Проверить, что структура всех БД кликхауса из конфига(checkDBs) едина.")
    chVariables *bool = flag.Bool("vs", false, "Проверить, что настройки всех БД из конфига совпадаютю")
    queryTables string = "SELECT database, name, engine FROM system.tables FORMAT JSON"
    queryVariables = "SELECT name, value, changed FROM system.settings FORMAT JSON"
)

func (e MyError) Error() string {
    return fmt.Sprintf("%v", e.Value)
}

func (e MyError) Monrun(status int) {
    fmt.Printf("%d; %s\n", status, e.Value)
}

func (jd JsonMembers) ReplicationStatus() {
    /* Метод, позволяет распределить таблицы по трем категориям: 
         replicate - таблицы с репликацией,
         ignore - таблицы, которые были проигнорированы,
         others - таблицы, которые без репликации и которым она требуется.
       Вывод отправляется в os.stdout. */  
    var replicate, ignore, others []string
    for _, jl := range jd {
        fullName :=  strings.Join([]string{jl.Database, jl.Name}, ".")
        switch jl.Engine {
        case "ReplicatedMergeTree": 
            replicate = append(replicate, fullName)
        case "ReplicatedCollapsingMergeTree":
            replicate = append(replicate, fullName)
	case "ReplicatedReplacingMergeTree":
	    replicate = append(replicate, fullName)
        case "IgnoreTable":
            ignore = append(ignore, fullName)
        default:
            others = append(others, fullName)
        }
    }
    if *debug {
        fmt.Printf("===Replicated tables===\n%s\n\n", strings.Join(replicate, ", "))
        fmt.Printf("===Ignored tables===\n%s\n\n", strings.Join(ignore, ", "))
        fmt.Printf("===Others(not replicated) tables===\n%s\n", strings.Join(others, ", "))
        return
    } 
    if len(others) == 0 { 
        r := MyError{ "OK" }
        r.Monrun(0)
    } else {
        r := MyError{ fmt.Sprintf("Found %d not replicated tables. Use \"%s -d\" for show it.\n", len(others), os.Args[0]) }
        r.Monrun(1)
    }
}


func parseTables(f func(string) ([]byte, error), config *MyConfig, db string) (values JsonMembers, errTables MyError){
    /* Принимает JSON в виде последовательности байтов, конфигурационный файл для извледения списока патернов таблиц, 
       которые требуется обработать или проигнорировать, имя инстанса к которуму следует подключится. В ответ возвращается 
       nil или MyError{} в зависимости от наличия или отсутствия ошибки. Вывод: список []JsonData, и блок ошибок MyError. */
    js1 := JsonDict{}

    var allowPatterns []string
    if *chStruct {
         allowPatterns = config.Databases 
    } else {
         allowPatterns = config.Replicated
    }
    if *debug { fmt.Printf( "Use patterns: %v\t", allowPatterns ) }
    body, err := f( db )
    if err != nil { return nil, MyError{ fmt.Sprintln(err) } }
    err = json.Unmarshal(body, &js1)
    if err != nil { return nil, MyError{ fmt.Sprintln(err) } }
    for _, val1 := range js1.Data {
        tableName := strings.Join( []string{val1.Database, val1.Name}, "." )
        var founded bool 
        for _, pattern := range allowPatterns {
            rx := regexp.MustCompile("^"+pattern+"$")
            if (len(rx.FindStringSubmatch(tableName)) != 0) {
                founded = true
                break
            }
        }
        if ! founded { continue } 
        for _, pattern := range config.Ignore {
            rx := regexp.MustCompile("^"+pattern+"$")
            if (len(rx.FindStringSubmatch(tableName)) != 0) {
                val1.Engine = "IgnoreTable"
                break
            }
        }
        values = append(values, val1)
    }
    return
}

func parseVariables(f func(string) ([]byte, error), config *MyConfig, db string) (values VariableMembers, errTables MyError){
    /*
    Получаем список переменных, установленных в clickhouse.
     */
    js2 := VariableDict{}
    body, err := f( db )
    if err != nil { return nil, MyError{ fmt.Sprintln(err) } }
    if body == nil { return nil, MyError{ "" } }
    err = json.Unmarshal(body, &js2)
    if err != nil { return nil, MyError{ fmt.Sprintln(err) } }
    var found int
    for _, val3 := range js2.Data {
	found = 0
	for _, ignore := range config.IgnoreSettings {
	    if (val3.Name == ignore) {
	        found = 1
	        break
	    }
	}
        if (found == 0) {
            values = append(values, val3)
        }
    }
    return values, errTables
}

func groups2Hosts( config *MyConfig ) ( result []string, err1 MyError ) {
    for _, group1 := range config.ConductorGroups {
        err := os.MkdirAll( config.CacheDir, 0755 )
        if *debug && err != nil && os.IsNotExist(err) { fmt.Printf( "Error create directory %s", config.CacheDir ) }
        fileCache := fmt.Sprintf( "%s/%s", config.CacheDir, group1 )
        bytes1, err := ioutil.ReadFile( fileCache )
        if *debug { fmt.Printf( "Read code: %s\n", err ) }
        old := func() bool {
            if stat1, _ := os.Stat( fileCache ); time.Since( stat1.ModTime() ) > cacheTime { 
                if *debug { fmt.Println( "Found old conductor cache file. Try update it.") } 
                return true
            } 
            return false
        }
    
        if err := json.Unmarshal( bytes1, &result ); ( len(result) > 0 ) && ( err == nil ) && ( ! old() ) { return }

	tr := &http.Transport{
		TLSClientConfig: &tls.Config{InsecureSkipVerify: true},
	}
        client := http.Client{
            Timeout: time.Duration(10*time.Second),
	    Transport: tr,
        }
        resp, err := client.Get( config.ConductorAPI + "/" + group1 )
    
        if err != nil && len(result) == 0 { return nil, MyError{ fmt.Sprintf("%s", err) } }
        if err != nil && len(result) > 0 {
            if *debug { fmt.Println( "Conductor dont working. Used cache." ) } 
            return result, err1
        } 

        defer resp.Body.Close()
        js2, _ := ioutil.ReadAll( resp.Body )
        if resp.StatusCode != 200 && len(result) == 0  { 
            return nil, MyError{ fmt.Sprintf( "Cache empty and conductor status code = %d %s", resp.StatusCode, js2) } 
        }
        result = []string{}  
        for _, line := range  bytes.Split( js2, []byte{ '\n' } ) {
            if len(line) == 0 { continue }
            result = append(result, string(line))
        }
    
        if len(result) == 0 { return nil, 
            MyError{ fmt.Sprintf( "Not found hosts for conductor group  = %s", group1) } 
        } else { 
            bytes1, _ := json.Marshal( result )
            err = ioutil.WriteFile( fileCache, bytes1, 0444 )
            if *debug { fmt.Printf( "Write code: %s\n", err ) }
        }
    }
    return result, err1
}


func diffSlice( slice1, slice2 []string ) ( diffSlice []string ) {
    m := map[string]int{}
    
    for _, val1 := range slice1 {
        m[val1] += 1
    }
    for _, val2 := range slice2 {
        m[val2] += 1
    }
    
    for key, val := range m {
        if val == 1 {
            diffSlice = append( diffSlice, key )
        }
    }
    return
}

func main() {
    
    flag.Parse()
    
    config := &MyConfig{
        ServerAddr: "localhost",
        ServerPort: 8123,
        Ignore: []string{ "system.*" },
        Databases: []string{ "*" },
        Replicated: []string{ "*" },
        ConductorAPI: "http://c.yandex-team.ru/api/groups2hosts",
        ConductorGroups: []string{ "" },
        CacheDir: "/var/cache/conductor/groups",
	IgnoreSettings: []string{ "" },
    }
    
    if _, err := os.Stat(*myConfig); os.IsNotExist(err) {
        if *debug { fmt.Printf("Config file %s dosn't exist. Used default settings.\n", *myConfig) }
    } else {
        if *debug { fmt.Printf("Used config file %s\n", *myConfig) }
        cfg, err := ini.Load(*myConfig)
        
        if err != nil {
            fmt.Printf("1; Error loading %s file\n", *myConfig)
            os.Exit(1)
        }
        
        if cfg.Section("main").HasKey("host") {
            config.ServerAddr = cfg.Section("main").Key("host").String()   
        }
        if cfg.Section("main").HasKey("ignore") {
            config.Ignore = cfg.Section("main").Key("ignore").Strings(",")
        }
        if cfg.Section("main").HasKey("replicated") {
            config.Replicated = cfg.Section("main").Key("replicated").Strings(",")
        }
        if cfg.Section("main").HasKey("databases") {
            config.Databases = cfg.Section("main").Key("databases").Strings(",")
        }
        if cfg.Section("main").HasKey("conductor_api") {
            config.ConductorAPI = cfg.Section("main").Key("conductor_api").String()         
        }
        if cfg.Section("main").HasKey("conductor_groups") { 
            config.ConductorGroups = cfg.Section("main").Key("conductor_groups").Strings(",")
        }
        if cfg.Section("main").HasKey("cache_dir") {
            config.CacheDir = cfg.Section("main").Key("cache_dir").String()
        }
	if cfg.Section("main").HasKey("ignore_settings") {
	    config.IgnoreSettings = cfg.Section("main").Key("ignore_settings").Strings(",")
        }
    }
    if *debug { fmt.Printf("%#v, cacheTime=%d\n", config, cacheTime) }

    chr := func(v string) (b []byte, err error) {
        /* Функция принимает путь до кликхауса, в ответ возвращает JSON в виде последовательности байтов, 
        nil или MyError{}, в зависимости от наличия или отсутвия ошибки. */
        
	conn, err := net.DialTimeout("tcp", v, time.Duration(2*time.Second))
	if err != nil {
	    if *debug { fmt.Println(err) }
	    return nil, MyError{ "CONNERR" }
	} else {
	    conn.Close()
	}

        if err := strings.HasPrefix(v, "http:"); ! err {
            v = "http://"+v
        }
        if *debug { fmt.Println(v) }
        var q string
        if *chVariables {
            q = queryVariables
        } else {
            q = queryTables
        }
        body := url.Values{ "query": { q } }

	tr := &http.Transport{
	    TLSClientConfig: &tls.Config{InsecureSkipVerify: true},
	}
        client := http.Client{
            Timeout: time.Duration(10*time.Second),
	    Transport: tr,
        }
        resp, err := client.Get( v+"/?"+body.Encode() )

        if err == nil {
            defer resp.Body.Close()
            js1, err := ioutil.ReadAll( resp.Body )
            if resp.StatusCode == 200 {
                return js1, err
            } else {
                return nil, MyError{ fmt.Sprintf("Wrong response. Status code %s", resp.Status) }
            }
        }
        return nil, MyError{ fmt.Sprintf("Error request: %v", err) }
    }

    sock := func( h string, p int ) string { return fmt.Sprintf( "%s:%d", h, p ) }

    if ( ! *chStruct ) && ( *chRepl ) && ( ! *chVariables ){
        val2, err2 := parseTables(chr, config, sock(config.ServerAddr, config.ServerPort))
        if len(err2.Value) == 0 {
            val2.ReplicationStatus()
	} else if strings.Contains(err2.Value, "CONNERR") {
	    err2.Value = fmt.Sprintf("connection timeout to %s:%s", config.ServerAddr, config.ServerPort)
            err2.Monrun(0)
        } else {
            err2.Monrun(1)
        }
    } else if ( ! *chStruct ) && ( *chRepl ) && ( *chVariables ) {
        variablesDatabases := make(map[VariableData][]string)
        var structError, connErr []string
        hostList, err6 := groups2Hosts( config )
        if len(err6.Value) != 0 { err6.Monrun(2); os.Exit(0) }
        for _, hostname := range hostList {
            val6, err6 := parseVariables(chr, config, sock(hostname, config.ServerPort))
	    if strings.Contains(err6.Value, "CONNERR") {
		connErr = append(connErr, hostname)
	    } else if len(err6.Value) == 0 {
                for _, i6 := range val6 {
                    variablesDatabases[i6] = append( variablesDatabases[i6], hostname)
                }
	    } else {
                structError = append(structError, fmt.Sprintf("%s: %s", hostname, err6.Value))
            }
        }
        for k, v := range variablesDatabases {
            if (len(v) + len(connErr)) == len(hostList) {
                delete(variablesDatabases, k)
            }
        }

        var diffVars VariableMembers
        for k, _ := range variablesDatabases {
            diffVars = append(diffVars, k)
        }
        if *debug {
            sort.Sort(diffVars)
            for _, k := range diffVars {
                found := variablesDatabases[k]
                lost := diffSlice(hostList, found)
                fmt.Printf("[%s] (value: %s, changed: %b)\n Found in %s \n Lost in %s\n", k.Name, k.Value, k.Changed, found, lost)
            }
            if len(diffVars) == 0 {
                fmt.Println("Not found diffent settings in cluster.")
            }
        } else {
            r := new(MyError)
            if (len(structError) == 0) && (len(diffVars) == 0) {
                r.Value = "OK"
                r.Monrun(0)
            } else {
                r.Value = fmt.Sprintf("Found different variables. Use '%s -vs -d' for verbose message.", os.Args[0])
                r.Monrun(1)
            }
        }
    } else {
        structDatabases := make(map[string][]string)
        var lostTables, structError, connErr, ignoreTables []string
        hostList, err2 := groups2Hosts( config )
        if len(err2.Value) != 0 { err2.Monrun(2); os.Exit(0) }  
        for _, hostname := range hostList {
            val3, err3 := parseTables(chr, config, sock(hostname, config.ServerPort))
	    if strings.Contains(err3.Value, "CONNERR") {
	        connErr = append(connErr, hostname)
	    } else if len(err3.Value) == 0 {
                for _, i1 := range val3 {
                    j1 := fmt.Sprintf( "%s.%s/%s", i1.Database, i1.Name, i1.Engine )
                    if i1.Engine == "IgnoreTable" { ignoreTables = append( ignoreTables, j1 ); continue }
                    structDatabases[j1] = append( structDatabases[j1], hostname )
                }
	    } else {
                structError = append( structError, fmt.Sprintf("%s: %s", hostname, err3.Value))
            }
        }
        for key4, val4 := range structDatabases {
            if len(hostList) != (len(val4) + len(connErr)) {
		    lostTables = append( lostTables, key4)
	    }
        }
        r := new(MyError)
        if (len(structError) == 0) && (len(lostTables) == 0) {
            r.Value = "OK"; r.Monrun(0)
        } else if (len(structError) == 0) {
            if *debug { 
                fmt.Println( "Lost tables:") 
                for _, key5 := range lostTables {
                    fmt.Printf( "%s:\n\t found in %s\n\t lost in %s\n", key5, structDatabases[key5], diffSlice( hostList, structDatabases[key5] ) )
                if len(ignoreTables) > 0 { fmt.Printf( "Ignore tables: %s\n", ignoreTables ) }
                }
            }
            if ! *debug {
                r.Value = fmt.Sprintf("Found lost tables. Use '%s -st -d' for varbose message.", os.Args[0])
                r.Monrun(1) }
        } else {     
            if *debug { fmt.Println(structError) }
            if ! *debug {
                r.Value = fmt.Sprintf("Found errors. Use '%s -d' for verbose message.", os.Args[0])
                r.Monrun(1)
            }
        }    
    } 
}
