package main

import (
    "fmt"
    "flag"
    "os"
    "net/url"
    "errors"
    "sort"
    "encoding/json"
    "strconv"
    "net/http"
    "bytes"
    "time"
    "strings"
    "log/syslog"
    "path/filepath"
    "io/ioutil"
)

const (
    shadowDir = "/opt/clickhouse/shadow"
    metaDir = "/opt/clickhouse/metadata"
    cacheDir = "/var/cache/clickhouseBackup"
)

var (
    tables sliceString
    databases sliceString
    partitions sliceIntToStr
    logLevel syslog.Priority
    logWriter *syslog.Writer
    err error
    t func() string = func() string { t := time.Now(); return fmt.Sprintf("%d%s%d_%02d%02d", t.Year(), t.Month(), t.Day(), t.Hour(), t.Minute() ) }
    host *string = flag.String("h", "localhost", "Имя сервера ClickHouse.")
    port *int = flag.Int("p", 8123, "Порт для подключения к БД.")
    backupName *string = flag.String("n", t(), "Имя снапшота в /opt/clickhouse/shadow. По умолчанию берется текущая дата.")
    debug *bool = flag.Bool("d", false, "Включить debug режим. Логи ищи  в /var/log/syslog.")
    minTimeBackup *time.Duration = flag.Duration("time", 1440*time.Hour, "Время после которого можно удалять бекапы(modify time).")    
    maxCountSaved *int = flag.Int("count", 60, "Общая сумма бекапов, после которых возможно удаление.")
)

// Для парсинга флагов массивов чисел и строк.
type sliceString []string
type sliceIntToStr []string

func (t *sliceString) String() string {
    return fmt.Sprintf("%s", *t)
}

func (t *sliceString) Set(value string) error {
    for _, s := range strings.Split(value, ",") {
        *t = append(*t, s)
    }
    return nil
} 

func (p *sliceIntToStr) String() string {
    return fmt.Sprintf("%s", *p)
}

func (p *sliceIntToStr) Set(value string) error {
    for _, s :=range strings.Split(value, ",") {
        _, err := strconv.Atoi(s)
        if err != nil { 
            logWriter.Warning(fmt.Sprintf("WARNING [Flag.sliceIntToStr] %s.\n", err))
        } else {
            *p = append(*p, s)
        }
    }
    if len(*p) == 0 { 
        return errors.New("Список партиций(partition) пуст. Имена партиций могут быть только целыми числами.\n") 
    }
    return nil
}

// Создаем сокет
type socket struct {
    host *string
    port *int
} 

// Структура json ответа из clickhouse
type systemResponse struct {
    Meta systemMetaList `json: meta`
    Data systemDataList `json: data`
    Rows int32 `json: rows`
}

type systemDataList []systemData
type systemMetaList []systemMeta

// Структура одной пачки данных из clickhouse.
type systemData struct {
    Database string `json: database`
    Table string `json: table`
    Partition string `json: partition`
    Size string `json: size`
}

// Структура одной пачки метаданных из clickhouse.
type systemMeta struct {
    Name string `json: database`
    Type string `json: type`
}

// Для работы с дирректориями /opt/clickhouse/shadow и удаления старых.
type fileInfoList []fileInfo

type fileInfo struct {
    Name string
    ModTime time.Time
}

func (f fileInfoList) Len() int { return len(f) }

func (f fileInfoList) Less(i, j int) bool {
    fi := f[i].ModTime
    fj := f[j].ModTime
    return fi.Sub(fj).Seconds() > 0 
}

func (f fileInfoList) Swap(i, j int) {
    f[i], f[j] = f[j], f[i]
}

// Структура под запрос одной БД и одной таблицы.
type requestSingle struct {
    table *string
    database *string
    partition *string
    data *[]byte
    socket
}

func (r *requestSingle) FreezePartition(n string) (err error) {
    /* Делаем hardlink для таблицы в /opt/clickhouse/shadow/<n>/<table>,
       где <p> - идентификатор партиции, над которой делаем бекап,
           <n> - имя дирректории, в которую поместить бекап,
           <table> - имя таблицы, которую бекапим.
    */
    var nm string = "FreezePartition"
    r.data = new([]byte)
    str1 := fmt.Sprintf("ALTER TABLE %s.%s FREEZE PARTITION %s WITH NAME '%s'", *r.database, *r.table, *r.partition, n)
    logWriter.Debug(fmt.Sprintf("DEBUG [FreezePartition] Запрос в clickhouse: %s\n", str1))
    buff1 := bytes.NewReader([]byte(str1)) 
    url1 :=  fmt.Sprintf("http://%s:%d", *r.host, *r.port)  
    resp, err := http.Post(url1, "application/octet-stream", buff1)
    if err != nil {
        logWriter.Crit(fmt.Sprintf("ERROR [%s] Ошибка запроса: %v\n", nm, err))
        return errors.New(fmt.Sprintf("[%s] %v",nm, err))
    }
    defer resp.Body.Close()
    data1, _ := ioutil.ReadAll(resp.Body)
    if resp.StatusCode != 200 {
        logWriter.Crit(fmt.Sprintf("ERROR [%s] Ошибка создания снапшота %s.%s/%s. Код ответа: %d\n", nm, *r.database, *r.table, *r.partition, resp.StatusCode))
        logWriter.Crit(fmt.Sprintf("ERROR [%s] Запрос %s, тело ответа: %s\n", nm, str1, data1))
        return errors.New(fmt.Sprintf("Status code %d", resp.StatusCode))
    }
    r.data = &data1
    logWriter.Debug(fmt.Sprintf("DEBUG [FreezePartition] Ответ сервера: %v", *r.data))
    return
}

func (r requestSingle) CopyMetadata(n string) (err error) {
    // Сохраняем структуры таблиц БД clickhouse.
    var data []byte
    var nm string = "CopyMetadata"
    // /opt/clickhouse/metadata/<db>/<table>.sql
    metaSrc := fmt.Sprintf("%s/%s/%s.sql", metaDir, *r.database, *r.table)
    // /opt/clickhouse/shadow/<backupName>/meta/<db>/<table>.sql
    metaDir := fmt.Sprintf("%s/%s/metadata/%s", shadowDir, n, *r.database)
    os.MkdirAll(metaDir, 0755)
    metaDst := fmt.Sprintf("%s/%s.sql", metaDir, *r.table)
    if data, err = ioutil.ReadFile(metaSrc); err != nil {
        logWriter.Err(fmt.Sprintf("ERROR [%s] Ошибка сохранения таблицы с метаданными(metadata) %s.%s: %s\n", nm, *r.database, *r.table, err))
        return err
    }
    err = ioutil.WriteFile(metaDst, data, 0644) 
    return
}

// Структура под запрос нескольких таблиц, партиций и баз. 
type requestMulty struct {
    tables *sliceString
    partitions *sliceIntToStr
    databases *sliceString
    data *[]byte
    socket
} 

func (m *requestMulty) GetPartitions() (err error) {
    /* Выбираем партиции,из system.parts для снятия снапшота. Виртуальные(сагрерированные поверх) таблицы не
       отображаются. Т.ж. выбрасываем из выборки таблицы размером в 0байт.
    */
    var nm string = "GetPartition"
    m.data = new([]byte)
    reqStr := "SELECT database, table, partition, sum(bytes) AS size FROM system.parts WHERE bytes>0"
    if len(*m.tables) > 0 { reqStr = fmt.Sprintf("%s AND table IN ('%s')", reqStr, strings.Join(*m.tables, "','")) }
    if len(*m.partitions) > 0 { reqStr = fmt.Sprintf("%s AND partition IN ('%s')", reqStr, strings.Join(*m.partitions, "','")) }
    if len(*m.databases) > 0 { reqStr = fmt.Sprintf("%s AND database IN ('%s')", reqStr, strings.Join(*m.databases, "','")) }
    reqStr = fmt.Sprintf("%s GROUP BY partition, database, table FORMAT JSON", reqStr) 
    logWriter.Debug(fmt.Sprintf("DEBUG [GetPartition] Request clickHouse: %s\n", reqStr))
    req := url.Values{ "query": { reqStr }}
    url1 := fmt.Sprintf("http://%s:%d/?%s", *m.host, *m.port, req.Encode())
    resp, err := http.Get(url1)
    if err != nil {
        logWriter.Crit(fmt.Sprintf("ERROR [%s] Ошибка подключения к %s:%d: %s", nm, *m.host, *m.port, err))
        return errors.New(fmt.Sprintf("[%s] %v",nm, err))
    }
    defer resp.Body.Close()
    data1, _ := ioutil.ReadAll(resp.Body)
    if resp.StatusCode != 200 {
        logWriter.Crit(fmt.Sprintf("ERROR [%s] Ошибка получения списка партиций(system.parts). Код ответа: %d\n", nm, resp.StatusCode))
        logWriter.Crit(fmt.Sprintf("ERROR [%s] Запрос %s, тело ответа: %s\n", nm, reqStr, data1))
        return errors.New(fmt.Sprintf("Status code %d", resp.StatusCode))
    }
    m.data = &data1
    return
}
 
func rotateSnapshot() error {
    // Удаляем старые снапшоты из /opt/clickhouse/shadow. По-умолчанию храним 60 копий. 
    var oldFilesList, deletedList fileInfoList
    
    errBuff := bytes.NewBuffer(nil)
    files, _ := ioutil.ReadDir(shadowDir)
    
    for _, file := range files {
        if time.Since(file.ModTime()) < *minTimeBackup { continue }
        f1 := fileInfo{ Name: file.Name(), ModTime: file.ModTime() }
        oldFilesList = append(oldFilesList, f1)
    }
    
    //Список пуст? Выходим.
    if len(oldFilesList) == 0 {
        logWriter.Warning(fmt.Sprintf("WARN [rotateSnapshots] Не обнаружены снапшоты старше %s\n", *minTimeBackup))
        return nil
    } 
    
    //Удаляем старые бекапы при условии, что их больше положенного maxCountSaved.
    if delta:=(len(oldFilesList)-*maxCountSaved); delta>0 {
        sort.Sort(sort.Reverse(oldFilesList)) //Сортируем по возрастанию. Удалены будут первые объекты в списке.append
        logWriter.Debug(fmt.Sprintf("INFO [rotateSnapshot] Будет удалено %d бекапов\n", delta))
        deletedList = (oldFilesList[:delta])
    } 
    
    if (len(deletedList) == 0) && *debug {
        logWriter.Debug(fmt.Sprintf("DEBUG [rotateSnapshot] Список бекапов для удаления пуст.\n"))
    }
    
    for _, delFile := range deletedList {
        path := fmt.Sprintf("%s/%s", shadowDir, delFile.Name)
        if err := os.RemoveAll(path); err != nil { 
            errBuff.WriteString(fmt.Sprintln(err))
        }
    }
    
    if errBuff.Len() != 0 {
        logWriter.Crit(fmt.Sprintf("CRIT [rotateSnapshot] Error: %s\n", errBuff))
        return errors.New(errBuff.String()) 
    }
    return nil     
}   

func monrun(status bool) {
    monFile := fmt.Sprintf("%s/status", cacheDir) 
    _, err := os.Stat(cacheDir)
    if (err!=nil) && (os.IsNotExist(err)) {
        err = os.MkdirAll(cacheDir, 0755)
        logWriter.Warning(fmt.Sprintf("WARN [monrun] %s", err))
    }
    writer := func(s string) (err error) { 
        err = ioutil.WriteFile(monFile, []byte(s), 0644) 
        return 
    }
    syslogger := func(m error) { 
        logWriter.Warning(fmt.Sprintf("WARN [monrun] Error: %s", err)) 
    }
    switch status {
    case true:
        if err = writer("0; Backup success\n") ; err != nil {
            syslogger(err)
        }
        logWriter.Info(fmt.Sprintf("INFO [monrun] Создание бекапа завершено успешно"))
        os.Exit(0)
    case false:
        if err = writer("2; Backup failed\n"); err != nil {
            syslogger(err)
        }
        logWriter.Crit(fmt.Sprintf("CRIT [monrun] Ну ты понимаешь, что что-то сломалось?"))
        os.Exit(1)
    }
}

func main() {
    flag.Var(&tables, "tables", "Список таблиц для бекапа.")
    flag.Var(&partitions, "partitions", "Список партиций для бекапа.")
    flag.Var(&databases, "databases", "Список БД для бекапов.")
    flag.Parse()

    var jsonResponse systemResponse
    var status bool = true
    
    switch *debug {
    case true:
        logLevel = syslog.LOG_DEBUG | syslog.LOG_LOCAL0
    case false:
        logLevel = syslog.LOG_CRIT | syslog.LOG_LOCAL0
    default:
        panic("Debug variable not bool!")
    }
    
    logWriter, err = syslog.New(logLevel, "clickhouseBackup")
    if err != nil {
        panic("Ошибка в запуске логирования через /dev/log")
    }
    defer logWriter.Close()

    logWriter.Debug(fmt.Sprintf("DEBUG [Flags] Список БД от flag.databases: %v", databases))
    logWriter.Debug(fmt.Sprintf("DEBUG [Flags] Список таблиц от flag.tables: %v", tables))
    logWriter.Debug(fmt.Sprintf("DEBUG [Flags] Список партиций от flag.partitions: %v", partitions))
    
    sock := socket{ host, port }
    system := requestMulty{ tables: &tables, partitions: &partitions, databases: &databases, socket: sock}
    
    if err = system.GetPartitions(); err != nil { monrun(false) }
       
    if err = json.Unmarshal(*system.data, &jsonResponse); err != nil { 
        logWriter.Crit(fmt.Sprintf("ERROR [json.Unmarshal] %s", err))
        monrun(false) 
    }
    if len(jsonResponse.Data) == 0 {
        status = false
        logWriter.Warning(fmt.Sprintf("WARN [json.Unmarshal] Не найдено ни одной подходящей таблицы для бекапа."))
    } 
    
    for _, value := range jsonResponse.Data { 
        req := requestSingle{ database: &value.Database, table: &value.Table, partition: &value.Partition, socket: sock }
        tPath := filepath.Join(shadowDir,*backupName, "data", value.Database, value.Table) //Дирректория для бекапа таблицы.
        tCorrupt := fmt.Sprintf("%s.%s", tPath, "bad") //Сохраняем битые бекапы с суффиксом ".bad".
        for nTry := 0; nTry < 3; nTry++ {
            if _, err := os.Stat(tCorrupt); err == nil {
                logWriter.Warning(fmt.Sprintf("WARN [FreezePartition] Удаление битого бекапа %s.", tCorrupt))
                os.RemoveAll(tCorrupt) //Удаляем битый бекапт таблицы. Будет еще несколько попыток.
            }
            if err = req.FreezePartition(*backupName); err == nil { break }
            logWriter.Warning(fmt.Sprintf("WARN [FreezePartition] Попытка №%d заморозить партицию неуспешна.", nTry+1))
            if ok := os.Rename(tPath, tCorrupt); ok == nil {
                logWriter.Warning(fmt.Sprintf("WARN [FreezePartition] Перемещение битого бекапа %s -> %s", tPath, tCorrupt))
            } else { 
                logWriter.Warning(fmt.Sprintf("WARN [FreezePartition] Перемещение битого бекапа %s -> %s неуспешно: %s", tPath, tCorrupt, ok))
            }
        }
        if err != nil { status = false; continue }
        err = req.CopyMetadata(*backupName)
        if err != nil { status = false }
    } 
    
    if err := rotateSnapshot(); err != nil { status = false }
    monrun(status)
}