#!/usr/bin/env python
#
# This program is free software: you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; version 2.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# this program. If not, see http://www.gnu.org/licenses/.

import os
import sys
import time
import socket

import re

SYSFS_BCACHE_PATH = '/sys/fs/bcache/'
WRITABLE_KEYWORDS = ('attach', 'clear', 'detach', 'stop', 'create', 'register', 'prune', 'trigger')

hostname = socket.getfqdn()

def file_to_lines(fname):
    if os.path.islink(fname) or not os.path.isfile(fname) or any([x in fname for x in WRITABLE_KEYWORDS]):
        return [] 
    try:
        with open(fname, "r") as fd:
            return fd.readlines()
    except:
        return []


def file_to_line(fname):
    ret = file_to_lines(fname)
    if ret:
        return ret[0].strip()
    return ''


def interpret_bytes(x):
    '''Interpret a pretty-printed disk size.'''
    factors = {
        'k': 1 << 10,
        'M': 1 << 20,
        'G': 1 << 30,
        'T': 1 << 40,
        'P': 1 << 50,
        'E': 1 << 60,
        'Z': 1 << 70,
        'Y': 1 << 80,
    }

    m = re.search('(-?\d+\.?\d*)(\w?)', x)
    factor = 1
    if m.group(2) and m.group(2) in factors:
        factor = factors[m.group(2)]
    return int(float(m.group(1)) * factor)


def bcache_uuids():
    uuids = []

    if not os.path.isdir(SYSFS_BCACHE_PATH):
        print('# bcache is not loaded.')
        return uuids

    for cache in os.listdir(SYSFS_BCACHE_PATH):
        if not os.path.isdir('%s%s' % (SYSFS_BCACHE_PATH, cache)):
            continue
        uuids.append(cache)

    return uuids

def map_uuid_to_bcache(uuid):
    devices = []
    for obj in os.listdir(os.path.join(SYSFS_BCACHE_PATH, uuid)):
        if obj.startswith('bdev'):
           devices.append(os.path.basename(os.readlink(os.path.join(SYSFS_BCACHE_PATH, uuid, obj, 'dev'))))
    return devices

def mk_graphite_msg(*args, **kwargs):
    msg = '.'.join([re.sub('\/', '.', re.sub('[^0-9a-zA-Z-/]', '_', x)) for x in args])

    if kwargs.get('value') != None:
        msg += " " + str(int(kwargs['value']))
        msg += " " + str(int(kwargs.get('time', time.time())))
    return msg

def get_cachedev_result(uuid, stat):
   pass
   for fname in files_list:
       for obj in os.listdir(os.path.join(SYSFS_BCACHE_PATH, uuid)):
           if not (obj.startswith('cache') and os.path.isdir(obj)):
               continue
           value = int(file_to_line('%s/%s/%s/%s' % (SYSFS_BCACHE_PATH, uuid, obj, stat)))

def dir_to_graphite_messages(path, graphite_prefix):
    messages = []
    for fname in os.listdir(path):
        try:
            value = interpret_bytes(file_to_line(os.path.join(path, fname)))
            graphite_path = [x for x in graphite_prefix] + [fname]
            messages.append(mk_graphite_msg(*graphite_path, value=value))
        except:
            pass
    return messages

def kv_stats_to_graphite_messages(path, graphite_prefix):
    messages = []
    reg = re.compile('^([\w\s-]+):\s*(-?\d+\.?\d*\w?)')
    try:
        data = file_to_lines(os.path.join(path))
        for line in data:
            m = reg.match(line)
            (key, value) = (m.group(1), interpret_bytes(m.group(2)))
            graphite_path = [x for x in graphite_prefix] + [key]
            messages.append(mk_graphite_msg(*graphite_path, value=value))
    except:
        pass
    return messages

def get_cacheset_messages(uuid, device):
   messages = []
   base_path = os.path.join(SYSFS_BCACHE_PATH, uuid)
   dirs_to_graph=('internal', 'cache', 'bdev', 'stats_')

   # cache set data
   messages.extend(dir_to_graphite_messages(base_path, ('one_min', hostname, 'bcache-' + device)))
   for fname in os.listdir(base_path):
       path = os.path.join(base_path, fname)

       if not (os.path.isdir(path) and (any([fname.startswith(x) for x in dirs_to_graph]))):
           continue 
       # bdev0, cache0, stats_, etc
       messages.extend(dir_to_graphite_messages(path, ('one_min', hostname, 'bcache-' + device, fname)))

   key_value_paths = ('internal/bset_tree_stats', 'bdev0/writeback_rate_debug')
   for fname in key_value_paths:
       path = os.path.join(base_path, fname)
       messages.extend(kv_stats_to_graphite_messages(path, ('one_min', hostname, 'bcache-' + device, fname)))
   return messages

def main():
    uuids = bcache_uuids()
    for uuid in uuids:
        devices = map_uuid_to_bcache(uuid)
        for device in devices:
            print '\n'.join(get_cacheset_messages(uuid, device))
    sys.stdout.flush()

if __name__ == '__main__':
    main()
