#!/usr/bin/env python
# *-* encoding: utf-8 *-*

# Программа для мониторинга состояния конфигурационного файла реплик в монге.

from optparse import OptionParser
from ConfigParser import ConfigParser
import logging
import os
import pymongo
import sys
import unittest 

global config 

DEFAULTS = {
    'mongo_port': 27700,
    'mongo_user': 'root',
    'mongo_password': '',
    'mongo_host': 'localhost',
    'mongo_replica': 'moddb.rs0',
    'need_votes': 3,
    'all_tags': ["dc"],    #тэг нужен для всех записей
    'once_tags': ["backup", "heavy"], #тэг нужен для одной и более записей
    'err_modes': ["sortOfImportant", "backedUp"], #тэги для правильной работы writeConcern
}

DEBUG = False

body1 = [ {u'_id': u'moddb.rs0', 
           u'version': 9, 
           u'members': 
                [   {u'votes': 0, u'tags': {u'backup': u'B1', u'dc': u'sas'}, u'priority': 0.0, u'host': u'localhost:27700', u'hidden': True, u'_id': 0}, 
                    {u'priority': 4.0, u'host': u'moddb-mongo08f.yandex.ru:27700', u'_id': 1, u'tags': {u'dc': u'myt'}}, {u'priority': 5.0, u'host': u'moddb-mongo08h.yandex.ru:27707', u'_id': 2, u'tags': {u'dc': u'ugr'}}, 
                    {u'priority': 2.0, u'host': u'moddb-mongo08i.yandex.ru:27700', u'_id': 3, u'tags': {u'heavy': u'1', u'dc': u'sas'}}
                ], 
           u'settings': {u'getLastErrorModes': { u'sortOfImportant': {u'dc': 2}, 
                                                 u'veryImportant': {u'dc': 3}, 
                                                 u'backedUp': {u'backup': 1, u'dc': 2}}}
        } ]

class Test(unittest.TestCase):
    def test_run_once(self):
        config = DEFAULTS
        self.assertEqual(countVotes(body1), 3)
        self.assertEqual(checkTags(body1), 0)
        self.assertEqual(checkWriteConc(body1), 0)
        self.assertEqual(checkPrioruty(body1), 3)

def getConfig(filename):
    '''
        Возвращает опции, полученные из конфига. Недостающие опции берутся из DEFAULT.
    '''
    options = DEFAULTS
    parser = ConfigParser()
    if filename and os.path.exists(filename):
	parser.read(filename)
        for key, value in parser.items('main'):
            if options.has_key(key) and type(options[key]) == int: 
                options[key] = int(value)
            elif options.has_key(key) and type(options[key]) == list:
                options[key] = value.split(',')
            else:
                options[key] = str(value)
    if DEBUG: print 'Connect to mongodb:\n {0}'.format(options)
    return options

def getReplConf():
    '''
        Функция для получения настроки конфигурации реплики.
        Читает значение конфига программы (по умолчанию берется DEFAULTS). 
        В ответе возвращается словарь(хеш) с конфигурацией репликации сервера.
    '''
    
    conn = pymongo.MongoClient(config["mongo_host"], int(config["mongo_port"]), replicaSet=None)
    conn.admin.authenticate(config["mongo_user"], config["mongo_password"])

    db = conn.get_database("local")
    coll = db.get_collection("system.replset")

    replConf = [ i for i in coll.find() ]
    conn.close()

    if len(replConf) == 0: raise ValueError("Конфиг rs.conf() пуст.") 

    return replConf

def countVotes(rc):
    '''
        Функция для подсчета голосующих реплик.
        Принимает словарь(хеш) с конфигурацией репликации сервера. 
        В ответе возвращает сумму значений поля votes.
    '''
    listVotes = [ m.get('votes', 1) for m in rc[0]["members"] 
                                      if rc[0]["_id"] == config["mongo_replica"] ]
    if len(listVotes) == 0:
        raise ValueError("Empty listVotes in countVotes")
    cntVotes = reduce( lambda x, y: x+y, listVotes )
    if cntVotes < config["need_votes"]:
        raise ValueError("Votes smaller than {0}: current {1}".format(config["need_votes"], cntVotes))
    if len(set(listVotes)) > 2:
        _diff = list(set(listVotes)-set((0, 1)))
        raise ValueError("Votes can be 0 or 1. Found: {0}".format(_diff))
    return reduce( lambda x, y: x+y, listVotes )

def checkPrioruty(rc):
    '''
        Функция проверяет, что количество машин с priority >= need_votes, исключая машины с backup.
        Принимает словарь(хеш) от mongodb сервера.
        В ответе возвращает количество машин с выставленным флагом.
    '''
    listPriority = [ m.get('priority', 1) for m in rc[0]["members"] 
                                            if rc[0]["_id"] == config["mongo_replica"] and
                                               not m["tags"].has_key("backup") ]
    if len(listPriority) < config["need_votes"]:
        raise ValueError("Count priority servers smaller {0}: {1}".format(config["need_votes"], len(listPriority)))
    return len(listPriority)

def checkWriteConc(rc):
    '''
        Функция проверяет settings.getLastErrorModes.
        Принимает словарь(хеш) от mongodb сервера.
        В ответе возвращает количество отсутствующих значений getLastErrorModes.
    '''    
    listModes = rc[0]['settings']['getLastErrorModes']
    g = set(config["err_modes"])-set(listModes.keys())
    if len(g) > 0: raise ValueError('Not found getLastErrorModes: "{0}"'.format(', '.join(g)))
    return len(g)

def checkTags(rc):
    '''
        Функция проверяет тэги, выставленные у реплик.
        Принимает словарь(хеш), содержащий конфигурацию mongodb сервера.
        В ответе возвращает количество потерянных тэгов.
    '''
    dictTags = dict([ (m["host"], m["tags"].keys()) for m in rc[0]["members"]])
    emptyTags = list() 

    for host in dictTags:
        t = (set(config["all_tags"]) - set(dictTags[host]))
        if len(t) > 0: emptyTags.append('{0} dont have tag(s): "{1}"'.format(host, ', '.join(t)))

    uniqTags = set(reduce(lambda x, y: x+y, dictTags.values()))
    e = set(config["once_tags"]) - set(uniqTags)
    if len(e) > 0: emptyTags.append('Dont found uniq tag(s): "{0}"'.format(', '.join(e))) 

    if len(emptyTags) > 0: raise ValueError(', '.join(emptyTags))
    return len(emptyTags)

def main():
    try:
        rc = getReplConf()
        cntVotes = countVotes(rc)
        cntTags = checkTags(rc)
        cntWc = checkWriteConc(rc)
        cntPriority = checkPrioruty(rc)
    except Exception as err:
        print "1;{0}".format(err)
        if DEBUG: raise
    else:
        print "0;OK. Count votes/priority: {0}/{1}, empty tags/writeConcern: {2}/{3}.".format(cntVotes, cntPriority, cntTags, cntWc)

if __name__ == '__main__':

    # Go to executable dir - to make all pathnames work
    os.chdir(os.path.dirname(os.path.realpath(__file__)))

    # Get command-line options
    opt_parser = OptionParser()
    opt_parser.add_option("-f", "--filename", dest="filename", help="configuration file", type="string", default=None)
    opt_parser.add_option("-d", "--debug", dest="debug", help="enable debug messages", action="store_true", default=False)
    opt_parser.add_option("-t", "--run-test", dest="test", help="run unittest", action="store_true", default=False)
    (opts, args) = opt_parser.parse_args()
    if opts.filename and opts.test: 
        print "Run test without -f/--filename."
        sys.exit(1)
    if opts.debug: DEBUG = True

    config = getConfig(opts.filename)

    if opts.test:
        sys.argv[1:] = args
        unittest.main()
    else:
        main()
