#!/bin/bash
#
### BEGIN INIT INFO
# Provides:          mysql
# Required-Start:    $syslog mysql-ndb-mgm
# Required-Stop:     $syslog
# Should-Start:      $local_fs $remote_fs $network $named $time
# Should-Stop:       $local_fs $remote_fs $network $named $time
# Default-Start:     2 3 4 5
# Default-Stop:      0 1 6
# Short-Description: Start and stop the mysql database server daemon
# Description:       Controls the main MySQL database server daemon "mysqld"
#                    and its wrapper script "mysqld_safe".
### END INIT INFO
#
set -e
set -u
${DEBIAN_SCRIPT_DEBUG:+ set -v -x}

test -x /usr/sbin/mysqld || exit 0

. /lib/lsb/init-functions

SELF=$(cd $(dirname $0); pwd -P)/$(basename $0)
NAME={{ db_instance_name }}
CONF="/etc/mysql/$NAME.cnf"
MYADMIN="/usr/bin/mysqladmin --defaults-file=$CONF"
STARTUP_TIMEOUT=900

test -d "/var/run/mysqld.$NAME" || mkdir "/var/run/mysqld.$NAME" && chown mysql:mysql "/var/run/mysqld.$NAME"

# priority can be overriden and "-s" adds output to stderr
ERR_LOGGER="logger -p daemon.err -t /etc/init.d/mysql.$NAME -i"

# Safeguard (relative paths, core dumps..)
cd /
umask 077

# mysqladmin likes to read /root/.my.cnf. This is usually not what I want
# as many admins e.g. only store a password without a username there and
# so break my scripts.
export HOME=/etc/mysql/

## Fetch a particular option from mysql's invocation.
#
# Usage: void mysqld_get_param option
mysqld_get_param() {
	/usr/sbin/mysqld --defaults-file=$CONF --print-defaults \
		| tr " " "\n" \
		| grep -- "--$1" \
		| tail -n 1 \
		| cut -d= -f2
}

## Do some sanity checks before even trying to start mysqld.
sanity_checks() {
  # check for config file
  if [ ! -r $CONF ]; then
    log_warning_msg "$0: WARNING: $CONF cannot be read. See README.Debian.gz"
    echo                "WARNING: $CONF cannot be read. See README.Debian.gz" | $ERR_LOGGER
  fi

  # check for buggy options
  if grep -q ^expire.logs.days $CONF  &&  ! grep -q ^log.bin $CONF; then
    log_failure_msg "$0: ERROR: Using expire_logs_days without log_bin crashes the server. See README.Debian.gz"
    echo                "ERROR: Using expire_logs_days without log_bin crashes the server. See README.Debian.gz" | $ERR_LOGGER
    exit 1
  fi

  # check for diskspace shortage
  datadir=`mysqld_get_param datadir`
  if LC_ALL=C BLOCKSIZE= df --portability $datadir/. | tail -n 1 | awk '{ exit ($4>4096) }'; then
    log_failure_msg "$0: ERROR: The partition with $datadir is too full!"
    echo                "ERROR: The partition with $datadir is too full!" | $ERR_LOGGER
    exit 1
  fi

  #
  for conffile in $CONF /etc/mysql/$NAME.conf.d/*.cnf; do
    if grep -q FIXME $conffile 2>/dev/null; then
      log_failure_msg "$0: ERROR: Unfixed conf file: $conffile"
      echo                "ERROR: Unfixed conf file: $conffile" | $ERR_LOGGER
      exit 1
    fi
  done

}

## Checks if there is a server running and if so if it is accessible.
#
# check_alive insists on a pingable server
# check_dead also fails if there is a lost mysqld in the process list
#
# Usage: boolean mysqld_status [check_alive|check_dead] [warn|nowarn]
mysqld_status () {
    ping_output=`$MYADMIN ping 2>&1`; ping_alive=$(( ! $? ))

    ps_alive=0
    pidfile=`mysqld_get_param pid-file`
    if [ -f "$pidfile" ] && ps `cat $pidfile` >/dev/null 2>&1; then ps_alive=1; fi
    
    if [ "$1" = "check_alive"  -a  $ping_alive = 1 ] ||
       [ "$1" = "check_dead"   -a  $ping_alive = 0  -a  $ps_alive = 0 ]; then
	return 0 # EXIT_SUCCESS
    else
  	if [ "$2" = "warn" ]; then
  	    echo -e "$ps_alive processes alive and '$MYADMIN ping' resulted in\n$ping_output\n" | $ERR_LOGGER -p daemon.debug
	fi
  	return 1 # EXIT_FAILURE
    fi
}

#
# main()
#

case "${1:-''}" in
  'start')
    sanity_checks;
    # Start daemon
    log_daemon_msg "Starting MySQL database server" "mysqld"
    if mysqld_status check_alive nowarn; then
       log_progress_msg "already running"
       log_end_msg 0
    else
        /usr/bin/mysqld_safe --defaults-file=$CONF > /dev/null 2>&1 &
        # 6s was reported in #352070 to be too few when using ndbcluster
        for i in `seq 1 $STARTUP_TIMEOUT`; do
                sleep 1
            if mysqld_status check_alive nowarn ; then break; fi
        log_progress_msg "."
        done
        if mysqld_status check_alive warn; then
                log_end_msg 0
            # Now start mysqlcheck or whatever the admin wants.
            output=`/etc/mysql/$NAME-start`
        [ -n "$output" ] && log_action_msg "$output"
        else
            log_end_msg 1
        log_failure_msg "Please take a look at the syslog"
        fi
    fi

    # Some warnings
        if $MYADMIN variables | egrep -q have_bdb.*YES; then
            echo "BerkeleyDB is obsolete, see /usr/share/doc/mysql-server-5.0/README.Debian.gz" | $ERR_LOGGER -p daemon.info
        fi
        if [ -f /etc/mysql/debian-log-rotate.conf ]; then
            echo "/etc/mysql/debian-log-rotate.conf is obsolete, see /usr/share/doc/mysql-server-5.0/NEWS.Debian.gz" | $ERR_LOGGER -p daemon.info
        fi
    ;;


  # Операция bootstrap-pxc доступна только для серверов с Percona XtraDB Cluster.
  # Она нужна для того, чтобы запустить сервер, который создаст новый кластер.
  # Это нужно делать, когда кластера ещё нет; если он уже есть, нужно
  # записать хотя бы один другой узел в wsrep_cluster_address в my.cnf
  # и запустить с помощью start.
  'bootstrap-pxc')
	sanity_checks;
	# Start daemon
	log_daemon_msg "Starting MySQL database server" "mysqld"
	if mysqld_status check_alive nowarn; then
	   log_progress_msg "already running"
	   log_end_msg 0
	else
  	    /usr/bin/mysqld_safe --defaults-file=$CONF --wsrep-new-cluster > /dev/null 2>&1 &
	    # 6s was reported in #352070 to be too few when using ndbcluster
	    for i in `seq 1 $STARTUP_TIMEOUT`; do
                sleep 1
	        if mysqld_status check_alive nowarn ; then break; fi
		log_progress_msg "."
	    done
	    if mysqld_status check_alive warn; then
                log_end_msg 0
	        # Now start mysqlcheck or whatever the admin wants.
	        output=`/etc/mysql/$NAME-start`
		[ -n "$output" ] && log_action_msg "$output"
	    else
	        log_end_msg 1
		log_failure_msg "Please take a look at the syslog"
	    fi
	fi

	# Some warnings
        if $MYADMIN variables | egrep -q have_bdb.*YES; then
            echo "BerkeleyDB is obsolete, see /usr/share/doc/mysql-server-5.0/README.Debian.gz" | $ERR_LOGGER -p daemon.info
        fi
        if [ -f /etc/mysql/debian-log-rotate.conf ]; then
            echo "/etc/mysql/debian-log-rotate.conf is obsolete, see /usr/share/doc/mysql-server-5.0/NEWS.Debian.gz" | $ERR_LOGGER -p daemon.info
        fi
	;;


  'stop')
	# * As a passwordless mysqladmin (e.g. via ~/.my.cnf) must be possible
	# at least for cron, we can rely on it here, too. (although we have 
	# to specify it explicit as e.g. sudo environments points to the normal
	# users home and not /root)
	log_daemon_msg "Stopping MySQL database server" "mysqld"
	if ! mysqld_status check_dead nowarn; then
	  set +e
	  shutdown_out=`$MYADMIN shutdown 2>&1`; r=$?
	  set -e
	  if [ "$r" -ne 0 ]; then
	    log_end_msg 1
	    [ "$VERBOSE" != "no" ] && log_failure_msg "Error: $shutdown_out"
	    log_daemon_msg "Killing MySQL database server by signal" "mysqld"
	    killall -15 mysqld
            server_down=
	    for i in 1 2 3 4 5 6 7 8 9 10; do
              sleep 1
              if mysqld_status check_dead nowarn; then server_down=1; break; fi
            done
          if test -z "$server_down"; then killall -9 mysqld; fi
	  fi
        fi

        if ! mysqld_status check_dead warn; then
	  log_end_msg 1
	  log_failure_msg "Please stop MySQL manually and read /usr/share/doc/mysql-server-5.0/README.Debian.gz!"
	  exit -1
	else
	  log_end_msg 0
        fi
	;;

  'restart')
	set +e; $SELF stop; set -e
	$SELF start 
	;;

  'reload'|'force-reload')
  	log_daemon_msg "Reloading MySQL database server" "mysqld"
	$MYADMIN reload
	log_end_msg 0
	;;

  'status')
	if mysqld_status check_alive nowarn; then
	  log_action_msg "$($MYADMIN version)"
	else
	  log_action_msg "MySQL is stopped."
	  exit 3
	fi
  	;;

  *)
	echo "Usage: $SELF start|stop|restart|reload|force-reload|status"
	exit 1
	;;
esac

