package main
// Программа для выборки уникальных IP по clickhouse. Вывод отправляется в сокет графита(localhost:42000).

import (
    "fmt"
    "net"
    "net/url"
    "flag"
    "time"
    "errors"
    "bytes"
    "encoding/json"
    "net/http"
    "strings"
    "io/ioutil"
)

const (
    host = "localhost"
    port = 8123 
    graphiteHost = "localhost"
    graphitePort = 42000
)

var (
    services sliceString
    values ultraData
    database *string = flag.String("db", "ppclogpusher", "Database name for query.")
    table *string = flag.String("tb", "logs_direct", "Table name for query.")
    debug *bool = flag.Bool("debug", false, "Enable debugging.")
)

type sliceString []string

func (s *sliceString) String() string {
    return fmt.Sprintf("%s", *s)
}

func (s *sliceString) Set(line string) error {
    for _, value := range strings.Split(line, ",") {
        *s = append(*s, value)
    }
    if len(*s) == 0 { 
        return errors.New("Empty list values!")
    }
    return nil 
}


type ultraData struct {
    response
    service string
    graphiteBuff *bytes.Buffer
}

type response struct {
    Meta responseMetaList `json: meta`
    Data responseDataList `json: data`
    Rows int `json: rows`
} 

type responseDataList []responseData
type responseMetaList []responseMeta

type responseData struct {
    Times int `json: times`
    Uniqip string `json: uniqip`
    Total string `json: total`
}

type responseMeta struct {
    Name string `json: name`
    Type string `json: type`
}

func (r *ultraData) requestClickhouse() (err error) {
    // Получаем ответ с количеством уникальных IP за интервал в 10 минут.  
    cTime := time.Now().Add(-1 *time.Hour)
    strReq := `SELECT (600*floor(toUnixTimestamp(date)/600)) AS times, count() AS total, uniq(src_ip) AS uniqip
               FROM %s.%s WHERE toDate(date)='%d-%02d-%02d' AND toHour(date)=%d AND vhost='%s' 
               GROUP BY times FORMAT JSON`
    reqData := fmt.Sprintf(strReq, *database, *table, cTime.Year(), cTime.Month(), cTime.Day(), cTime.Hour(), r.service)
    requestByte := url.Values{"query": {reqData}}
    reqUrl := fmt.Sprintf("http://%s:%d/?%s", host, port, requestByte.Encode())
    resp, err := http.Get(reqUrl)
    if err != nil {
        fmt.Printf("[requestClickhouse] Error: %s\n", err)
        return 
    }
    defer resp.Body.Close()
    respData, _ := ioutil.ReadAll(resp.Body)
    if resp.StatusCode != 200 {
        fmt.Printf("[requestClickhouse] Error request. Response code: %d\n", resp.StatusCode)
        fmt.Printf("[requestClickhouse] Request %s, тело ответа: %s\n", reqData, respData)
        return 
    }

    if err = json.Unmarshal(respData, &values); err != nil {
        fmt.Printf("[requestClickhouse] Error unmarshal. Error: %s\n", err)
        return 
    }

    if *debug { fmt.Printf("[requestClickhouse] %s\n", values) }
    return nil
}

func (r *ultraData) formatGraphite() (err error) {
    // Создаем буффер и складываем все метрики в него.
    buff := bytes.NewBuffer(nil)
    name := strings.Replace(r.service, ".", "_", -1)
    if *debug { fmt.Printf("[formatGRaphite] %T %#v\n", buff, r.Data) }
    for _, val1 := range r.Data {
        // Сумма всех запросов от IP адресов
        _, err = buff.Write([]byte(fmt.Sprintf("one_min.%s.countip.total %s %d\n", name, val1.Total, val1.Times)))
        // Сумма уникальных IP адресов
        _, err = buff.Write([]byte(fmt.Sprintf("one_min.%s.countip.uniq %s %d\n", name, val1.Uniqip, val1.Times)))
    }
    r.graphiteBuff = buff
    return
}

func (r *ultraData) sendGraphite() (err error) {
    // Пробуем подключится к graphite-sender и передать []bytes в его сокет.
    tcpAddr, err := net.ResolveTCPAddr("tcp", fmt.Sprintf("%s:%d", graphiteHost, graphitePort))
    if err != nil {
        fmt.Printf("[sendGraphite] Error resolve: %s\n", err)
        return
    }

    conn, err := net.DialTCP("tcp", nil, tcpAddr)
    if err != nil { 
        fmt.Printf("[sendGraphite] Error connect: %s\n", err)
        return
    }
    defer conn.Close()

    data, err := ioutil.ReadAll(r.graphiteBuff)
    if err != nil {
        fmt.Printf("[sendGraphite] Error read buffer: %s\n", err)
        return errors.New("Error read buffer.")
    }
    if len(data) == 0 {
        fmt.Printf("[sendGraphite] Nothing write to socket graphite sender.\n")
        return errors.New("Empty graphiteBuff list.")
    }
    if *debug { fmt.Printf("[sendGraphite] Data: %s", data) }

    cnt, err := conn.Write(data)
    if cnt != len(data) { 
        fmt.Printf("[sendGraphite] Write data != len buffer data: %d != %d.\n", cnt, len(data))
        return errors.New("Count wrtite data != len buffer data.")
    }
    if *debug { fmt.Printf("[sendGraphite] Count write bytes: %d\n", cnt) }
    if err != nil {
        fmt.Printf("[sendGraphite] Error write graphite-sender socket: %s\n", err)
    }

    return
}

func main() {
    flag.Var(&services, "services", "Список сервисов для генерации UniqIP.")
    flag.Parse()

    if *debug { fmt.Println(services) }
    for _, values.service = range services {
        // Получаем данные из clickhouse и кладем их в values.Data
        if err := values.requestClickhouse(); err != nil { continue }
        if *debug { fmt.Println(values.Data) }
        // Данные из values.Data преобразуем в формат graphite-sender. Результат кладем в values.graphiteBuff.
        if err := values.formatGraphite(); err != nil { continue }
        // Отправляем данные в сокет graphite-sender.
        if err := values.sendGraphite(); err != nil { continue }
    }
}