#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
Читает juggler-sdk скрипты из указанного svn-пути
Запускает с dry-run, если успешно - запускает еще раз с указанными аргументами

Пример:
directadmin-make-juggler.py --svn-path svn.yandex.ru/direct-utils/directadmin-juggler-checks/checks --svn-user ... --svn-key /etc/... --check-first --apply -vvp
"""

from __future__ import print_function

import argparse
import json
import logging
import os
import sys
import shutil
import yaml
from subprocess import Popen, check_call, check_output, PIPE, STDOUT
from tempfile import mkdtemp

from direct_juggler import juggler as dj


def die(msg):
    logging.info(msg + ' - exiting')
    sys.exit(1)


def read_file(fname):
    with open(fname, 'r') as f:
        return f.read()


def jdumps(jdict, pretty=False):
    if pretty:
        return '\n' + json.dumps(jdict, sort_keys=True, ensure_ascii=False, indent=4, separators=(',', ': '))
    return json.dumps(jdict, sort_keys=True, ensure_ascii=False)


def make_svn_url(svn_path, svn_rev, svn_user):
    url = 'svn+ssh://'
    if svn_user:
        url += svn_user + '@'
    url += svn_path
    if svn_rev:
        url += '@' + svn_rev
    return url


def svn_export(svn_path, svn_rev, svn_user, svn_binary, env):
    logging.info('svn: get configs from path %s, rev %s' % (svn_path, svn_rev))

    cwd = mkdtemp()
 
    svn_url = make_svn_url(svn_path, svn_rev, svn_user)
    cmd = [svn_binary, 'export', '--force', svn_url, cwd]
    logging.info('svn: run cmd ' + ' '.join(cmd) + ' with env ' + str(env))
    p = Popen(cmd, stdout=PIPE, stderr=PIPE, cwd=cwd, env=env)
    out, err = p.communicate()
    logging.info('svn: cmd finished with retcode: %s\nstdout:\n%s\nstderr:\n%s' % (p.returncode, out, err))

    if p.returncode != 0:
        shutil.rmtree(cwd)
        die('bad svn return code')

    return cwd


def parse_args():
    parser = argparse.ArgumentParser(formatter_class=argparse.ArgumentDefaultsHelpFormatter, description=__doc__)
    parser.add_argument('-p', '--svn-path', required=True, help='')
    parser.add_argument('-u', '--svn-user', help='пользователь, с которым ходить в svn')
    parser.add_argument('-k', '--svn-key', help='путь до ключа, с которым ходить в svn')
    parser.add_argument('-s', '--svn-binary', default='/usr/bin/svn', help='путь к бинарнику svn')
    parser.add_argument('-d', '--check-first', default=False, action='store_true', help='перед запуском команд проверить, что они не падают в --dry-run')
    args, extra = parser.parse_known_args()

    return args, extra


def main():
    args, juggler_args = parse_args()

    logging.basicConfig(level=logging.DEBUG, stream=sys.stderr,
                        format='[%(asctime)s]\t%(message)s')
    logging.info('running with args: ' + jdumps(vars(args)))

    env = None
    if args.svn_key:
        env = {'SVN_SSH': 'ssh -i ' + args.svn_key}
    path = svn_export(args.svn_path, 'HEAD', args.svn_user, args.svn_binary, env)
    logging.info('exported to ' + path)

    failed = 0
    total = 0
    for item in os.listdir(path):
        fpath = path + '/' + item
        logging.info('processing file ' + fpath)
        if not os.path.isfile(fpath) or not fpath.endswith('.py'):
            logging.info('skipping file ' + fpath)
            continue

        cmd_prefix = ['/usr/bin/env', 'python', '-B', fpath]
        cmd_check = cmd_prefix + [x for x in juggler_args if x != '--apply']
        cmd = cmd_prefix + juggler_args

        total += 1
        if args.check_first:
            try:
                logging.info('[%s]: check cmd (dry-run): %s' % (item, ' '.join(cmd_check)))
                check_call(cmd_check, cwd=path)
            except Exception as e:
                logging.info('[%s]: skipping: %s %s' % (item, type(e), e))
                failed += 1
                continue

        logging.info('[%s]: run cmd: %s' % (item, ' '.join(cmd)))
        check_call(cmd)

    shutil.rmtree(path)

    event = {
        'service': 'directadmin-make-juggler',
        'status': 'OK',
        'description': 'OK'
    }
    if failed:
        event['status'] = 'WARN' if failed < total else 'CRIT'
        event['description'] = 'some juggler scripts failed, see /var/log/ppcback-directadmin-cron/directadmin-make-juggler.log*'

    logging.info('Finished with: ' + jdumps(event))
    dj.queue_events([event])


if __name__ == '__main__':
    main()
