# -*- coding: utf-8 -*-

from juggler_sdk import JugglerApi, Check, Child, FlapOptions, NotificationOptions
from juggler_sdk.check_sync import CheckDifference, FullCheckDifference
import copy
import os
import json

import logging
import logging.handlers
logging.getLogger(__name__).addHandler(logging.NullHandler())

class AggrChecks(object):
    '''
    Хранилище для агрегатных проверок
    Для удобства, children в формате make_aggr_checks можно задать прямо в конструкторе, и затем переопределять в методах
    '''
    def __init__(self, namespace, host, token, default_kwargs={}, children=None, aggr_child_tags=[]):
        self.namespace = namespace
        self.aggr_host = host
        self.token = token

        # не надо добавлять в default_check поле host!
        # иначе, все child агрегаты будут заводиться на этот хост, по-умолчанию он генерируется из aggr_host + suffix
        default_kwargs.pop('host', None)
        self.default_check = hash_merge(default_kwargs, {'namespace': namespace})

        self.children = children
        self.aggr_child_tags = aggr_child_tags
        self.all_checks = []


    def append(self, **kwargs):
        '''
        Добавляет обычную juggler_sdk Check, смешанную с default_check
        Использует juggler_sdk Check kwargs, ничего не знает про специальный формат children с suffix
        '''
        self.all_checks.append(Check(**hash_merge(self.default_check, kwargs)))


    def append_aggr(self, **child_kwargs):
        '''
        Добавляет агрегатную проверку по заданным children (в формате make_aggr_checks)
        Некоторые параметры неявно применяются к верхнему агрегату (tags)
        Пример: append_aggr(service='ssh', active='ssh', **UNREACH_PING)

        Схема агрегации: смотри make_aggr_checks

        self.aggr_host:child_kwargs['service']
        <- child_kwargs.get('host', self.aggr_host + '.' + suffix):child_kwargs.get('aggregator_service', child_kwargs['service'])
           <- suffix:child_kwargs['service']

        direct.prod_ppcdata:ssh <- direct.prod_ppcdata.ppcdata15:ssh <- ppcdata15:ssh
        '''
        children = child_kwargs.pop('children', None)
        children = children if children is not None else self.children

        # потому что обычно хочется менять теги верхнего уровня агрегации, для более сложных конструкций есть _full вариант метода
        aggr_kwargs = {}
        tags = child_kwargs.pop('tags', None)
        if tags is not None:
            aggr_kwargs['tags'] = tags

        self.append_aggr_full(children=children, child_kwargs=child_kwargs, aggr_kwargs=aggr_kwargs)


    def append_aggr_single(self, **child_kwargs):
        '''
        Добавляет проверки без верхнеуровневого агрегатного хоста на хост, указанный в параметрах
        По-умолчанию host - агрегатный хост, заданный при создании объекта

        Схема агрегации: смотри make_aggr_checks

        child_kwargs.get('host', self.aggr_host + '.' + suffix):child_kwargs.get('aggregator_service', child_kwargs['service'])
        <- suffix:child_kwargs['service']

        direct.prod_ppcdata:slave_ok_ppcdata15 <- ppcdata15:mysql_slave_running_ppcdata15
        или
        direct.prod_ppcdata.ppcdata15:ssh <- ppcdata15:ssh
        '''
        children = child_kwargs.pop('children', None)
        children = children if children is not None else self.children

        host = child_kwargs.pop('host', None)
        host = host if host is not None else self.aggr_host
        child_kwargs['host'] = host

        # мерджим с default_check, чтобы подхватить теги, предназначенные для верхнеуровневого агрегата
        # иначе они в aggr_full затрутся self.aggr_child_tags
        child_kwargs = hash_merge(self.default_check, child_kwargs)

        self.append_aggr_full(children=children, child_kwargs=child_kwargs, aggr_kwargs=dict(do_aggr_host=False))


    def append_aggr_full(self, child_kwargs, aggr_kwargs={}, children=None):
        '''
        Расширенная версия append_aggr

        Схема агрегации: смотри make_aggr_checks
        '''
        # заменяем теги из default_checks
        child_kwargs = hash_merge(self.default_check, {'tags': self.aggr_child_tags}, child_kwargs)
        aggr_kwargs = hash_merge(self.default_check, {'host': self.aggr_host, 'service': child_kwargs['service']}, aggr_kwargs)
        children = children if children is not None else self.children
        log_info_vvv('append_aggr_full:\ndefault_check: %s\nchild_kwargs: %s\naggr_kwargs: %s\nchildren: %s' %
                (jdumps(self.default_check, True), jdumps(child_kwargs, True), jdumps(aggr_kwargs, True), jdumps(children, True)))

        checks = make_aggr_checks(children, child_kwargs, aggr_kwargs)
        if len(checks) == 0:
            # пустой список в children может приводить к такому, это может быть удобно для удаления проверок
            # но обрабатывать такое лучше ближе к самим проверкам, поэтому warning (тем более, что пустые children в juggler имеют другой смысл)
            log_info("No checks for children: %s, child_kwargs: %s, aggr_kwargs: %s" % (children, child_kwargs, aggr_kwargs))
        self.all_checks += checks


    def apply(self, dry_run=True, pretty=False):
        '''
        Применяет все проверки. Возвращает True, если все данные в juggler соответствуют применяемым проверкам
        '''
        res = apply_checks(self.all_checks, mark=self.aggr_host, token=get_juggler_token(self.token), dry_run=dry_run, pretty=pretty)
        return res['changed'] == 0


### helpers

def get_juggler_token(filename):
    with open(os.path.expanduser(filename), 'r') as f:
        token = f.read().strip()
    return token


def hash_merge(*dict_args):
    result = {}
    for d in dict_args:
        for key, value in d.iteritems():
            if isinstance(value, dict):
                result[key] = hash_merge(result.get(key, {}), value)
            else:
                result[key] = copy.deepcopy(value)
    return result

def juggler_merge(*dict_args):
    result = {}
    for d in dict_args:
        for key, value in d.iteritems():
            if isinstance(value, dict):
                result[key] = hash_merge(result.get(key, {}), value)
            elif key == 'tags' and (isinstance(value, list) or isinstance(value, tuple)):
                result[key] = array_merge(result.get(key, []), value)
            else:
                result[key] = copy.deepcopy(value)
    return result

def array_merge(*list_args):
    result = set()
    for l in list_args:
        for value in l:
            result.add(copy.deepcopy(value))
    return list(result)

def hash_replace(*dict_args):
    result = {}
    for d in dict_args:
        for key, value in d.iteritems():
            result[key] = copy.deepcopy(value)
    return result


#def hash_render(template, **kwargs):
#    '''
#    Заменить в словаре строки через .format(**kwargs)
#    Может быть удобно для more_than_limit агрегатов
#    '''
#    result = {}
#    for key, value in template.iteritems():
#        if isinstance(value, dict):
#            result[key] = hash_render(value, **kwargs)
#        elif isinstance(value, list):
#            result[key] = [ hash_render(x, **kwargs) for x in value ]
#        elif isinstance(value, str):
#            result[key] = value.format(**kwargs)
#        else:
#            result[key] = copy.deepcopy(value)
#    return result


def make_aggr_checks(children, child_kwargs, aggr_kwargs={}):
    '''
    Возвращает агрегатные проверки над набором шардов (кондукторных групп, и тд), и
    общие агрегаты над ними всеми.

    children:
        list вида
            [{'suffix': 'modlog', 'children': [{'host': 'mysql_modlog', 'group_type': 'CGROUP', ...}]}, ...],
        где:
            suffix - суффикс для генерации различных сервисов дочерних проверок, например, mysql_slave_running_modlog
            children - список словарей с kwargs для juggler_sdk Child

    child_kwargs: juggler_kwargs для дочерних проверок, возможны дополнительные параметры, см. ниже
    aggr_kwargs: juggler_kwargs для общего агрегата, возможны дополнительные параметры

    Схема агрегации:
         aggr_kwargs['host']:aggr_kwargs['service']
         <- child_kwargs.get('host', self.aggr_host + '.' + suffix):child_kwargs.get('aggregator_service', child_kwargs['service'])
            <- suffix:child_kwargs['service']

        direct.prod_ppcdata:slave_ok <- direct.prod_ppcdata.ppcdata15:slave_ok <- ppcdata15:mysql_slave_running_ppcdata15

    Параметры child_kwargs:
        service: обязательный параметр, имя сервиса (или функция) с физического хоста
        host: по-умолчанию - aggr_kwargs['host'] + '.' + suffix, можно переопределить на константную строку или функцию
        aggregator_service: по-умолчанию - child_kwargs['service'], имя сервиса (или функция) для дочерних агрегатов
        unreach_service: по-умолчанию - None, имя (или функция) проверки, от которой зависит дочерняя проверка
        ...любые параметры juggler_sdk...

    Параметры aggr_kwargs:
        host: обязательный параметр, имя агрегатного хоста (строка)
        service: обязательный параметр, имя агрегатного сервиса (строка)
        do_aggr_host: по-умолчанию - True, создавать ли агрегатный хост (иногда хочется создать только дочерние хосты)
        ...любые параметры juggler_sdk...

    Функции в качестве параметров вызываются с одним аргументом - текущим суффиксом проверки. Например:
    lambda suff: 'mysql_slave_running_' + suff

    Пример вызова:
    for service in ['lfw-', 'lm-schema-']:
        children = ['ppcmonitor']
        make_aggr_checks(cgroups_to_children(children),
                         dict(service=lambda suff: service + suff, aggregator_kwargs={'unreach_mode': 'skip'}, unreach_service=lambda suff: 'mysql_alive_' + suff),
                         dict(service=service[0:-1]))

    '''
    aggr_host = aggr_kwargs.get('host')
    do_aggr_host = aggr_kwargs.pop('do_aggr_host', True)
    children = children_to_dict(children)

    # удаляем (чтобы потом не было двух service), но падаем, если вообще нет ключа
    child_svc = child_kwargs.pop('service')
    # удаляем наши кастомные параметры, juggler_api их не примет
    child_aggr_svc = child_kwargs.pop('aggregator_service', child_svc)
    child_unreach_svc = child_kwargs.pop('unreach_service', None)
    # host по-умолчанию генерируется из aggr_host + '.' + suff
    child_host = child_kwargs.pop('host', lambda s: aggr_host + '.' + s)

    all_checks = []
    aggr_children = []
    for suff in sorted(children.keys()):
        # чтобы генерить сервис от children suffix
        svc = child_svc(suff) if callable(child_svc) else child_svc
        aggr_svc = child_aggr_svc(suff) if callable(child_aggr_svc) else child_aggr_svc

        child_svc_children = [Child(**hash_merge({'service': svc}, x)) for x in children.get(suff)]

        # дочерний агрегат (direct.prod_ppcdata.ppcdata15)
        aggr_child = child_host(suff) if callable(child_host) else child_host
        # зависимость от direct.prod_ppcdata15:mysql_alive_ppcdata15
        if child_unreach_svc is not None:
            unreach_svc = child_unreach_svc(suff) if callable(child_unreach_svc) else child_unreach_svc
            unreach_svc = copy.deepcopy(unreach_svc)
            aggregator_kwargs = child_kwargs.get('aggregator_kwargs', {})

            if isinstance(unreach_svc, str):
                unreach_svc = [{'check': ':' + unreach_svc}]
            for unreach_obj in unreach_svc:
                if unreach_obj['check'].startswith(':'):
                    unreach_obj['check'] = aggr_child + unreach_obj['check']

            aggregator_kwargs = hash_merge(aggregator_kwargs, {'unreach_service': unreach_svc})
            child_kwargs['aggregator_kwargs'] = aggregator_kwargs

        # добавляем дочернюю проверку в список
        all_checks.append(Check(host=aggr_child, service=aggr_svc, children=child_svc_children, **child_kwargs))
        # сохраняем в список для родительской агрегатной проверки
        aggr_children.append(Child(host=aggr_child, service=aggr_svc))

    if do_aggr_host and aggr_children:
        all_checks.append(Check(children=aggr_children, **aggr_kwargs))
    return all_checks


'''
Преобразует список формата:
['ppcdict', {'suffix': 'modlog', 'children': [{'host': 'mod_ng_mysql_modlog', 'group_type': 'CGROUP'}]}]
в список
[{'suffix': 'ppcdict', children: [{'host': 'ppcdict', ...
'''
def cgroups_to_children(children):
    assert isinstance(children, list) or isinstance(children, tuple)

    res = []
    for child in children:
        if isinstance(child, dict):
            res.append(child)
        elif isinstance(child, str):
            # https://wiki.yandex-team.ru/sm/juggler/group-types/
            res.append({'suffix': child, 'children': [{'host': child, 'group_type': 'CGROUP'}]})
        elif isinstance(child, unicode):
            res.append({'suffix': str(child), 'children': [{'host': str(child), 'group_type': 'CGROUP'}]})
    return res


def children_to_dict(children):
    res = {}
    for child in children:
        if child['suffix'] in res:
            raise Exception("Suffix '" + child['suffix'] + "' already exists!")
        res[child['suffix']] = child['children']
    return res


def process_upsert_result(res, check, pretty):
    changes_type = 'not_changed'
    if res.changed:
        changes_type = 'updated'
        if isinstance(res.diff, FullCheckDifference):
            changes_type = 'inserted'
        log_info_v("check %s: (%s) %s:%s, diff: %s" % (changes_type, check.namespace, check.host, check.service, jdumps(res.diff.to_dict(), pretty)))
    else:
        log_info_vv("check %s: (%s) %s:%s" % (changes_type, check.namespace, check.host, check.service))

    return changes_type


def apply_checks(all_checks, mark, token, dry_run=True, pretty=False):
    changes_cnt = {'not_changed': 0, 'updated': 0, 'inserted': 0, 'removed': 0}

    with JugglerApi('http://juggler-api.search.yandex.net', mark=mark, oauth_token=token, dry_run=dry_run) as api:
        for check in all_checks:
            log_info_vv('trying to upsert check: (%s) %s:%s' % (check.namespace, check.host, check.service))
            log_info_vvv('raw check: ' + jdumps(check.to_dict(), pretty))
            res = api.upsert_check(check)

            changes_type = process_upsert_result(res, check, pretty)
            changes_cnt[changes_type] += 1

        cres = api.cleanup()
        for host, service in cres.removed:
            changes_cnt['removed'] += 1
            log_info_v("check removed: %s:%s" % (host, service))

    total = len(all_checks) + changes_cnt['removed']
    changes_cnt['changed'] = total - changes_cnt['not_changed']
    changes_str = 'changed: %3d (%s)' % (
        changes_cnt['changed'],
        ', '.join(['%s: %3d' % (k, changes_cnt[k]) for k in ('inserted', 'updated', 'removed')])
    )
    log_info(', '.join(['dry_run: ' + str(dry_run), changes_str, 'total: ' + str(total)]))

    return changes_cnt


def log_info(msg, *args, **kwargs):
    logging.getLogger(__name__).log(logging.INFO - 0, msg, *args, **kwargs)
def log_info_v(msg, *args, **kwargs):
    logging.getLogger(__name__).log(logging.INFO - 1, msg, *args, **kwargs)
def log_info_vv(msg, *args, **kwargs):
    logging.getLogger(__name__).log(logging.INFO - 2, msg, *args, **kwargs)
def log_info_vvv(msg, *args, **kwargs):
    logging.getLogger(__name__).log(logging.INFO - 3, msg, *args, **kwargs)


def jdumps(jdict, pretty=False):
    def serialize(obj):
        try:
            s = obj.__dict__
        except:
            s = obj
        return s

    if pretty:
        return '\n' + json.dumps(jdict, sort_keys=True, ensure_ascii=False, indent=4, separators=(',', ': '), default=serialize)
    return json.dumps(jdict, sort_keys=True, ensure_ascii=False, default=serialize)
