#!/usr/bin/python
# -*- coding: utf8 -*-

import direct_dist_pkgdata
import argparse
import apt_pkg


description = """
Скрипт для отлавливания недостающих зависимостей пакетов среди бранчей заданного репозитория на dist

Например, если пакет pkg1 из бранча stable заданного репозитория имеет зависимость pkg2, отсутствующую в stable, но содержащуюся 
в некотором другом бранче (например, testing), то pkg2 будет фигурировать в итоговом отчете.

Если же, к примеру, pkg1 в testing, а зависимость pkg2 отсутствует в testing, но есть в stable (т.е. в более "старшем" бранче), 
то pkg2 в отчет НЕ попадет. Таким образом, зависимости, которых нет ровно в заданном бранче, но есть в каком-то более "старшем", 
засчитываются как "нормальные".

При указании --brief полученный отчет будет в кратком формате (удобном для передачи его на вход dmove: только сами найденные зависимости и без повторов). 
В предположении, что такой формат нужен в основном для dmove, также из отчета будут исключены зависимости, находящиеся только в prestable и отсутствующие 
в unstable и testing.

Примеры:

Найти недостающие зависимости пакетов в репозитории direct-common в бранче stable:
direct-dist-check-dependencies.py -r direct-common
"""

branches = ['unstable', 'testing', 'prestable', 'stable']


def parse_arguments():
    parser = argparse.ArgumentParser(description=description, formatter_class=argparse.RawDescriptionHelpFormatter)
    
    parser.add_argument('-r', '--repository', dest='repository', help='Проверяемый репозиторий dist-а (обязательный параметр)', type=str, required=True)
    parser.add_argument('-b', '--branch', dest='branch', help='Проверяемый бранч выбранного репозитория (по умолчанию stable)', type=str, choices=['stable', 'prestable', 'testing'], default='stable')
    parser.add_argument('-a', '--arch', dest='arch', help='Архитектура проверяемых пакетов; all или amd64 (по умолчанию all)', type=str, default='all')
    parser.add_argument('--brief', dest='brief', help='Сгенерировать краткий отчет (например, для передачи его на вход dmove)', action='store_true')

    args = parser.parse_args()

    return args


def check_dependencies(data, branch):
    report = []

    # sorting packages in order to skip older versions and process only the latest one
    data[branch].sort(cmp=lambda x,y: direct_dist_pkgdata.pkg_cmp((x[0],x[1]),(y[0],y[1])))
    data[branch].reverse()
    for i in range(len(data[branch])):
        # an older version of the same package is ignored because data is sorted
        if i > 0 and data[branch][i][0] == data[branch][i - 1][0]:
            continue

        for dep in data[branch][i][2]:
            pkg, ver = dep
            flag = 0
            for br in branches:
                if pkg in [x[0] for x in data[br]]:
                    flag = 1
                    break
            if flag == 0:
                # dep doesn't appear in current repo at all, so we just ignore it
                continue
            if ver == '':
                flag = 0
                for br in reversed(branches):
                    if pkg in [x[0] for x in data[br]]:
                        flag = 1
                        break
                    if br == branch:
                        break
                if flag == 0:
                    report.append((data[branch][i][0], dep))
            else:
                flag = 0
                for br in reversed(branches):
                    if (pkg, ver) in [(x[0], x[1]) for x in data[br]]:
                        flag = 1
                        break
                    if br == branch:
                        break
                if flag == 1:
                    pass
                else:
                    flag = 0
                    for br in branches:
                        if br == branch:
                            break
                        if (pkg, ver) in [(x[0], x[1]) for x in data[br]]:
                            flag = 1
                            break
                    if flag == 1:
                        report.append((data[branch][i][0], dep))

    return report


def find_max_version(pkg, data, branch):
    # max version of pkg in branches <= branch
    mx = ''
    for br in branches:
        if br == branch:
            break
        if br == 'prestable':
            # because it's mostly for dmove
            continue

        for x in data[br]:
            if x[0] == pkg and (mx == '' or apt_pkg.version_compare(mx, x[1]) < 0):
                mx = x[1]
    return mx


def print_report(report, data, branch, brief):
    if brief:
        # seems that this is for dmove
        # so, we'll skip prestable here (dmove: unstable -> testing -> stable)
        brf, brf1 = [], []
        for it in report:
            if it[1] not in brf:
                brf.append(it[1])
        for it in brf:
            pkg, ver = it
            if ver == '':
                # searching for the newest version of this package in given repo
                ver = find_max_version(pkg, data, branch)
                if ver == '':
                    # probably, this package appears ONLY in prestable => we don't dmove it
                    continue
            else:
                flag = 0
                for br in branches:
                    if br == branch:
                        break
                    if br == 'prestable':
                        continue
                    if it in [(x[0], x[1]) for x in data[br]]:
                        flag = 1
                        break
                if flag == 0:
                    # probably, this version of package appears ONLY in prestable => we don't dmove it
                    continue
            brf1.append((pkg, ver))
        brf = []
        for it in brf1:
            if it not in brf:
                brf.append(it)
        for it in brf:
            print it[0]+' '+it[1]
    else:
        for it in report:
            print it[0]+':  '+it[1][0]+' '+it[1][1]



def run():
    args = parse_arguments()

    data = {}
    for br in branches:
        path = direct_dist_pkgdata.get_path(args.repository, br, args.arch)
        data[br] = direct_dist_pkgdata.get_parsed_data(path)
    apt_pkg.init()

    report = check_dependencies(data, args.branch)
    # report's format: [ (package, (missed_dependency, version)) ]

    print_report(report, data, args.branch, args.brief)

    return report



if __name__ == '__main__':
    run();
