#!/usr/bin/python
# -*- coding: utf8 -*-

import direct_dist_pkgdata
import argparse
import apt_pkg
import sys


description = """
Скрипт для отлавливания внешних (т.е. находящихся в другом репозитории по отношению к данному) зависимостей пакетов в репозиториях dist

Например, если пакет pkg1 из репозитория repo1 имеет зависимость pkg2, отсутствующую в repo1, но содержащуюся в некотором другом repo2, 
то pkg2 будет фигурировать в итоговом отчете (при указании скрипту соответствующих параметров repo1, repo2).

При указании --brief полученный отчет будет в кратком формате (только сами найденные зависимости и без повторов).

Примеры:

Найти внешние зависимости пакетов из direct-trusty/stable, содержащиеся в yandex-trusty/stable:
direct-dist-check-ext-dependencies.py -r direct-trusty/stable -x yandex-trusty/stable
или (если branch не указан, он устанавливается как stable):
direct-dist-check-ext-dependencies.py -r direct-trusty -x yandex-trusty
"""

branches = ['unstable', 'testing', 'prestable', 'stable']


def parse_arguments():
    parser = argparse.ArgumentParser(description=description, formatter_class=argparse.RawDescriptionHelpFormatter)
    
    parser.add_argument('-r', '--repository', dest='repository', help='Проверяемый репозиторий и бранч в формате repo/branch, где branch=unstable|testing|prestable|stable (обязательный параметр)', type=str, required=True)
    parser.add_argument('-x', '--ext-repository', dest='ext_repository', help='Внешний репозиторий и бранч в том же формате (обязательный параметр)', type=str, required=True)
    parser.add_argument('-a', '--arch', dest='arch', help='Архитектура проверяемых пакетов; all или amd64 (по умолчанию all)', type=str, default='all')
    parser.add_argument('--brief', dest='brief', help='Сгенерировать краткий отчет (например, может быть удобно при bmove)', action='store_true')

    args = parser.parse_args()

    return args


def split_repo(s):
    # splitting string of format 'repo/branch' into (repo, branch)

    a = s.replace('/', ' ').split()
    if len(a) < 1 or len(a) > 2:
        sys.exit('Invalid format: '+s)
    if len(a) == 2 and a[1] not in branches:
        sys.exit('Invalid branch name: '+a[1])
    # if branch seems to be missed, let's set it to stable
    if len(a) < 2:
        a.append('stable')
    
    return (a[0], a[1])


def check_dependencies(data1, data2):
    report = []

    # sorting packages in order to skip older versions and process only the latest one
    data1.sort(cmp=lambda x,y: direct_dist_pkgdata.pkg_cmp((x[0],x[1]),(y[0],y[1])))
    data1.reverse()
    for i in range(len(data1)):
        # an older version of the same package is ignored (because data1 is sorted)
        if i > 0 and data1[i][0] == data1[i - 1][0]:
            continue

        for dep in data1[i][2]:
            pkg, ver = dep
            if ver == '':
                if (pkg in [x[0] for x in data2]) and (pkg not in [x[0] for x in data1]):
                    report.append((data1[i][0], dep))
            else:
                if (dep in [(x[0], x[1]) for x in data2]) and (dep not in [(x[0], x[1]) for x in data1]):
                    report.append((data1[i][0], dep))

    return report


def find_max_version(pkg, data):
    # max version of pkg in data
    mx = ''
    for x in data:
        if x[0] == pkg and (mx == '' or apt_pkg.version_compare(mx, x[1]) < 0):
            mx = x[1]
    return mx


def print_report(report, data2, brief):
    if brief:
        brf, brf1 = [], []
        for it in report:
            if it[1] not in brf:
                brf.append(it[1])
        for it in brf:
            pkg, ver = it
            if ver == '':
                # searching for the newest version of this package in given repo
                ver = find_max_version(pkg, data2)
            brf1.append((pkg, ver))
        brf = []
        for it in brf1:
            if it not in brf:
                brf.append(it)
        for it in brf:
            print it[0]+' '+it[1]
    else:
        for it in report:
            print it[0]+':  '+it[1][0]+' '+it[1][1]



def run():
    args = parse_arguments()
    repo, branch = split_repo(args.repository)
    ext_repo, ext_branch = split_repo(args.ext_repository)

    path1 = direct_dist_pkgdata.get_path(repo, branch, args.arch)
    data1 = direct_dist_pkgdata.get_parsed_data(path1)
    path2 = direct_dist_pkgdata.get_path(ext_repo, ext_branch, args.arch)
    data2 = direct_dist_pkgdata.get_parsed_data(path2)
    apt_pkg.init()

    report = check_dependencies(data1, data2)
    # report's format: [ (package, (ext_dependency, version)) ]

    print_report(report, data2, args.brief)

    return report



if __name__ == '__main__':
    run();
