#!/usr/bin/perl


=head1 DESCRIPTION

Скрипт для отбора и просмотра логов

результат поиска пишет в файл /tmp/<login>/<type>.json
или в другой указанный файл

Параметры:

0. Вспомогательные

  -h, --help 
    справка

  --list 
    список доступных логов

1. Обязательно должен быть указан диапазон дат

  --today
    только сегодня

  --week
    последние 7 дней

  --month
    последние 30 дней

  --df, --date-from, --from 'YYYY-MM-DD'
    дата, с которой (включительно) начинать поиск

  --dt, --date-to, --to 'YYYY-MM-DD'
    дата, по которую (включительно) производить поиск
    по умолчанию -- сегодня

  -d, --date 'YYYY-MM-DD'
    производить поиск за одну указанную дату

  --ld, --last-days N
    производить поиск за последние N дней

2. Можно указать, что ищем и в каком количестве

  --type <str>
    обязтельный параметр: тип логов.
    Список доступен по --list

  --stat
    не искать полные данные, посмотреть статистику по загрузке файлов;
    сочетается с указанием времени и условиями поиска

    Можно указывать поля, по которым делать группировку (по-умолчанию - log_date), можно указать несколько через запятую
    Сортировка производится по первому указанному полю
        --stat log_date
        --stat cmd,log_date

    Если указать дополнительные условия (--cid/--bid/--cmd/...) - можно получить статистику того, что происходило с интересующими объектами

  --cid NNN
  --bid NNN
  --pid NNN
  -- и другие лог-специфичные параметры, см. --help --type <type>
    отобрать записи по указанному cid/pid/bid 
    пока можно только по одному значению каждого вида

    Некоторые параметры поддерживают LIKE, а некоторые даже делают LIKE по-умолчанию:
    --type ppclog --param '"cid_to":123' # => param LIKE '%"cid_to":123%' - для param -- LIKE по-умолчанию
    --type ppclog --param-eq '"cid_to":123' # => param = '"cid_to":123'   - дописываем -eq к имени параметра, чтобы получить точное соответствие
    --type ppclog --cmd showCamp # => cmd = 'showCamp' - строгое совпадение по-умолчанию
    --type ppclog --cmd-like 'showCamp' # => cmd LIKE '%showCamp%' - дописываем -like чтобы искать нечетко


  --limit NN
    отобрать не более NN записей

  Если запросы падают по памяти (тяжелый запрос, например только с --param за большой интервал дат), можно
  использовать профиль КлХ для тяжелых запросов:
  --profile heavy
  (но лучше такие запросы по возможности избегать)


3. Как поступить с результатами поиска
 
  --file, -f <filename>
    записать найденное в указанный файл вместо умолчального

  --less, -L
    после сохранения в файл показать найденное less'ом 

=head1 EXAMPLES

  Какие файлы за сегодня загружены до какого времени:
  dscribe-grep.pl --type bsexport_data --today --stat
  dscribe-grep.pl --type bsexport_prices --today --stat

  Записи по определенному pid за неделю:
  dscribe-grep.pl --type bsexport_data --pid 279170777 --week

  Записи по определенному cid за неделю:
  dscribe-grep.pl --type=bsexport_prices --cid 7769528 --week

  Записи по определенному cid за сегодня, результат сразу показывать less'ом 
  dscribe-grep.pl --type bsexport_data --today --cid 2807418 --less

  Записи по определенному cid за сегодня, записать в указанный файл и показать less'ом
  dscribe-grep.pl --type bsexport_data --today --cid 2807418 -f cid_2807418 --less

  Записи по определенному cid по диапазону дат:
  dscribe-grep.pl --type bsexport_data --date-from '2014-02-01' --date-to '2014-02-10' --cid 8400251

  В какие дни за последнюю неделю появлялась в экспорте определенная кампания:
  dscribe-grep.pl --type bsexport_data --stat --week --cid 2807418

  --------------------

  Что происходило с кампанией 263 за сегодня:
  dscribe-grep.pl --type ppclog --cid 263 --today

  Когда последний раз редактировали параметры кампании?
  dscribe-grep.pl --type ppclog --cid 263 --cmd saveCamp --limit 1 --month

  Поиск по параметрам
  dscribe-grep.pl --type ppclog --param '"cid_to":"5012010","sum":"37","cid_from":"3218105"'

=head1 QUERIES

where:

40 секунд 
select log_time, metadata, data from bsexport_data_2 where log_date >= toDate('2014-02-06') and log_date <= toDate('2014-02-13') and cid in ( 7769528)

1 секунда
select log_time, metadata, data from bsexport_data_2 where log_date >= toDate('2014-02-06') and log_date <= toDate('2014-02-13') and cid = 7769528

5 секунд
select log_time, metadata from bsexport_data_2 where log_date >= toDate('2014-02-06') and log_date <= toDate('2014-02-13') and cid in ( 7769528)

0.2 секунды
select log_time, metadata from bsexport_data_2 where log_date >= toDate('2014-02-06') and log_date <= toDate('2014-02-13') and cid = 7769528


prewhere:

1.03
select log_time, metadata, data from bsexport_data_2 prewhere log_date >= toDate('2014-02-07') and log_date <= toDate('2014-02-14') and cid in (7769528) 

0.17
select log_time, metadata, data from bsexport_data_2 prewhere log_date >= toDate('2014-02-07') and log_date <= toDate('2014-02-14') and cid = 7769528 

0.96
select log_time, metadata from bsexport_data_2 prewhere log_date >= toDate('2014-02-07') and log_date <= toDate('2014-02-14') and cid in (7769528) 

0.12
select log_time, metadata from bsexport_data_2 prewhere log_date >= toDate('2014-02-07') and log_date <= toDate('2014-02-14') and cid  = 7769528 


=head1 TODO

  + Встроенное измерение времени
  - сортировать и уникализировать результат
  + показывать количество строк в результате
  - показывать статистику: 
    + за какие даты сколько записей есть
    - какие файлы загружены до какой позиции
  - показывать статистику и по условиям (cid/pid/bid) -- объединить оба запроса в один, group by добавлять, если есть --stat
  - для ppclog по-умолчанию json
  - добавить --<param>-like и force_like => 1


=cut

use strict;
use warnings;
use open ':std' => ':utf8';

use lib::abs '../lib';

use Carp;
use Getopt::Long qw/GetOptionsFromArray/;
use Path::Tiny;
use List::Util qw/first/;
use DScribe::Grep::Utils;

my %known_types = map { $_ => 1 } DScribe::Grep::Utils::list_known_tables();

$SIG{__WARN__} = \&Carp::cluck;
$SIG{__DIE__}  = \&Carp::confess;

run();

sub run
{
    my (%opt, %opt_short);
    Getopt::Long::Configure('no_ignore_case');
    Getopt::Long::Configure('pass_through');

    # сначала опции, общие для всех типов логов
    GetOptionsFromArray(\@ARGV, \%opt_short,
        "type|t=s",
        "grep_class|grep-class|G=s",
        "help|h",
        "list",
        "limit=i",
        "today",
        "week",
        "month",
        "where=s",
        "date_from|date-from|from|df=s",
        "date_to|date-to|to|dt=s",
        "date|d=s",
        "last_days|last-days|ld=i",
        "verbose|v",
        "dry_run|dry|n",
        "less|L",
        "file|f|O:s",
        "stat:s",
        "debug",
        "clickhouse_user:s",
    ) or die "can't parse options, stop\n";
    if ($opt_short{help}) {
        usage();
    }
    if ($opt_short{list}) {
        print "Available types:\n".(join "\n", map { "\t$_" } sort keys %known_types )."\n";
        exit 0;
    }
    Getopt::Long::Configure('no_pass_through');
    my $type = $opt_short{type};
    unless ($type) {
        exit 0 if $opt_short{help};
        die "--type required\n";
    }
    unless ($known_types{$type}) {
        exit 0 if $opt_short{help};
        die "invalid type\n";
    }
    if (($opt_short{file}//'') eq '') {
        my $login = [getpwuid($<)]->[0];
        my $dir = "/tmp/$login";
        mkdir $dir;
        die "can't create dir $dir" unless -d $dir;
        my $output_file = "$dir/$type.json";
        $opt_short{file} = $output_file;
    }
    my $grep_class = get_grep_class($type, %opt_short);
    my @grep_options = $grep_class->grep_options;
    Getopt::Long::Configure('pass_through');
    GetOptionsFromArray(\@ARGV, \%opt, @grep_options) or die "can't parse additional options for grep, stop\n";
    my $grep = $grep_class->new($type, %opt_short, %opt);
    my $type_options = $grep->options;
    GetOptionsFromArray(\@ARGV, \%opt, @$type_options) or die "can't parse additional options, stop\n";
    if ($opt_short{help}) {
        $grep->usage();
        exit 0;
    }

    if (defined $opt_short{stat}) {
        return $grep->logs_stat({%opt_short, %opt, skip_fields => \%opt_short});
    }

    return $grep->grep_logs({%opt_short, %opt, skip_fields => \%opt_short});
}

sub get_grep_class
{
    my ($type, %opt) = @_;
    my $class = $opt{grep_class} // 'DScribe::Grep::Clickhouse::Direct';
    eval "require $class";
    if ($@) {
        die $@;
    }
    return $class;
}

sub usage
{
    my %O = @_;
    system("podselect -section NAME -section DESCRIPTION -section OPTIONS -section EXAMPLES $0 | pod2text-utf8");
    exit 0 if $O{exit};
}
