#!/usr/bin/perl

=head1 TODO

    Параметры: 
      тип лога
      тип хранилища
      имя лока (по умолчанию -- имя лога)


    Настройки:
      где КлХ
      где база с позициями
      где лог


    получить имя файла
    прочитать позицию из локального sqlite
    сделать seek 
    цикл
        прочитать блок строк
        превратить массив строк в массив структур
        отдать массив структур в хранилище
        записать позицию, до которой обработан файл
    повторить
    


    + позиции в файлах в ClH
      + сохранение обработанности файлов в таблицу в ClH
      + чтение и учет: если файл уже обработан до конца -- пропускать
    + "нормализованные" имена файлов (включая имя хоста)
    - локальные позиции, чтобы полностью обработанные файлы узнавать без хождения в ClH
    - !!! систематическая работа с неразобранными строками (логгировать, запоминать, перезаписывать)

=cut

=head1 DESCRIPTION

 Параметры именованные: 
  --parser, -p <parser_name>
      Имя класса парсера (dbshards_ids)

  --writer, -w <writer_name>
      Имя класса записывателя (Clickhouse)

  --no-lock
      Не брать блокировку. Блокировка берется для каждого парсера (--parser) своя.

  --lock-name <name>
      Задать свое имя для блокировки, вместо умолчального 'dscribe_upload_'.$parser 

  --conf, -c <filename.yaml>
      Конфиг файл

  --dir <dir>
    Директория, где нужно искать логи. Альтернатива позиционным параметрам (вместе
    с --mask). Можно указывать несколько: --dir dir1 dir2

  --mask <mask>
    При поиске в директориях из dir выбирать файлы, попадающие под mask
    mask - регулярное выражение

  --exclude, -x <mask>
    Маска, по которой будут исключаться файлы, попавшие под --mask
    Можно указывать несколько: -x ppcdev  ppctest-ts
    регулярное выражение

  --reset-since yyyy-mm-dd
    Сбросить данные о позиции начиная с указанной даты (позиции за эту дату и последующие будут удалены)

  --pos-saver, -S <class>
    Имя класса для сохранения позиции (JSON|SQLite)

  --help, -h
      Показать справку и выйти

  --version-table, -V <suffix> 
    Указание суффикса для записи данных в таблицу. Например, значение "v2" 
    будет означать <table_name>_v2_mergetree.

  --unlimit-memory, -u
    Выставление ulimit для RLIMIT_VMEM в -1. Например, размер строки очень большое и скрипт падает по OOM.
    Опция нужная для ручного запуска, т.к. потенциально может занять много памяти.

 Параметры позиционные:
    лог-файлы для загрузки

=cut

use strict;
use warnings;

use feature qw/state/;

use DBI;
use Carp;
use Getopt::Long;
use LWP::UserAgent;
use JSON;
use YAML::XS;
use Data::Dumper;
use Data::Dump 'pp';
use List::Util qw/sum/;
use IO::Uncompress::Gunzip qw(gunzip $GunzipError); 
use Time::Piece;
use Pid::File::Flock;
use Path::Tiny;
use BSD::Resource;

use Log::Any::Adapter;
use Log::Any qw/$log/;
use Yandex::Log;
use Yandex::Clickhouse;
use Yandex::JugglerQueue;
use Yandex::Profile;
use Yandex::TimeCommon qw/today/;

use lib::abs '../lib';
use Sys::Hostname qw/hostname/;

use utf8;
use open ':std' => ':utf8';

my $hostname = hostname();

$Yandex::Log::LOG_ROOT = "/var/log/dscribe";

$SIG{__WARN__} = sub { return if (!defined $^S) || $^S; $log->error(@_); Carp::cluck(@_) };
$SIG{__DIE__}  = sub { return if (!defined $^S) || $^S; $log->error(@_); Carp::confess(@_) };

my %LOG_OPTIONS = (
    log_file_name => 'clickhouse_upload_log',
    auto_rotate => 1,
    date_suf => '%Y%m%d',
    msg_prefix => "[$$]",
    tee => $ENV{LOG_TEE},
);

Log::Any::Adapter->set('YandexLog', %LOG_OPTIONS);

# $Yandex::Clickhouse::QUERIES_LOG = 'clh.log';
my $INSERT_CHUNK_SIZE = undef;
my $INSERT_CHUNK_BYTES = undef;

run() unless caller();

sub run
{
    my $opt = parse_options();

    setrlimit(RLIMIT_VMEM, -1, -1) if $opt->{umlim_memory};
    
    my $conf = YAML::XS::LoadFile($opt->{conf_file});
    
    if (disabled_on_this_host($conf->{regex_disabled_hosts})) {
        return;
    }

    $log->info(pp $opt);

    my $stop_file_dir = $conf->{stop_file_dir} || '/var/run/dscribe';
    mkdir $stop_file_dir unless -d $stop_file_dir;  # директория в /var/run может исчезнуть после перезагрузки машины
    my $stop_file = path($stop_file_dir, $opt->{parser}.".stop");
    if ($stop_file->is_file) {
        write_log("stop file $stop_file found, exit");
        return;
    }

    get_lock(no_lock => $opt->{no_lock}, type => $opt->{parser}, lock_name => $opt->{lock_name});

    $conf->{upload}->{chunk_size} = $INSERT_CHUNK_SIZE if defined $INSERT_CHUNK_SIZE;

    if ($conf->{upload}->{log_dir}) {
        $Yandex::Log::LOG_ROOT = $conf->{upload}->{log_dir};
    }

    my $table_name = $opt->{table_name} ? $opt->{table_name} : $opt->{parser};
    my $writer = get_writer($opt->{writer}, $conf->{writer}, $table_name);
    my $parser = get_parser($opt->{parser}, $conf->{parser});
    
    my $pos_saver = get_pos_saver($opt->{pos_saver}, writer => $writer->name, parser => $parser->name, conf => $conf->{positions});

    my @files = @ARGV;
    unless (@files) {
        @files = find_files(
            dir => $opt->{dir},
            mask => $opt->{mask},
            exclude => $opt->{exclude},
            pos_saver => $pos_saver,
            reset_since => $opt->{reset_since},
            parser => $opt->{parser},
            version_table => $opt->{version_table},
        );
        write_log("$opt->{parser}: found ".(scalar @files)." logs");
        # write_log(\@files);
    }

    my $cnt = 0;
    for my $file (@files) {
        $cnt++;
        my $st = script_timer("upload:$file");
        write_log("file $cnt of ".(scalar @files));
        my $res = upload_file(
            file => $file,
            parser => $opt->{parser},
            writer => $opt->{writer},
            conf => $conf,
            writer => $writer,
            parser => $parser,
            pos_saver => $pos_saver,
            version_table => $opt->{version_table},
            service => $opt->{service},
        );
        $st = undef;
    }
    juggler_ok(service => "dscribe-upload-files.$opt->{parser}.working");
    write_log("finish");
}

=head1 METHODS

=head2

    берет лок с нужным именем
    Если не получилось -- умирает

    Параметры именованные
    no_lock
    lock_name

=cut
sub get_lock
{
    my %O = @_;
    my $name = $O{lock_name} || "dscribe_upload_".($O{type}||"undefined");
    eval {
        Pid::File::Flock->new(raise => 1, name => $name, quiet => 1) unless $O{no_lock};
    };
    if ($@) {
        exit 1;
    }
}


=head2 

    Разбирает параметры, переданные скрипту

=cut
sub parse_options
{
    my %O = (
        dir => [],
        exclude => [],
        writer => 'Clickhouse',
        conf_file => '/etc/dscribe/dscribe-upload-files.yaml',
    );
    Getopt::Long::Configure('no_ignore_case');
    GetOptions(
        "p|parser=s" => \$O{parser},
        "w|writer=s" => \$O{writer},
        "no-lock" => \$O{no_lock},
        "lock-name=s" => \$O{lock_name},
        "c|conf=s"  => \$O{conf_file},
        "reset-since=s" => \$O{reset_since},
        "dir=s{,}" => $O{dir},
        "exclude|x=s{,}" => $O{exclude},
        "mask=s" => \$O{mask},
        "pos-saver|S=s" => \$O{pos_saver},
        "chunk-size|C=i" => \$INSERT_CHUNK_SIZE,
        "chunk-bytes=i" => \$INSERT_CHUNK_BYTES,
        "help|h" => \&usage,
        "version-table|V=s"  => \$O{version_table},
        "table-name=s" => \$O{table_name},
	"unlimit-memory|u" => \$O{umlim_memory},
        "service=s" => \$O{service},
    ) or die "can't parse options, stop";

    die "bad writer '$O{writer}', stop" unless $O{writer} =~ m/^[a-z0-9_-]+$/i;
    die "unsupported writer '$O{writer}'" unless $O{writer} =~ /^(Clickhouse|Stderr)$/;

    die "bad parser '$O{parser}', stop" unless $O{parser} =~ m/^[a-z0-9_-]+$/i;

    die "bad pos saver" unless $O{pos_saver} =~ /^[a-z0-9_-]+$/i;

    if ($O{reset_since}) {
        $O{reset_since} =~ s!\D!!g;
        die "invalid date for reset_since '$O{reset_since}'" unless $O{reset_since} =~ /^\d{8}$/;
    }

    return \%O;
}


=head2

    Обработать файл

=cut
sub upload_file
{
    my %O = @_;
    my $file = $O{file};

    my $t = timer();

    my $writer = $O{writer};
    my $parser = $O{parser};
    my $position_saver = $O{pos_saver};

    my $version_table = $O{version_table};

    # нормализуем имя файла (для записи позиции и логгирования)
    my $norm_filename = $position_saver->get_norm_filename($file);
    my $norm_source = "file:$norm_filename";
    write_log({file => $file, norm_filename => $norm_filename, action => "start"});

    # читаем позицию, на которой остановились 
    my $pos = $position_saver->read_position($norm_filename);
    write_log({file => $file, writer => $writer->name, position => $pos, action => "read position"});

    return if $position_saver->is_complete($file);

    # открываем файл, прыгаем на нужную позицию
    write_log("$file: position=".($pos->{position}//0)."");
    my $fh = open_logfile($file, $pos->{position} || 0);
    return unless $fh;

    # читаем строки...
    my $lines_count = 0;
    my $err_count = 0;
    write_log({ file => $file, action => "reading file" });
    my @lines;
    my $chunk_bytes = 0;
    my $today_ymd = localtime->ymd;
    $today_ymd =~ s!-!!g;
    my $max_chunk_bytes = $INSERT_CHUNK_BYTES // $O{conf}->{upload}->{chunk_size} * 4_000;
    my $service = $O{service};
    while (my $line = <$fh>) {
        if (substr($line,-1,1) ne "\n") {
            # если строка недописанная (без перевода строки в конце) -- падаем,
            # но только в том случае, если мы обрабатываем файл за сегодня. файлы за прошлые
            # даты уже не изменятся, их молча пропускаем
            my ($fdate) = ($file =~ /(\d{8})/);
            last if $today_ymd > $fdate;
            juggler_warn(service => "dscribe-upload-files.".$parser->name.".working", description => "incomplete line: $file");
            log_die("$file: incomplete line");
        }
        
        $lines_count++;
        $chunk_bytes += length($line);
        # собираем блок строк для обработки
        chomp $line;
        push @lines, $line;

        if (@lines >= $O{conf}->{upload}->{chunk_size} || $chunk_bytes > $max_chunk_bytes ) {
            # printf STDERR "inserting %d bytes, %d/%d lines\n", $chunk_bytes, scalar(@lines), $O{conf}->{upload}->{chunk_size};
            # когда накопилось достаточно много строк -- парсим и записываем
            $err_count += process_lines(\@lines, source => $norm_source, writer => $writer, parser => $parser, version_table => $version_table, service => $service);
            # и записываем позицию, до которой дочитали файл
            $position_saver->write_position($file, tell($fh), complete => 0);
            @lines = ();
            $chunk_bytes = 0;
        }
    }
    # файл кончился, дописываем накопившиеся строки
    if (@lines > 0) {
        $err_count += process_lines(\@lines, source => $norm_source, writer => $writer, parser => $parser, version_table => $version_table, service => $service);
    }
    # пишем позицию, указываем, что файл дочитали до конца
    $position_saver->write_position($file, tell($fh), complete => 1);

    write_log({file => $file, writer => $writer->name, action => "finish", lines => $lines_count, errors => $err_count});

    return {};
}


=head2 

    создает объект для записи в Хранилище

=cut
sub get_writer
{
    my ($writer_name, $conf, $table) = @_;
    my $t = timer();
    my $writer_package = "DScribe::Writer::$writer_name";
    my $writer_pm = ($writer_package =~ s!::!/!gr) =~ s/$/.pm/r;
    require $writer_pm;
    state $writer = "$writer_package"->new({conf => $conf->{$writer_name}, table => $table});
    return $writer;
}

=head2 

    создает объект для парсинга логов в Формате

=cut
sub get_parser
{
    my ($parser_name, $conf) = @_;
    my $t = timer();
    my $parser_package = "DScribe::Parser::$parser_name";
    my $parser_pm = ($parser_package =~ s!::!/!gr) =~ s/$/.pm/r;
    eval {
        require $parser_pm;
    };
    if ($@) {
        die "failed to load parser $parser_name:\n".$@;
    }
    state $parser = "$parser_package"->new({name => $parser_name, group_by => "month", %$conf});
    return $parser;
}

sub get_pos_saver
{
    my ($saver_name, %opt) = @_;
    my $package = "DScribe::FilePositions::$saver_name";
    my $package_pm = ($package =~ s!::!/!gr) =~ s/$/.pm/r;
    eval {
        require "$package_pm";
    };
    if ($@) {
        die "failed to load saver $package\n".$@;
    }
    my $saver = "$package"->new(writer => $opt{writer}, parser => $opt{parser}, %{$opt{conf}});
    return $saver;
}

=head2 

    Получает массив текстовых строк, вызывает для них парсер
    и отдает то, что получилось, writer'у

    Возвращает количество ошибок

=cut
sub process_lines 
{
    my ($lines, %O) = @_;
    my ($parser, $writer, $version_table) = @O{qw/parser writer version_table/};

    # write_log({ action => 'process_lines', begin => 'parse_lines', batch_size => scalar @$lines });
    my $t = timer(op => 'parse');
    my $parsed = $parser->parse_lines($lines, source => $O{source}, service => $O{service});

    my $err_cnt = 0;
    for (@{$parsed->{error}||[]}) {
        # write_parse_errors_log -- куда должен писать? Файлы или хранилище?
        write_parse_errors_log({error => $_, source => $O{source}});
        $err_cnt++;
    }

    undef $t;
    $t = timer(op => 'write');
    # write_log({ action => 'process_lines', begin => 'write_data' });
    $writer->write_grouped_data($parsed->{grouped_data}, $version_table);
    # write_log({ action => 'process_lines', END => 1 });

    return $err_cnt;
}


=head2 

    Получает полное имя файла, возвращает файловый хендлер, промотанный до указанной позиции

=cut
sub open_logfile
{
    my ($filename, $position) = @_;
    if ($filename =~ m/\.gz$/){
        open(my $fh, "-|", "zcat $filename") or log_die("can't open zcat $filename ($!), stop");
        my $todo = $position;
        while($todo > 0) {
            my $cnt = read($fh, my $buf, $todo > 1e6 ? 1e6 : $todo)
                // log_die("can't read gzipped file: $!");
            # дочитали до конца, типа ok
            last if !$cnt;
            $todo -= $cnt;
        }
        return $fh;
    } else {
        open(my $fh, "<", $filename) or log_die("can't open file $filename ($!), stop");
        seek($fh, $position, 0) if $position;
        return $fh;
    }
}

=head2

    Запись собственного лога о прогрессе работы

=cut
sub write_log
{
    state $log;
    unless ($log) {
        $log ||= Yandex::Log->new(
            %LOG_OPTIONS,
        );
    }
    $log->msg_prefix("[$$]");
    $log->out(@_);
}

sub log_die
{
    write_log(@_);
    die $_[0];
}


=head2

    Запись собственного лога об ошибках парсинга

=cut
sub write_parse_errors_log
{
    state $errorlog;
    unless ($errorlog) {
        $errorlog ||= Yandex::Log->new(
            log_file_name => 'clickhouse_upload_parse_errors_log', 
            date_suf => '%Y%m%d',
            msg_prefix => "[$$]",
            tee => 0,
        );
    }
    $errorlog->msg_prefix("[$$]");
    $errorlog->out(@_);
}

sub usage
{
    system("podselect -section NAME -section DESCRIPTION -section OPTIONS -section EXAMPLES $0 | pod2text-utf8");
    exit 0;
}

sub find_files
{
    my (%O) = @_;
    my $t = timer(op => 'find');
    my @files;
    return @files unless $O{mask} && $O{dir};
    my $x_re = $O{exclude} ? join '|', @{$O{exclude}} : '';
    my $pos_saver = $O{pos_saver};

    for my $dir (@{$O{dir}}) {
        next if $x_re && $dir =~ $x_re;
        write_log(''.$dir);
        my $iter = path($dir)->iterator({ recurse => 1, follow_symlinks => 1 });
        while (my $f = $iter->()) {
            push @files, ''.$f->absolute();
        }
        @files = grep { /[0-9]{8}/ } @files;
    }
    @files = grep { /$O{mask}/ } grep { !$x_re || !/$x_re/ } @files;

    undef $t;
    $t = timer(op => 'is_complete');
    if ($O{reset_since}) {
        for my $f (@files) {
            if (get_filename_date($f) >= $O{reset_since}) {
                $log->out("reset position for $f");
                $pos_saver->reset_position($f);
            }
        }
        $log->out('EXIT');
        exit 0;
    }
    
    my $is_complete = $pos_saver->mass_is_complete(\@files);

    return grep { !$is_complete->{$_} } sort by_filename_date @files;
}

sub disabled_on_this_host
{
    my $regexes = shift;
    for my $re (@$regexes) {
        if ($hostname =~ /$re/) {
            return 1;
        }
    }
    return 0;
}

=head2 get_filename_date

=cut

sub get_filename_date
{
    $_[0] =~ /(\d{8})/;
    return $1;
}

sub by_filename_date
{
    my ($a_date) = ($a =~ /(\d{8})/);
    my ($b_date) = ($b =~ /(\d{8})/);
    $a_date cmp $b_date;
}

