#!/usr/bin/perl

use strict;
use warnings;
use utf8;
no warnings 'uninitialized';
use feature 'state';
$|++;

use Getopt::Long;
use Data::Leaf::Walker;
use JSON;
use Data::Dump 'pp';
use List::MoreUtils qw/uniq/;

=head1 DESCRIPTION

    Скрипт для написания однострочников для обработки логов api/cmd

    Читает файлы из командной строки или stdin
    Для каждой входной строки выполняет перловый код, указанный через -e

    Без дополнительных опций доступны переменные и функции:
        $json - JSON с параметрами
        $r - распаршенный $json
            В $r лежат $r->{cmd}, $r->{uid}, $r->{param} и все остальное
        pp() - Data::Dump::pp
        isw($r) - возвращает 1 если в $r была модифицирующая команда
            Пока что только для API

    С опцией -u:
        $r->{login}  - раскрывает $r->{uid}
        $r->{clogin} - раскрывает [первый] $r->{cluid}
        Подгружает Yandex::DBTools и Settings, нужно правильно установить PERL5LIB, см. прмер ниже

    С опцией -w
        $w - объект Data::Leaf::Walker($r)
        w($name) - возвращает значение первого листа по имени $name:
            $r = { param => { BannerID => [1,2,3] } }
            w('BannerID') => 1
        p(qw/names/) - печатает все значения по указанным именам
            p(qw/logtime cmd CampaignID/)


    С опцией -m
        Работать только с модифицирующими командами. Пропускает записи, если isw($r) возвращает 0. 

    Примеры:

    1) "Почему баннеры ушли на модерацию? Кто и когда менял кампанию?"
    export PERL5LIB=/var/www/beta.icenine.8500/protected
    clickhouse_grep_logs.pl --cid 10200093 --date 20140829 --type ppclog_api --less | logskim.pl -m -u -w -e 'p(qw/logtime cmd login/)'

    2) "Какие баннеры редактировались в указанный день?"
    clickhouse_grep_logs.pl --cid 10200093 --date 20140829 --type ppclog_api --less | logskim.pl -m -u -w -e 'w("cmd") eq "CreateOrUpdateBanners" && p(qw/logtime login BannerID/)'

=cut

GetOptions(\my %opt,
    "eval|e=s",
    "walker|w",
    "uid|user|login|u",
    "modifiers|m",
    "help|h",
) or usage();

if ($opt{help}) {
    usage();
}

if ($opt{uid}) {
    # eval string instead of {} to make debosh happy
    eval qq#
        require Yandex::DBTools;
        import  Yandex::DBTools;
        require Settings;
        import  Settings;
    #;
    if ($@) {
        die "can not import DBTools & Settings\n$@\n";
    }
}

sub isw
{
    my $r = shift;
    my %read = map { $_ => 1 } qw/
        GetBalance     GetBannerPhrases     GetBannerPhrasesFilter  
        GetBannersStat     GetCampaignParams     GetCampaignsList    
        GetCampaignsListFilter     GetChanges     GetClientInfo     GetClientsUnits    
        GetForecast     GetForecastList     GetReportList     GetStatGoals    
        GetSummaryStat     GetWordstatReportList     PingAPI
        GetBanners GetCampaignsParams

        adminPage     adminPageMan     advancedForecast     ajaxCheckUrl    
        ajaxDataForBudgetForecast     ajaxGetBannersCount     ajaxGetSuggestion    
        ajaxGetUrlPhrases     ajaxSearchMetro     ajaxSpellCheck    
        ajaxValidatePhrasesForForecast     ajaxValidateStreet     authForm    
        calcForecast     checkAntispam     checkMirrors     clientWallet    
        currencyConvertInProgress     editCamp     ForecastByWords     getAdGroup    
        getBanners     getImportedCampXls     getSuggestPhrases     listPdfReport    
        listStatMPReport     listWarnCtr     listWarnPlace     loadPdfReport    
        manageVCards     monitorTargetsList     normWords     profileStatsByEla    
        reportUncheckedContext     search     searchBanners     showAgReps    
        showBalanceLogs     showCamp     showCampMultiEdit     showCamps    
        showCampSettings     showCampStat     showCampStatOfflineMessage    
        showCaptcha     showClients     showCompetitors     showContactInfo    
        showDiag     showDiagDetails     showExportedXlsList     showgeo     showLogs  
        showMailLogs     showManagerLogs     showManagerMyClients     showMediaplan  
        showPlotPage     showQuestion     showRbac     showRegionalTraffic    
        showRetargetingCond     showSearchPage     showSmsLogs     showStaff    
        showSubClientCamps     tmplProc     userSettings     wordstat    
        yaShowUserCounters
    /;

    return !$read{$r->{cmd}};
}

sub get_login
{
    my $uid = shift;
    if (ref $uid eq 'ARRAY') {
        $uid = $uid->[0];
    }
    state $cache = {};
    return $cache->{$uid} //= get_one_field_sql(PPC(uid => $uid), "select login from users where uid = ?", $uid);
}

my $pre = q#
    my @r = split /\t/, $_;
    my $json = $r[-1];
    return unless $json =~ /^[\[\{]/;
    my $r = from_json($json);
    if ($opt{uid}) {
        $r->{login}  = get_login($r->{uid})   // $r->{uid};
        $r->{clogin} = get_login($r->{cluid}) // $r->{cluid};
    }
    if ($opt{modifiers}) {
        return unless isw($r);
    }
#;
if ($opt{walker}) {
    $pre .= q!
    my $w = Data::Leaf::Walker->new($r);
    local *w = sub {
        my $name = shift;
        $w->reset();
        while (my ($k, $v) = $w->each()) {
            my $i = $#$k;
            while ($i >= 0 && $k->[$i] =~ /^\d+$/) {
                $i--;
            }
            if ($k->[$i] eq $name) {
                return $v;
            }
        }
        return undef;
    };
    local *p = sub {
        my %line = ();
        for my $name (@_) {
            $w->reset();
            while (my ($k, $v) = $w->each()) {
                my $i = $#$k;
                while ($i >= 0 && $k->[$i] =~ /^\d+$/) {
                    $i--;
                }
                if ($k->[$i] eq $name) {
                    push @{$line{$name}}, $v;
                }
            }
        }
        if (%line) {
            print ''.(join "\t", map {
                uniq @{$line{$_}}
            } @_)."\n";
        }
    };
!;
}

my $code = $pre.$opt{eval};

while (<>) {
    eval $code;
    if ($@) {
        warn $@;
    }
}

sub usage
{
    system("podselect -section NAME -section SYNOPSIS -section DESCRIPTION $0 | pod2text-utf8");
    exit 0;
}
