use strict;
use warnings;

package DScribe::ClickhouseTable;

=head1 


=cut

use lib::abs;


use Mouse;
use YAML::XS;
use Cwd;
use Carp;
use DScribe::Grep::Utils;

has 'base_table_name' => (is => 'ro', isa => 'Str', required => 1);
has 'table_fields' => (is => 'ro', isa => 'ArrayRef[HashRef]', required => 1);
has 'table_engine' => (is => 'ro', isa => 'Str', required => 1);
has 'description_file' => (is => 'ro', isa => 'Str', required => 1);
has 'fields_hash' => (is => 'ro', isa => 'HashRef', required => 1);

sub BUILDARGS
{
    my ($self, $table) = @_;
    
    my $data = {
        base_table_name => $table,
    };

    my $tables_dir = DScribe::Grep::Utils::descriptions_dir();

    $data->{description_file} = "$tables_dir/$data->{base_table_name}.yaml";
    -f $data->{description_file} or die "can't find table description '$data->{description_file}', stop";

    my $table_description = YAML::XS::LoadFile($data->{description_file});
    $table_description->{table_name} eq $data->{base_table_name} or die "unexpected table name '$table_description->{table_name}' in file '$data->{description_file}', stop";

    $data->{$_} = $table_description->{$_} for qw/table_fields table_engine/;
    $data->{fields_hash} = { map { $_->{name} => $_ } @{$data->{table_fields}} };

    return $data;
}


sub BUILD {
    my $self = shift;

    return;
}


=head2 

    возвращает массив строк -- имена полей в табилце

=cut
sub db_fields
{
    my $self = shift;
    return [ map { $_->{name} } grep { !$_->{hide} } @{$self->table_fields} ];
}

sub field_descr
{
    my $self = shift;
    my $name = shift;
    return (grep { $_->{name} eq $name } @{$self->{table_fields}})[0];
}

=head2 field_get_prop

$table->field_get_prop(params => 'json');

Получить значение указанного свойства для указанного поля

=cut

sub field_get_prop
{
    my ($self, $field, $prop) = @_;
    croak "table does not have field $field" unless exists $self->fields_hash->{$field}; 

    return $self->fields_hash->{$field}->{$prop};
}

=head2 

    Параметры именованные
    suffix -- группировочный суффикс для таблицы

=cut 
sub table_name 
{
    my $self = shift;
    my %O = @_;

    my $table_name = $O{suffix} ? $self->base_table_name."_$O{suffix} " : $self->base_table_name; 

    die unless $table_name;
    
    return $table_name;
}

=head2 

    Возвращает sql для создания таблицы

    Параметры именованные
    suffix -- группировочный суффикс для таблицы
    engine -- table engine (по-умолчанию - из описания таблицы)

=cut 
sub schema_sql
{
    my $self = shift;
    my %O = @_;

    my $table_name = $O{suffix} ? $self->base_table_name."_$O{suffix}" : $self->base_table_name; 
    my $engine = $O{engine} // $self->table_engine;

    my $schema = sprintf 
qq{CREATE TABLE %s 
( 
%s
) ENGINE = %s
}, 
$table_name, 
join(",\n", map { "$_->{name} $_->{type}" } grep { !$_->{hide} } @{$self->table_fields}), 
$engine;

    return $schema;
}

=head2 merge_table_schema_sql

Возвращает sql для merge-таблицы

Options:
    db_name - имя базы данных, по-умолчанию 'default'
    merge_re - регулярка по которой будут объединяться таблицы
        по-умолчанию: ^имя_таблицы_\\d+$'
        обратные слеши нужно экранировать

=cut

sub merge_table_schema_sql
{
    my $self = shift;
    my (%O) = @_;
    my $db_name = $O{db_name} // 'default';
    my $re = $O{merge_re} // '^'.quotemeta($self->base_table_name).'_\\\\d+$';
    my $engine = "Merge($db_name, '$re')";

    return $self->schema_sql(engine => $engine);
}

=head2 distr_table_schema_sql

sql для создания Distributed таблицы

=cut

sub distr_table_schema_sql
{
    my $self = shift;
    my (%opt) = @_;
    my $engine = "Distributed(ppclog, default, ".$self->base_table_name."_mergetree)";
    return $self->schema_sql(engine => $engine);
}


1;

