package DScribe::FilePositions::JSON;

use DBI;
use Data::Dumper;
use POSIX qw/strftime/;
use JSON;
use Path::Tiny;

=head1 

пишет и читает позиции, до которых обработаны файлы логов

хранит данные в json-файлах вида
$dir/ppclog_cmd/Clickhouse/ci01f_ppclog_cmd_20140101.log.json

=cut


use Mouse;

extends 'DScribe::FilePositions';

has json => (is => 'ro', isa => 'Object', required => 1);
has dir => (is => 'ro', isa => 'Str', required => 1);

sub BUILDARGS
{
    my ($self, %O) = @_;
    
    my $data = {
        writer => $O{writer},
        parser => $O{parser},
        json => JSON->new->utf8(1),
        dir => $O{dir},
    };

    return $data;
}


sub BUILD {
    my $self = shift;
    my $dir = path($self->dir, $self->parser, $self->writer);
    # print STDERR "mkdir $dir\n";
    $dir->mkpath();
}

sub prepare_filename
{
    my ($self, $fn) = @_;
    $fn =~ s!/!_!g;
    return "$fn.json";
}

sub get_state_file
{
    my ($self, $fn) = @_;
    my ($yyyy,$mm,$dd) = ($fn =~ /(\d{4})(\d{2})(\d{2})/);
    unless ($yyyy && $mm && $dd) {
        die "can not find date in filename '$fn'";
    }
    my $dir = path($self->dir, $self->parser, $self->writer, $yyyy, $mm, $dd);
    unless ($dir->exists) {
        $dir->mkpath();
    }
    return path($dir, $self->prepare_filename($fn));
}

sub read_position
{
    my $self = shift;
    my ($filename) = @_;

    my $state_file = $self->get_state_file($filename);
    return {} unless $state_file->is_file;
    return $self->json->decode($state_file->slurp_utf8);
}

=head2 mass_read_position

    $files - ссылка на массив с нормализованными именами файлов
    возвращает хеш
        имя_файла => {
            position => ...,
            log_time => ...,
            filesize => ...,
            complete => 1|0,
            filename => ...,
        }

=cut

sub mass_read_position
{
    my ($self, $files) = @_;
    return { map { $_ => $self->read_position($_) } @$files };
}

=head2 

    Параметры позиционные
    $filename -- имя файла
    $pos -- позиция (результат tell)

    Далее -- параметры именованные:
    complete

    Возвращает ссылку на хеш (если позиций не нашлось -- ссылку на пустой хеш)

=cut

sub write_position
{
    my $self = shift;
    my ($filename, $position, %O) = @_;
    my $norm_filename = $self->get_norm_filename($filename);
    my $filesize = $O{filesize} // (stat($filename))[7] // 0;

    my $now = strftime("%Y-%m-%d %H:%M:%S", localtime);
    $self->get_state_file($norm_filename)->spew_utf8($self->json->encode({
        position => $position,
        log_time => $now, 
        filename => $norm_filename,
        filesize => $filesize,
        complete => $O{complete}//0,
    }));

    return;
}

=head2 reset_position

Сбросить позицию для файла
    Параметры:
    $filename - имя файла

=cut

sub reset_position
{
    my ($self, $fn) = @_;
    $self->get_state_file($self->get_norm_filename($fn))->remove;
}

1;

