package DScribe::FilePositions::SQLite;

use DBI;
use Data::Dumper;
use POSIX qw/strftime/;

=head1 

пишет и читает позиции, до которых обработаны файлы логов

хранит данные в sqlite, в файле $db_file

=cut


use Mouse;

extends 'DScribe::FilePositions';

has 'db_file' => (is => 'ro', isa => 'Str', required => 1);
has 'dbh' => (is => 'ro', isa => 'Object', required => 1);

sub BUILDARGS
{
    my ($self, %O) = @_;
    
    my $data = {
        writer => $O{writer},
        parser => $O{parser},
        db_file => $O{db_file},
        dbh => DBI->connect("dbi:SQLite:dbname=".$O{db_file},"",""),
    };

    return $data;
}


sub BUILD {
    my $self = shift;

    my $positions_schema = "
CREATE TABLE positions(
filename text, 
parser text, 
writer text, 
log_time text, 
position int, 
filesize int, 
complete int, 
PRIMARY KEY(filename, writer)
);
";

    my ($name, $schema) = $self->dbh->selectrow_array(
        "select name, sql
        from sqlite_master 
        where type='table' and name='positions'
        ",
        {},
    );
    unless ($name) {
        $self->dbh->do($positions_schema) or die $self->dbh->errstr;
    }

    return;
}

=head2 mass_read_position

    $files - ссылка на массив с нормализованными именами файлов
    возвращает хеш
        имя_файла => {
            position => ...,
            log_time => ...,
            filesize => ...,
            complete => 1|0,
            filename => ...,
        }

=cut

sub mass_read_position
{
    my ($self, $files) = @_;

    my $filenames = join ",", map { qq/'$_'/ } @$files;
    my $res = $self->dbh->selectall_arrayref(qq{ select log_time, position, filesize, complete, filename
        from positions
        where parser = ? and writer = ? and filename in ($filenames)}, { Slice => {} },
        $self->parser, $self->writer
    ) or die $self->dbh->errstr;

    return { map { $_->{filename} => $_ } @$res };
}

=head2 

    Параметры позиционные
    $filename -- имя файла
    $pos -- позиция (результат tell)

    Далее -- параметры именованные:
    filesize
    complete

    Возвращает ссылку на хеш (если позиций не нашлось -- ссылку на пустой хеш)

=cut

sub write_position
{
    my $self = shift;
    my ($filename, $pos, %O) = @_;
    my $norm_filename = $self->get_norm_filename($filename);
    my $filesize = $O{filesize} // (stat($filename))[7];

    my $now = strftime("%Y-%m-%d %H:%M:%S", localtime);
    $self->dbh->do("insert or replace 
        into positions 
        (parser, writer, log_time, filename, position, filesize, complete)
        values 
        (?, ?, ?, ?, ?, ?, ?)
        ", {}, 
        $self->parser, $self->writer, $now, $norm_filename, $pos, $filesize, $O{complete} || 0
    ) or die $self->dbh->errstr;

    return;
}

=head2 reset_position

Сбросить позицию для файла

=cut

sub reset_position
{
    my $self = shift;
    my ($filename) = @_;
    $filename = $self->get_norm_filename($filename);
    
    $self->dbh->do("delete from positions
        where parser = ? and writer = ? and filename = ?
        ", {}, 
        $self->parser, $self->writer, $filename
    ) or die $self->dbh->errstr;
}

sub iter
{
    my ($self) = @_;

    my $all = $self->dbh->selectall_arrayref(qq{
        select * from positions
        where writer = ? and parser = ?
        order by filename
    }, {Slice=>{}}, $self->writer, $self->parser);

    my $i = 0;
    return sub {
        return $all->[$i++];
    };
}


1;
