package DScribe::Grep::Clickhouse::Direct;

use strict;
use warnings;
use utf8;
use Mouse;
use Yandex::ListUtils qw/as_array/;
use Yandex::HashUtils qw/hash_grep hash_merge/;
use List::MoreUtils qw/uniq/;
use Yandex::DBTools;
use Yandex::DBShards;
use JSON;

use lib '/var/www/ppc.yandex.ru/protected';
use lib '/var/www/ppc.yandex.ru/perl/rbac-elementary';
use lib '/var/www/ppc.yandex.ru/perl/settings';
use SettingsALL;
use Settings;
use Primitives;

extends 'DScribe::Grep::Clickhouse';

has uid2login => (
    is => 'rw',
    isa => 'HashRef',
    default => sub { +{} },
);

has logins_fetched => (
    is => 'rw',
    isa => 'Bool',
    default => sub { 0 },
);

sub BUILDARGS
{
    my ($self, $type, %opt) = @_;
    my $args = $self->SUPER::BUILDARGS($type, %opt);
    return $args;
}

sub options
{
    my ($self) = @_;
    my $opt = $self->SUPER::options();
    push @$opt, 'login=s';
    return $opt;
}

sub postprocess
{
    my ($self) = @_;
    $self->SUPER::postprocess();
    if (!$self->logins_fetched) {
        $self->add_logins();
    }
}

sub postprocess_parse_line
{
    my ($self, $line) = @_;
    if ($self->logins_fetched) {
        # второй проход: строчка в json, с временем вначале
        $line =~ s!^\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2}\t!!;
        return from_json($line);
    }
    else {
        return $self->SUPER::postprocess_parse_line($line);
    }
}

sub postprocess_line
{
    my ($self, $row) = @_;
    $self->SUPER::postprocess_line($row);
    if ($self->logins_fetched) {
        if ($row->{uid}) {
            $row->{login} = $self->uid2login->{$row->{uid}};
        }
        if ($row->{cluid}) {
            $row->{cl_login} = [ map { $self->uid2login->{$_} } @{as_array($row->{cluid})} ];
        }
    }
    else {
        for my $uid (grep { defined } @{as_array($row->{uid})}, @{as_array($row->{cluid})}) {
            next unless $uid;
            $self->uid2login->{$uid} = undef;
        }
    }
    return $row;
}

sub add_logins
{
    my ($self) = @_;
    return if $self->logins_fetched;
    my @uids = keys %{$self->uid2login};
    if ($self->verbose) {
        printf STDERR "fetching logins for %d uids\n", scalar @uids;
    }
    my $uid2login = get_hash_sql(PRODUCTION_PPC_HEAVY(uid => \@uids), [
        "select uid, login from users",
        where => {
            uid => SHARD_IDS,
        }
    ]);
    my @unknown = grep { !$uid2login->{$_} } @uids;
    if (@unknown) {
        if ($self->verbose) {
            printf STDERR "fetching %d logins from passport\n", scalar @unknown;
        }
        hash_merge $uid2login, { map { $_ => Primitives::get_login_by_uid_passport($_) } @unknown };
    }
    $self->uid2login($uid2login);
    $self->logins_fetched(1);
    $self->postprocess();
}

sub preprocess
{
    my ($self, $opt) = @_;
    return $opt unless $opt->{login};

    my @logins = split /\s*,\s*/, delete $opt->{login};
    
    my $uids = get_hash_sql(PRODUCTION_PPC_HEAVY(login => \@logins), "select login, uid from users where login = ?", SHARD_IDS);
    
    # если не нашли uid-ы в базе - ищем в паспорте
    hash_merge $uids, { map { $_ => Primitives::get_uid_by_login($_) } grep { !$uids->{$_} } @logins };
    
    if (scalar(uniq(values %$uids)) != scalar @logins) {
        # TODO some uids were not resolved, warn to cli and web
        return $opt;
    }

    my @sql;
    for my $uid_field (qw/uid cluid/) {
        my $field_desc = $self->table->fields_hash->{$uid_field};
        next unless $field_desc;
        my $is_array = $field_desc->{type} =~ /Array/;
        if ($is_array) {
            my ($type) = ($field_desc->{type} =~ /Array\((.+?)\)/);
            push @sql, '('.(join " OR ", map { "has($uid_field, to$type('$_'))" } values %$uids).')';
        }
        else {
            my $type = $field_desc->{type};
            push @sql, '('.(join " OR ", map { "$uid_field = to$type('$_')" } values %$uids).')';
        }
    }
    if (@sql) {
        $self->add_sql_condition('('.(join " OR ", @sql).')');
    }
    return $opt;
}


1;
