package DScribe::Parser::base_direct_log;

=head1 

    Данных не хранит.

    Умеет получить текстовые строки и распарсить их в хеши соответственно с форматом и смыслом лога.

    Группировка делается для удобства/оптимизации хранения данных. В данный момент группировка
    происходит помесячно, название группы получается в виде YYYYmm

=cut

use JSON;
use Time::Local;
use Yandex::HashUtils qw/hash_merge/;
use List::MoreUtils qw/all/;
use Log::Any qw/$log/;

use Mouse;

has 'name' => (is => 'ro', isa => 'Str', required => 1);
has 'group_by' => (is => 'ro', isa => 'Str', default => 'month', trigger => sub {
    my ($self, $val) = @_;
    die "invalid value '$val'" unless $val =~ /^(month)$/;
});
has json => (is => 'ro', isa => 'Object', default => sub { JSON->new->allow_nonref->canonical; });

# писать данные в общую таблицу $type_mergetree
has single_merge_table => (
    is => 'ro',
    isa => 'Bool',
    default => 0,
);
# писать данные в промежуточную таблицу $type_buffer
has single_buffer_table => (
    is => 'ro',
    isa => 'Bool',
    default => 0,
);

#sub BUILDARGS
#{
#}

=head2 

    Параметры позиционные 
    $lines -- массив строк

    Возвращает 
    {
        error => [...],
        grouped_data => {
            group_1 => [ {}. {}. ... ],
            group_2 => [],
        },
    }

=cut
sub parse_lines
{
    my ($self, $lines, %O) = @_;

    my $parsed_data = {
        error => [],
        grouped_data => {},
    };

    my ($first_id, $last_id);

    return $parsed_data unless $lines;

    for my $l (@$lines) {
        my $parsed_line = $self->parse_single_line($l);
        for my $rec (@{$parsed_line->{rec}}) {
            next unless %$rec;
            if ($O{source}){
                $rec->{source} //= $O{source};
            }
            if ($O{service}) {
                $rec->{service} //= $O{service};
            }
            if ($rec->{reqid}) {
                $first_id //= $rec->{reqid};
                $last_id = $rec->{reqid};
            }
        }
        $parsed_line->{group} = $self->get_group_from_rec($parsed_line->{rec}->[0]);
        push @{$parsed_data->{error}}, @{$parsed_line->{error}||[]};
        push @{$parsed_data->{grouped_data}->{$parsed_line->{group}}}, grep { keys %$_ }  @{$parsed_line->{rec}||[]};
    }

    if (defined $first_id && defined $last_id) {
        $log->info("parsed chunk: reqid $first_id .. $last_id");
    }

    return $parsed_data;
}


=head2 parse_single_line

Извлечь полезные данные из строчки syslog

=cut

sub parse_syslog_line
{
    my ($self, $line) = @_;
    
    # print STDERR "$line\n";

    my (undef, $datetime_syslog, $host, undef, undef, undef, undef, $data_str) = ($line =~ /
        (\<\d+\>\d+) \s+
        (\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}\+\d{2}:\d{2}) \s+
        (\S+) \s+
        (\S+) \s+
        (\d+) \s+
        (\-)  \s+
        (\-)  \s+
        (.*$)
        /xms);
    
    unless ($data_str) {
        return {
            rec => [],
            error => [ "can't parse line ($line)" ],
        };
    }

    my ($datetime, $date) = _parse_datetime($datetime_syslog);
    unless ($datetime) {
        return {
            rec => [],
            error => [ "can't parse datetime '$datetime_syslog' ($line)" ],
        };
    }

    return {
        rec => [
            {
                log_time => $datetime, 
                log_date => $date,
                host => $host, 
                data => $data_str,
            }
        ],
        error => [],
    };
}

=head2 parse_datetime_json_line

    Извлечь полезные данные из строчки формата "2016-01-01 12:12:12 {...json object...}"

=cut

sub parse_datetime_json_line
{
    my ($self, $line) = @_;
    
    my ($log_datetime, $data_json) = ($line =~ /^
        (\d{4}-\d{2}-\d{2}[ T:]\d{2}:\d{2}:\d{2})
        \s+
        (\{.*\})
        \s*$
        /x);
    
    unless ($data_json) {
        return {
            rec => [],
            error => [ "can't parse line ($line)" ],
        };
    }

    my ($datetime, $date) = _parse_datetime($log_datetime);
    unless ($datetime) {
        return {
            rec => [],
            error => [ "can't parse datetime '$log_datetime' ($line)" ],
        };
    }

    my $data = eval { $self->json->decode($data_json) };
    if (!defined $data && $@) {
        return { 
            rec => [],
            error => [ "can't parse json data ($line): $@" ],
        };
    }

    return {
        rec => [
            {
                log_time => $datetime, 
                log_date => $date,
                data => $data,
            }
        ],
        error => [],
    };
}

=head2 parse_json_line

    Извлечь полезные данные из строчки формата "{...json object...}"
    Время должно быть в поле log_time

=cut

sub parse_json_line
{
    my ($self, $line) = @_;

    my $time_field = "log_time";
    
    my $data = eval { $self->json->decode($line) };
    if (!defined $data && $@) {
        return { 
            rec => [],
            error => [ "can't parse json data ($line): $@" ],
        };
    }

    my ($datetime, $date) = _parse_datetime($data->{$time_field});
    unless ($datetime) {
        return {
            rec => [],
            error => [ "can't parse datetime '$data->{$time_field}' ($line)" ],
        };
    }

    return {
        rec => [
            {
                log_time => $datetime, 
                log_date => $date,
                data => $data,
            }
        ],
        error => [],
    };
}


=head2

=cut
sub parse_syslog_json_line 
{
    my ($self, $line) = @_;

    my $h = $self->parse_syslog_line($line);
    
    if (@{$h->{error}}) {
        return $h;
    }

    # фикс, если перед началом json что-то есть
    # TODO починить Yandex::Log, чтобы такого не было
    $h->{rec}->[0]->{data} =~ s!^[^\[\{]+!!;

    $h->{rec}->[0]->{data} = eval { $self->json->decode($h->{rec}->[0]->{data}) };

    if ($@) {
        return { 
            rec => [],
            error => [ "can't parse json data '$h->{rec}->[0]->{data}' ($line)\n$@" ],
        };
    }

    return $h;
}

=head2 parse_metadata_str

из строки вида key=value,key2=value2 сделать хеш

=cut

sub parse_metadata_str
{
    my ($self, $metadata) = @_;
    my %meta =  map { /=/ ? (split /=/, $_, 2) : () } split /,/, $metadata;
    unless ( all { defined $_ && length $_ } values %meta ) {
        die "invalid metadata in \n\n'$metadata'\n\n";
    }
    return \%meta;
}

=head2

2014-02-10      00:00:23        pid=188869909,cid=7150620,par_id=61,data_type=request   [{"BannerID":"136669797","AutoBudget":0,"CurrencyISOCode":-1,"AutoBroker":1,"OrderID":"2372819","hits":"0","pid":"188869909","PhraseID":"201373824","price":"0.16","cid":"7150620","id":"1529336732","price_context":0.16}]

Функция парсит строчку из логов экспорта

=cut

sub parse_bsexport_line
{
    my ($self, $line) = @_;
    # важно! в логах дата и время -- отдельные поля, разделенные табуляцией
    my @fields = split /\t/, $line, 4;

    unless (scalar @fields == 4) {
        return {
            rec => [],
            error => ["bad line \n'$line'\n"],
        }
    }

    # Ищем ошибки
    unless ( $fields[0] =~ /^(\d{4})-(\d{2})-\d{2}$/ ) {
        return { 
            rec => [], 
            error => ["bad date '$fields[0]' ($line)"],
        };
    }
    unless ( $fields[1] =~ /^\d{2}:\d{2}:\d{2}$/ ) {
        return { 
            rec => [], 
            error => ["bad time '$fields[1]' ($line)"],
        };
    }
    unless (all { defined $_ } @fields) {
        return {
            rec => [],
            error => ["too few fields in '$line'"],
        }
    }
    if ( $fields[3] =~ /\t/ ) {
        return { 
            rec => [], 
            error => ["unexpected tab in data ($line)"],
        };
    }
    #/конец поиска ошибок

    my $rec = {};
    $rec->{log_date} = $fields[0];
    $rec->{log_time} = "$fields[0] $fields[1]";
    $rec->{metadata} = $fields[2];
    
    my $meta = eval { $self->parse_metadata_str($rec->{metadata}) };
    if ($@) {
        return {
            rec => [],
            error => [$@],
        };
    }

    hash_merge $rec, $meta;
    
    $rec->{data} = $fields[3];

    return {
        rec => [$rec],
        error => [],
    };
}

=head2 get_group_from_rec($rec)


из распаршенной записи получить имя группы.
$rec = распаршенная запись

=cut

sub get_group_from_rec
{
    my ($self, $rec) = @_;
    if ($self->single_buffer_table) {
        return 'buffer';
    } elsif ($self->single_merge_table) {
        return 'mergetree';
    }
    my $group = '000000';
    if ($self->group_by eq 'month') {
        if ( $rec->{log_date} && $rec->{log_date} =~ /^(\d{4})-(\d{2})-/ ) {
            $group = "$1$2";
        }
    }
    return $group;
}


=head2 

    2014-04-02T15:03:43+04:00 => '2014-04-02 15:03:43'
    2014-04-02 15:03:43 => '2014-04-02 15:03:43'
    2014-10-13T13:33:13+04:00

=cut
sub _parse_datetime
{
    my $datetime_syslog = shift;
    if ($datetime_syslog && $datetime_syslog =~ /(\d{4})-(\d{2})-(\d{2})[ T:](\d{2}):(\d{2}):(\d{2})(?:\+\d{2}:\d{2})?/) {
        my $datetime = "$1-$2-$3 $4:$5:$6";
        my $date = "$1-$2-$3";
        return '' unless eval { timelocal( $6, $5, $4, $3, $2-1, $1-1900) };
        return ($datetime, $date);
    } else {
        return '';
    }
}

1;
