package DScribe::Parser::messages;

use Mouse;
use Time::Local;

extends "DScribe::Parser::base_direct_log";

# utf8 нужен, чтобы не билась кодировка вложенных данных при повторном encode
has json => (is => 'ro', isa => 'Object', default => sub { JSON->new->allow_nonref->utf8; });

=head2 

Возможные форматы данных:
    старый, с syslog-префикосм, без хостнейма в meta:
<134>1 2016-07-22T15:24:38+03:00 ppcdev4.yandex.ru direct-production.messages 69085 - - 2016-07-22:18:42:00 direct.script/ppcMonitorYTResourceUsage,1314708194963030138:0:1314708194963030138 message

    переходный, с syslog-префиксом и хостнеймом в meta:
<134>1 2016-07-22T15:24:38+03:00 ppcdev4.yandex.ru direct-production.messages 69085 - - 2016-07-22:18:42:00 ppcdev4.yandex.ru,direct.script/ppcMonitorYTResourceUsage,1314708194963030138:0:1314708194963030138 message

    новый, без syslog-префикса, с хостнеймом в meta:
2016-07-22:18:42:00 ppcdev4.yandex.ru,direct.script/ppcMonitorYTResourceUsage,1314708194963030138:0:1314708194963030138 message

    новый, без syslog-префикса, с хостнеймом в meta и с bulk-данными:
2016-07-22:18:42:00 ppcdev4.yandex.ru,direct.script/ppcMonitorYTResourceUsage,1314708194963030138:0:1314708194963030138#bulk [messages]
2016-07-22:18:42:00 ppcdev4.yandex.ru,direct.script/ppcMonitorYTResourceUsage.bad_data,1314708194963030138:0:1314708194963030138#bulk [messages]

   java
2017-05-12:00:00:00 ppcdev-java-2.haze.yandex.net,direct.jobs/campaignlastchange.CampAggregatedLastchangeFeeder,4370699468929123824:0:4370699468929123824 [direct-job-pool_Worker-5] INFO  ru.yandex.direct.jobs.interceptors.JobLoggingInterceptor - START shard_7

NB:
    в "старых" форматах префикс - не выделяется из лога и всегда равен пустой строке

=cut

sub parse_single_line
{
    my ($self, $line, %O) = @_;

    if ($line =~ /^</) {
        my $h = $self->parse_syslog_line($line);

        my @recs;
        for my $rec (@{$h->{rec}}) {
            my $data = delete($rec->{data}) // '';
            if ($data =~ /^
                         (\d\d\d\d-\d\d-\d\d:\d\d:\d\d:\d\d(?:\.\d+)?)
                         \s (?: ([^,]+) , )? (?<service>[^\/]+) \/ (?<method>[^,]+) , (?<trace_id>\d+) : (?<parent_id>\d+) : (?<span_id>\d+) 
                         \s (?<message>.*)
                         /x
            ) {
                my $datetime_messages = $1;
                my ($datetime, $date, $nanos) = _parse_datetime_messages($datetime_messages);
                unless ($datetime) {
                    push @{$h->{error}}, "can't parse datetime '$datetime_messages' ($data)";
                    next;
                }

                push @recs, {
                    prefix => '',
                    class_name => '',
                    log_level => '',
                    host => $2 || $rec->{host},
                    log_date => $date,
                    log_time => $datetime,
                    log_time_nanos => $nanos,
                    %+
                };
            } else {
                push @{$h->{error}}, "Can't parse '$data', incorrect format";
            }
        }
        $h->{rec} = \@recs;

        return $h;

    } elsif ($line =~ /^
                      (?<_datetime>\d\d\d\d-\d\d-\d\d:\d\d:\d\d:\d\d(?:\.\d+)?)
                      \s (?<host>[^,]+) , (?<service>[^\/]+) \/ (?<method>[^,]+) , (?<trace_id>\d+) : (?<parent_id>\d+) : (?<span_id>\d+)
                      (?:
                        (?<_bulk_flag>\#bulk) \s                    # в java этого формата нет, в перле он строго БЕЗ префикса
                        | \s (?:
                                \[ (?<prefix>[^\]]*?) \] \s         # текст [в квадратных скобках] за которыми обязательно идет один пробел
                                (?:
                                    (?<log_level>[_a-zA-Z0-9\.\-]+)
                                    \s+                             # пробелы дополняют log_level до 5 символов + 1 пробел перед именем класса
                                    (?<class_name>[_a-zA-Z0-9\.\-\$]+)
                                    \s - \s
                                )?                                  # отсутствие этой группы - это perl с префиксом
                            )?                                      # отсутствие этой группы - это вариант perl без префикса
                      )
                      (?<message>.*)
                      $/x
    ) {
        my ($datetime, $date, $nanos) = _parse_datetime_messages($+{_datetime});
        unless ($datetime) {
            return {
                rec => [],
                error => "can't parse datetime '$+{_datetime}' ($line)",
            };
        }

        my $rec_base = {
            log_date => $date,
            log_time => $datetime,
            log_time_nanos => $nanos,
            host => $+{host},
            service => $+{service},
            method => $+{method},
            trace_id => $+{trace_id},
            parent_id => $+{parent_id},
            span_id => $+{span_id},
            prefix => $+{prefix} // '',
            class_name => $+{class_name} // '',
            log_level => $+{log_level} // '',
        };
        my @recs;

        if ($+{_bulk_flag}) {
            my $data = eval { $self->json->decode($+{message}); };
            if (!$data) {
                return {
                    rec => [],
                    error => ["Can't parse json-data: $@ ($line)"],
                };
            } elsif (ref $data ne 'ARRAY') {
                return {
                    rec => [],
                    error => ["Can't parse: bulk-message is not array ($line)"],
                };
            } else {
                for my $message (@$data) {
                    push @recs, { %$rec_base, message => (ref $message ? $self->json->encode($message) : $message) };
                }
            }
        } else {
            push @recs, { %$rec_base, message => $+{message} };
        }

        return {
            rec => \@recs,
            error => [],
        };

    } else {
        return {
            rec => [],
            error => ["can't parse line ($line)"],
        };
    }
   
}

=head2 

    2016-07-22:18:42:00 => '2016-07-22 18:42:00.123', '2016-07-22', 123_000_000

=cut
sub _parse_datetime_messages
{
    my $datetime_messages = shift;
    if ($datetime_messages && $datetime_messages =~ /(\d{4})-(\d{2})-(\d{2}):(\d{2}):(\d{2}):(\d{2})(?:\.(\d+))?/) {
        my $datetime = "$1-$2-$3 $4:$5:$6";
        my $date = "$1-$2-$3";
        my $nanos = _nanos($7);
        return '' unless eval { timelocal( $6, $5, $4, $3, $2-1, $1-1900) };
        return ($datetime, $date, $nanos);
    } else {
        return '';
    }
}

sub _nanos {
    my $n = shift;
    return 0 unless $n;
    my $len = length($n);
    if (length($n) >= 9) {
        return int(substr($n, 0, 9));
    } else {
        return $n * (10 ** (9-$len));
    }
}

1;




