package DScribe::Writer::Clickhouse;

=head1 


=cut

use Mouse;
use feature 'state';

use Yandex::Clickhouse;
use DScribe::ClickhouseTable;
use DBConfigExplorer;

use Log::Any qw/$log/;

has 'name' => (is => 'ro', isa => 'Str', required => 1);
has 'table' => (is => 'ro', isa => 'Object', required => 1);
has 'clickhouse_host' => (is => 'ro', isa => 'Str', required => 1);
has 'clickhouse_port' => (is => 'ro', isa => 'Int', required => 1);
has 'clickhouse_user' => (is => 'ro', isa => 'Str', required => 1);
has 'clickhouse_pass' => (is => 'ro', isa => 'Str', required => 1);
has 'clickhouse_timeout' => (is => 'ro', isa => 'Int', required => 1);
has 'clickhouse' => (is => 'ro', isa => 'Object', required => 1);
has single_merge_table => (
    is => 'ro',
    isa => 'Bool',
    default => 0,
);
has single_buffer_table => (
    is => 'ro',
    isa => 'Bool',
    default => 0,
);
# создавать таблицы при их отсутствии - полезно для непродакшена
has allow_create_table => (
    is => 'ro',
    isa => 'Bool',
    default => 0,
);

# создавать ли merge-таблицу над всеми группировочными таблицами? ( http://clickhouse.yandex-team.ru/#Merge )
has 'merge' => (is => 'ro', isa => 'Bool', default => 1);

sub BUILDARGS
{
    my ($self, $opt) = @_;
    
    my $data = {
        table => $opt->{table},
        name => "Clickhouse",
        clickhouse_host => $opt->{conf}->{host},
        clickhouse_port => $opt->{conf}->{port},
        clickhouse_user => $opt->{conf}->{user} // "default",
        clickhouse_db   => $opt->{conf}->{db} // "default",
        clickhouse_pass => $opt->{conf}->{pass} // "",
        clickhouse_timeout => $opt->{conf}->{timeout} // 2*60*60,
        merge => $opt->{conf}->{merge} // 1,
        allow_create_table => $opt->{conf}->{allow_create_table},
        single_merge_table => $opt->{conf}->{single_merge_table},
        single_buffer_table => $opt->{conf}->{single_buffer_table},
    };

    if ($opt->{conf}->{dbconfig} && -f $opt->{conf}->{dbconfig}) {
        my $db_conf = DBConfigExplorer::load_config($opt->{conf}->{dbconfig});
        my $db_props = DBConfigExplorer::list_properties($db_conf);

        # TODO: падать с более внятным сообщением, чего именно не хватает в dbconfig
        # TODO: выбирать доступный хост из списка (чтобы вообще не трогать dbconfig при падении мастера)
        $data->{clickhouse_host} = $db_props->{"ppchouse:dscribe_writer.host"};
        $data->{clickhouse_port} = $db_props->{"ppchouse:dscribe_writer.port"};
        $data->{clickhouse_user} = $db_props->{"ppchouse:dscribe_writer.user"};
        $data->{clickhouse_db}   = $db_props->{"ppchouse:dscribe_writer.db"};
        $data->{clickhouse_pass} = $db_props->{"ppchouse:dscribe_writer.pass"};
        $data->{clickhouse_timeout} = $db_props->{"ppchouse:dscribe_writer.timeout"};
    }

    $data->{clickhouse} = Yandex::Clickhouse->new(
        host => $data->{clickhouse_host},
        port => $data->{clickhouse_port},
        user => $data->{clickhouse_user},
        password => $data->{clickhouse_pass},
        timeout => $data->{clickhouse_timeout},
        settings => {
            database => $data->{clickhouse_db},
        },
    );
    $data->{table} = DScribe::ClickhouseTable->new($data->{table});

    return $data;
}


sub BUILD {
    my $self = shift;

    if ($self->{merge}) {
        if ($self->single_merge_table) {
            $self->check_create_table(distr => 1);
        }
        else {
            $self->check_create_table(merge => 1)
        }
    }

    return;
}


=head2 

    Пишет данные в соответствующие группировочные таблицы

    Параметры позиционные 
    $grouped_data -- ссылка на хеш { имя_группы => массив хешей }

=cut
sub write_grouped_data
{
    my ($self, $grouped_data, $version_table) = @_; 

    for my $group (keys %$grouped_data){
        my $data_hashes = $grouped_data->{$group};
        next unless @$data_hashes;
        $self->write_single_group_data($group, $data_hashes, $version_table);
    }

    return;
}


=head2 

    пишет данные в одну группировочную таблицу

    Параметры позиционные
    $group -- название группы (суффикс таблицы)
    $data -- ссылка на массив хешей

=cut
sub write_single_group_data
{
    my ($self, $group, $data, $version_table) = @_;

    my ($first_id, $last_id);

    my @to_write;
    for my $rec (@{$data}) {
        if ($rec->{reqid}) {
            $first_id //= $rec->{reqid};
            $last_id = $rec->{reqid};
        }
        push @to_write, [map {$rec->{$_} // ''} @{$self->table->db_fields}];
    }
    unshift @to_write, $self->table->db_fields;

    my $table_name = $self->check_create_table( suffix => $group, version_table => $version_table );
    $self->clickhouse->insert($table_name, \@to_write, names => 1);
    if (defined $first_id && defined $last_id) {
        $log->info("inserted chunk reqid: $first_id .. $last_id");
    }
}


=head2 

    Проверяет существование и при необходимости создает группировочную таблицу.
    Создание работает только с истинным allow_create_table

    Параметры именованные
    suffix -- группировочный суффикс для таблицы
    merge => 1|0 -- проверить/создать группировочную таблицу
    distr => 1|0 -- проверить/создать распределенную таблицу

    Возвращает имя таблицы

=cut 
sub check_create_table
{
    my ($self, %O) = @_;

    my $full_suffix = $O{version_table} ? $O{version_table}.'_'.$O{suffix} : $O{suffix};
    my $table = $self->table->table_name( suffix => $full_suffix );
    
    state $cache = {};
    if ($cache->{$table}) {
        return $table;
    }

    if ($O{suffix} && ($O{merge} || $O{distr})) {
        die "both 'suffix' and ('merge' || 'distr') do not make sense";
    }

    eval { 
        $self->clickhouse->query("DESC $table");
    };
    if ( $@ ) {
        die "$@; create table is not allowed" unless $self->allow_create_table();

        my $schema;
        if ($O{merge}) {
            $schema = $self->table->merge_table_schema_sql();
        }
        elsif ($O{distr}) {
            $schema = $self->table->distr_table_schema_sql();
        }
        else {
            $schema = $self->table->schema_sql(suffix => $O{suffix});
        }
        $self->clickhouse->query($schema);
    }
    # еще раз спрашиваем описание таблицы -- окончательно убеждаемся, что она существует
    $self->clickhouse->query("DESC $table");

    $cache->{$table} = 1;

    return $table;
}


1;
