#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
Посмотреть список дежурств:
./bin/dt-app-duty-manage-shifts.py list

Посмотреть смены в указанный период:
./bin/dt-app-duty-manage-shifts.py -s direct-app-duty2 -f 2020-12-24 -t 2020-12-30 show

Посмотреть разницу между файлом и abc:
./bin/dt-app-duty-manage-shifts.py -s direct-app-duty2 --shifts-file data/direct-app-duty-2020.shifts.txt -f 2020-12-24 -t 2020-12-30 diff

Загрузить в abc смены из указанного файла:
./bin/dt-app-duty-manage-shifts.py -s direct-app-duty2 --shifts-file data/direct-app-duty-2020.shifts.txt -f 2020-12-24 -t 2020-12-30 assign

Подробнее про организационную сторону дежурства: https://docs.yandex-team.ru/direct-dev/jeri/app-duty-schedule
"""

"""
TODO
 - сравнивать смены в abc и в файле, и загружать только те, что различаются логином или подтвержденностью -- экономим время и 500-е
 - режим сравнения файла с abc -- для мониторинга
 - проверку параметров
"""

import sys
sys.path.insert(0, '/opt/direct-py/startrek-python-client-sni-fix')
import datetime
from collections import defaultdict
import os
import re
import argparse
import requests
import json
import tempfile
import shutil
import subprocess
import urllib


ABC_API_URL = 'https://abc-back.yandex-team.ru/api/v4/'
OAUTH_HEADER = {}


def abc_get_schedules():
    headers = OAUTH_HEADER
    result = []
    for service in ['direct', 'direct-app-duty']:
        url = ABC_API_URL + '/duty/schedules/?' + urllib.urlencode({
            'service__slug': service,
            })
        resp = requests.get(url, headers=headers, timeout=5)
        resp.raise_for_status()
        resp = resp.json()
        result += resp['results']

    return result


def get_abc_shifts(schedule, date_from, date_to):
    headers = OAUTH_HEADER

    #date_from=2020-05-01&date_to=2020-05-31&schedule__slug=direct_zbp_duty_backend'
    url = ABC_API_URL + '/duty/shifts/?' + urllib.urlencode({
        'date_from':date_from,
        'date_to': date_to,
        'schedule__slug': schedule,
        })
    next_url = url
    shifts = []
    while( next_url is not None ):
        resp = requests.get(next_url, headers=headers, timeout=15)
        resp.raise_for_status()
        resp = resp.json()

        for item in resp['results']:
            shifts.append(item)

        next_url = resp.get('next')

    return shifts


def print_shifts(shifts):
    #{"start_datetime": "2020-07-27T12:30:00+03:00", "is_approved": true, "start": "2020-07-27", "schedule": {"id": 1873, "name": "ZBP-backend"}, "person": {"login": "ivatkoegor"}, "end_datetime": "2020-08-03T12:30:00+03:00", "id": 654419}
    for sh in shifts:
        schedule_name = re.sub(r' ', '_', sh['schedule']['name'])
        print '%s "%s" / %s %s %s %s' % (sh['id'], schedule_name, sh['schedule']['id'], sh['start'], ( '+' if sh['is_approved'] else '?'), (sh['person']['login'] if sh['person'] != None else '--' ) )
    return


def assign_shift(shift_id, login):
    headers = OAUTH_HEADER

    url = ABC_API_URL + ('/duty/shifts/%s/?' % shift_id)
    data = {
        'person': login,
        'is_appruved': True,
        }
    resp = requests.patch(url, data=data, headers=headers, timeout=15)
    resp.raise_for_status()
    resp = resp.json()
    return


def prepare_schedule_logins(text):
    """
    составляем дикт "дата" => [логин_1, логин_2]
    """
    schedule_logins = {}
    for line in text.splitlines():
        line = re.sub(r'#.*', '', line)
        if re.match(r'^\s*$', line):
            continue
        line = re.sub('@', '', line)
        m = re.match(r'^([0-9\-]+) +([0-9])-([0-9]) +([^ ]+) +([^ ]+) *$', line)
        if not m:
            exit("can't parse line: '%s'" % line)
        start_date = datetime.datetime.strptime(m.group(1), "%Y-%m-%d")
        days = [ int(m.group(2)), int(m.group(3)) ]
        logins = [ m.group(4), m.group(5)]
        for d in days:
            if d < 1 or d > 7:
                exit("incorrect day of week '%s'\nline: %s" % (d, line))
        for day in range(days[0], days[1] + 1):
            date = start_date + datetime.timedelta(days=day-1)
            date_str = date.strftime("%Y-%m-%d")
            schedule_logins[date_str] = logins
    return schedule_logins


def prepare_schedule_shifts(shifts):
    """
    составляем дикт "дата" => [id_смены_1, id_смены_2]
    """
    schedule_shifts = defaultdict(list)
    for sh in shifts:
        schedule_shifts[sh['start']].append(sh['id'])
    return schedule_shifts


def prepare_dates(shifts, start=None, end=None):
    """
    составляем список дат, которые попадают в смены shifts, от start до end включительно
    """
    start_date = datetime.datetime.strptime(start, "%Y-%m-%d")
    end_date   = datetime.datetime.strptime(end,   "%Y-%m-%d")
    span = end_date - start_date
    dates = []
    for i in xrange(span.days + 1):
        date_str = (start_date + datetime.timedelta(days=i)).strftime("%Y-%m-%d")
        if date_str not in shifts:
            continue
        dates.append(date_str)
    return dates


def abc_get_members(service, role):
    """
    Получаем список пользователей из проектов в ABC с нужными ролями
    print "\n".join(abc_get_members('direct-app-duty', 'developer'))
    """
    headers = OAUTH_HEADER
    url = ABC_API_URL + '/services/members/?' + urllib.urlencode({
        'fields': 'person.login',
        'page_size': '1000',
        'service__slug': service,
        'role__code': role,
        })
    resp = requests.get(url, headers=headers, timeout=15)
    resp.raise_for_status()
    data = resp.json()

    result = [el['person']['login'] for el in data['results']]

    return result




def parse_args():
    parser = argparse.ArgumentParser(formatter_class=argparse.RawTextHelpFormatter, description=__doc__)
    parser.add_argument('--abc-token-file', default="~/.abc-auth-token", dest='token_file', help='путь до oauth токена для abc')
    parser.add_argument('-s', '--schedule', dest='schedules', action="append", help='текстовые id дежурства')
    parser.add_argument("-f", "--date-from", dest="date_from", help="Начальная дата", type=str)
    parser.add_argument("-t", "--date-to", dest="date_to", help="Конечная дата", type=str)
    parser.add_argument("--assign-shifts", dest="assign_shifts", help="Загрузить смены в abc", action='store_true')
    parser.add_argument("--shifts-file", dest="shifts_file", help="Файл со сменами", type=str)
    opts, extra = parser.parse_known_args()

    if len(extra) <= 0:
        exit("expecting action (list|show|diff|assign)")

    opts.cmd = extra.pop(0)
    return opts


def main():
    global OAUTH_HEADER
    args = parse_args()

    try:
        with open(os.path.expanduser(args.token_file), 'r') as fh:
            OAUTH_HEADER = {'Authorization': 'OAuth ' + fh.read().strip()}
    except:
        sys.exit("can't get oauth token")

    if args.cmd == 'list':
        schedules = abc_get_schedules()
        schedules_sorted = sorted(schedules, key=lambda row: (row['slug']))
        for s in schedules_sorted:
            print "(%s) %s" % (s['id'], s['slug'])
        return

    shifts = []
    for schedule_slug in args.schedules:
        shifts += get_abc_shifts(schedule_slug, args.date_from, args.date_to)

    schedule_shifts = prepare_schedule_shifts(shifts)
    dates = prepare_dates(schedule_shifts, args.date_from, args.date_to )

    schedule_logins = None
    if args.shifts_file:
        schedule_file = args.shifts_file
        with open(os.path.expanduser(schedule_file), "r") as f:
            schedule_text = f.read()
        schedule_logins = prepare_schedule_logins(schedule_text)

    if args.cmd == 'show':
        print_shifts(shifts)
    elif args.cmd == 'assign':
        for d in dates:
            for i in [0,1]:
                print "%s: %s / %s" % (d, schedule_shifts[d][i], schedule_logins[d][i])
                assign_shift( schedule_shifts[d][i], schedule_logins[d][i] )
    elif args.cmd == 'diff':
        shifts_dict = {}
        for sh in shifts:
            shifts_dict[sh['id']] = sh
        diff_count = 0
        for d in dates:
            for i in [0,1]:
                shift_id = schedule_shifts[d][i]
                shift_login_abc = shifts_dict[shift_id]['person']['login'] if shifts_dict[shift_id]['person'] != None else '--'
                shift_approved_abc = shifts_dict[shift_id]['is_approved']
                if shift_login_abc != schedule_logins[d][i] or not shift_approved_abc:
                    diff_count += 1
                    print "%s/%s abc: (%s) %s file: %s" % (d, schedule_shifts[d][i], ( '+' if shift_approved_abc else '?'), shift_login_abc, schedule_logins[d][i])
        print "total differences: %s" % diff_count
        exit(0 if diff_count == 0 else 1)


if __name__ == '__main__':
    main()

