#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
Читает из abc расписание дежурств и показывает статистику

Дежурные за период:
dt-duty-stat.py -f 2020-06-01 -t 2020-07-31 --schedule direct_java_release_accepter
dt-duty-stat.py -f 2020-06-01 -t 2020-07-31 --schedule direct_zbp_duty_backend

Несколько ротаций:
dt-duty-stat.py -f 2020-09-01 -t 2020-10-31 -s direct_zbp_duty_backend -s direct_java_release_accepter --stat

Все ротации:
dt-duty-stat.py -f 2021-01-01 -t 2021-03-19 --global

Список ротаций:
dt-duty-stat.py --list

Получить токен для abc:
    https://oauth.yandex-team.ru/authorize?response_type=token&client_id=23db397a10ae4fbcb1a7ab5896dc00f6
положить в файл ~/.abc-auth-token:
    echo -n 'здесь подставить токен со страницы oauth.y-t' > ~/.abc-auth-token
"""

"""
TODO
 + --stat -- выводить статистику по службам
 + --list -- показывать список дежурств
 + показывать статистку сразу по нескольким дежурствам (в т.ч. статистику общую)
 + в список служб добавить фронтендеров
"""

import sys
sys.path.insert(0, '/opt/direct-py/startrek-python-client-sni-fix')
import datetime
import os
import re
import argparse
import requests
import json
import tempfile
import shutil
import subprocess
import urllib


ABC_API_URL = 'https://abc-back.yandex-team.ru/api/v4/'
STAFF_API_URL = 'https://staff-api.yandex-team.ru/v3/'
OAUTH_HEADER = {}

TEAMLEADS = {
        'yandex_monetize_search_direct_interface_9733': 'kuhtich',
        'yandex_monetize_search_direct_interface_dep21745': 'pavryabov',
        'yandex_monetize_search_direct_interface_dep01717': 'hrustyashko',
        'yandex_infra_tech_testing_mondesk_quality_interface': 'sudar',
        'yandex_monetize_search_direct_interface_dev': 'collapsus',
        }

def abc_get_schedules():
    headers = OAUTH_HEADER
    result = []
    for service in ['direct', 'direct-app-duty']:
        url = ABC_API_URL + '/duty/schedules/?' + urllib.urlencode({
            'service__slug': service,
            })
        resp = requests.get(url, headers=headers, timeout=5)
        resp.raise_for_status()
        resp = resp.json()
        result += resp['results']

    return result


def get_abc_shifts(schedule, date_from, date_to):
    headers = OAUTH_HEADER

    #date_from=2020-05-01&date_to=2020-05-31&schedule__slug=direct_zbp_duty_backend'
    url = ABC_API_URL + '/duty/shifts/?' + urllib.urlencode({
        'date_from':date_from,
        'date_to': date_to,
        'schedule__slug': schedule,
        })
    next_url = url
    shifts = []
    while( next_url is not None ):
        resp = requests.get(next_url, headers=headers, timeout=5)
        resp.raise_for_status()
        resp = resp.json()

        for item in resp['results']:
            shifts.append(item)

        next_url = resp.get('next')

    return shifts


def normalize_shifts(shifts):
    # { login: 'lena-san', schedule: 'java-release-actepter', start: ..., end: ..., days: 7 }
    result = []

    shifts_sorted = sorted(shifts, key=lambda row: (row['schedule']['name'], row['person']['login'], row['start']))

    prev_login = ''
    prev_schedule = ''
    norm_shift = {'end': ''}
    for sh in shifts_sorted:
        login = sh['person']['login']
        schedule_name = sh['schedule_slug']
        start_date = datetime.datetime.strptime(sh['start'], "%Y-%m-%d")
        end_date   = datetime.datetime.strptime(sh['end'],   "%Y-%m-%d")
        days = (end_date - start_date).days
        #print "login: %s, prev_login: %s, start_date: %s, norm_shift['end']: %s" % (login, prev_login, start_date, norm_shift['end'])
        if login != prev_login or schedule_name != prev_schedule or sh['start'] != norm_shift['end']:
            #print "NEW norm_shift"
            norm_shift = {}
            result.append(norm_shift)
            prev_login = login
            prev_schedule = schedule_name
            norm_shift['start'] = sh['start']
            norm_shift['end']   = sh['end']
            norm_shift['login'] = login
            norm_shift['schedule'] = schedule_name
            norm_shift['days']  = days
        else:
            norm_shift['end']   = sh['end']
            norm_shift['days'] += days

    return result



def login_to_teamlead(login):
    # https://staff-api.yandex-team.ru/v3/persons?login=alesten&_fields=official.staff_agreement&_one=1
    # https://staff-api.yandex-team.ru/v3/persons?login=lena-san&_one=1
    if login == '-':
        return '-'
    headers = OAUTH_HEADER
    teamlead = '-'
    url = STAFF_API_URL + 'persons?' + urllib.urlencode({
        'login': login,
        '_one': 1,
        })
    resp = requests.get(url, headers=headers, timeout=5)
    resp.raise_for_status()
    resp = resp.json()

    for g in resp['groups']:
        #print "%s / %s" %( g['group']['name'], g['group']['url'])
        if g['group']['url'] in TEAMLEADS:
            teamlead = TEAMLEADS[g['group']['url']]
            break
        for super_group in g['group']['ancestors']:
            if super_group['url'] in TEAMLEADS:
                teamlead = TEAMLEADS[super_group['url']]
                break
        if teamlead != '-':
            break

    return teamlead


def print_shifts(shifts):
    #{"start_datetime": "2020-07-27T12:30:00+03:00", "is_approved": true, "start": "2020-07-27", "schedule": {"id": 1873, "name": "ZBP-backend"}, "person": {"login": "ivatkoegor"}, "end_datetime": "2020-08-03T12:30:00+03:00", "id": 654419}
    for sh in shifts:
        schedule_name = re.sub(r' ', '_', sh['schedule']['name'])
        person_login = sh['person']['login'] if sh['person'] else '-'
        print '%s "%s" / %s %s %s %s / %s' % (sh['id'], schedule_name, sh['schedule']['id'], sh['start'], ( '+' if sh['is_approved'] else '?'), person_login, login_to_teamlead(person_login))
    return

def print_stat(shifts):
    stat = {}
    total = 0
    for sh in shifts:
        teamleader = login_to_teamlead(sh['person']['login'])
        if teamleader not in stat:
            stat[teamleader] = 0
        stat[teamleader] += 1
        total += 1

    print "\n=== Statistics"
    for l in sorted(stat.keys()):
        print "%s %s = %s %%" % (l, stat[l], (stat[l]*100/total) )

    return


def parse_args():
    global SVN_PATH

    parser = argparse.ArgumentParser(formatter_class=argparse.RawTextHelpFormatter, description=__doc__)
    parser.add_argument('--abc-token-file', default="~/.abc-auth-token", dest='token_file', help='путь до oauth токена для abc')
    parser.add_argument('-s', '--schedule', dest='schedules', action="append", help='текстовые id дежурства')
    parser.add_argument("-f", "--date-from", dest="date_from", help="Начальная дата", type=str)
    parser.add_argument("-t", "--date-to", dest="date_to", help="Конечная дата", type=str)
    parser.add_argument("-l", "--list", dest="list", help="список дежурств", default=False, action='store_true')
    parser.add_argument("--stat", dest="show_stat", help="показать статистику", default=False, action='store_true')
    parser.add_argument("--global", dest="global_stat", help="данные по всем дежурствам", default=False, action='store_true')
    args = parser.parse_args()

    return args


def main():
    global OAUTH_HEADER
    args = parse_args()

    try:
        with open(os.path.expanduser(args.token_file), 'r') as fh:
            OAUTH_HEADER = {'Authorization': 'OAuth ' + fh.read().strip()}
    except:
        sys.exit("can't get oauth token")

    if args.list:
        schedules = abc_get_schedules()
        #print(json.dumps(schedules, indent=4))
        for s in schedules:
            print "%s %s" % (s['id'], s['slug'])

        return

    if args.global_stat:
        args.schedules = [s['slug'] for s in abc_get_schedules()]

    shifts = []
    for schedule_slug in args.schedules:
        #print schedule_slug
        shifts_chunk = get_abc_shifts(schedule_slug, args.date_from, args.date_to)
        for sh in shifts_chunk:
            sh['schedule_slug'] = schedule_slug
        shifts += shifts_chunk

    print_shifts(shifts)

    if False:
        norm_shifts = normalize_shifts(shifts)

        stat = {}
        shedules = set()
        print "Shifts raw"
        for sh in norm_shifts:
            print "%s,%s,%s,%s,%s" % (sh['schedule'], sh['login'], sh['start'], sh['end'], sh['days'])
            if sh['login'] not in stat:
                stat[ sh['login'] ] = {}
            if sh['schedule'] not in stat[ sh['login'] ]:
                stat[ sh['login'] ][ sh['schedule'] ] = {
                        'days': 0,
                        'shifts': 0,
                        }
                shedules.add( sh['schedule'] )
            stat[ sh['login'] ][ sh['schedule'] ]['days'] += sh['days']
            stat[ sh['login'] ][ sh['schedule'] ]['shifts'] += 1

        shedules_sorted = sorted(list(shedules))

        print "\n\n"
        print "Shifts stat"
        print "login," + ",".join(shedules_sorted)
        for login in stat:
            print login + "," + ",".join( [ str(stat[ login ][ s ]['shifts'] if s in stat[ login ] else 0) for s in shedules_sorted])

        print "\n\n"

        print "Days stat"
        print "login," + ",".join(shedules_sorted)
        for login in stat:
            print login + "," + ",".join( [ str(stat[ login ][ s ]['days'] if s in stat[ login ] else 0) for s in shedules_sorted])

    if args.show_stat:
        print_stat(shifts)

if __name__ == '__main__':
    main()

