#!/usr/bin/perl
use strict;
use warnings;

=head1 NAME

generate-mysql-wsrep-cnf.pl

=head1 SYNOPSIS

    generate-mysql-wsrep-cnf.pl \
        --instance=ppcdict \
        --cgroup=direct_ng_databases_mysql_ppcdict_xtradb_prod \
        --galera-port=17411 \
        --template=/etc/mysql/ppcdict-wsrep.cnf.tmpl \
        --outfile=/etc/mysql/ppcdict.conf.d/wsrep.cnf

=head1 DESCRIPTION

На самом деле этот скрипт руками запускать не нужно. Над скриптом должны быть обёртки
под названиями generate-mysql-wsrep-cnf-instance (например: generate-mysql-wsrep-cnf-ppcdict),
которые приезжают в пакете yandex-direct-mysql-instance-pxc. Эти обёртки запускаются руками
после изменений в соответствующих кондукторных группах и, может быть, из скриптов мониторинга.

Скрипт пишет конфиг wsrep.cnf для сервера с Percona XtraDB Cluster.
В конфиге обычно находятся настройки:
wsrep_provider_options (в ist.recv_addr подставляется полный hostname сервера)
wsrep_sst_receive_address (hostname сервера ":" содержимое опции --galera-port)
wsrep_cluster_address ("gcomm://" через запятую хост из кондукторной группы ":" содержимое --galera-port)

Как это работает: скрипт ходит в кондуктор, получает содержимое кондукторной группы,
потом с помощью TT выполняет шаблонизацию: в шаблон (--template), обычно /etc/mysql/instance-wsrep.cnf.tmpl,
подставляются переменные, результат записывается в файл вывода (--outfile), обычно /etc/mysql/instance.conf.d/wsrep.cnf.

Ожидается, что /etc/mysql/instance-wsrep.cnf.tmpl приедет в пакете yandex-direct-mysql-instance-pxc

=cut

use Getopt::Long;
use Path::Tiny;
use Template;

use Yandex::Conductor;

run() unless caller;

sub run {
    my %O;
    Getopt::Long::GetOptions(
        'help' => sub { system("podselect -section NAME -section SYNOPSIS -section DESCRIPTION $0 | pod2text-utf8"); exit 0; },

        'instance=s'    => \$O{instance},
        'galera-port=n' => \$O{galera_port},
        'cgroup=s'      => \$O{cgroup},
        'template=s'    => \$O{template},
        'outfile=s'     => \$O{outfile},
        'quiet'         => \$O{quiet},
    ) or die 'Invalid options';

    for my $required_param ( qw( instance galera_port cgroup ) ) {
        unless ( defined $O{$required_param} ) {
            die "Missing required parameter: $required_param";
        }
    }

    my $cgroup_hosts = conductor_groups2hosts( $O{cgroup} );

    my $preferred_donor;
    my ($sthost, $stdomain);
    for my $host (@$cgroup_hosts) {
        my $tags = conductor_get_host_tags($host);
        if ( grep { $_ eq 'standby' } @$tags ) {
            ($sthost, $stdomain) = split(/\./, $host, 2);
            $preferred_donor = $sthost;
            last;
        }
    }

    unless ($preferred_donor) {
        warn "No preferred donor tagged standby found in $O{cgroup}";
    }

    my $hostname = qx(hostname -f);
    my $shortname = qx(hostname);
    chomp $hostname;

    my $preferred_weight = 1;
    if ($shortname =~ m/$preferred_donor/ ) {
	$preferred_weight = 0;
    }

    my $template = Template->new( ABSOLUTE => 1 );
    my $vars = {
        auto_generated_config => 1,
        ipaddr => $hostname,
        shaddr => $shortname, 
        cluster_address => join( ',', map { "$_:$O{galera_port}" } sort @$cgroup_hosts ),
        preferred_donor_hostname => $preferred_donor,
	preferred_weight => $preferred_weight,
    };

    my $result = '';
    $template->process( $O{template} || "/etc/mysql/$O{instance}-wsrep.cnf.tmpl", $vars, \$result ) || die $template->error();

    $O{outfile} ||= "/etc/mysql/$O{instance}.conf.d/wsrep.cnf";

    unless ( $O{quiet} ) {
        if ( -e $O{outfile} ) {
            my $tmpname = "/tmp/mysql-wsrep-cnf-$O{instance}-new.out";
            write_file( $tmpname, $result );
            system("diff -u $O{outfile} $tmpname");
        } else {
            print "Writing a new file at $O{outfile}\n";
        }
    }

    write_file( $O{outfile}, $result );

    exit 0;
}

sub write_file {
    my ( $filename, $content ) = @_;

    path($filename)->spew($content);
}
